"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const math_1 = require("@js-draw/math");
const RenderingCacheNode_1 = __importDefault(require("./RenderingCacheNode"));
const CacheRecordManager_1 = require("./CacheRecordManager");
class RenderingCache {
    constructor(cacheProps) {
        this.recordManager = new CacheRecordManager_1.CacheRecordManager(cacheProps);
        this.sharedState = {
            props: cacheProps,
            currentRenderingCycle: 0,
            recordManager: this.recordManager,
            debugMode: false,
        };
        this.recordManager.setSharedState(this.sharedState);
    }
    render(screenRenderer, image, viewport) {
        const visibleRect = viewport.visibleRect;
        this.sharedState.currentRenderingCycle++;
        // If we can't use the cache,
        if (!this.sharedState.props.isOfCorrectType(screenRenderer)) {
            image.render(screenRenderer, visibleRect);
            return;
        }
        if (!this.rootNode) {
            // Adjust the node so that it has the correct aspect ratio
            const res = this.sharedState.props.blockResolution;
            const topLeft = visibleRect.topLeft;
            this.rootNode = new RenderingCacheNode_1.default(new math_1.Rect2(topLeft.x, topLeft.y, res.x, res.y), this.sharedState);
        }
        while (!this.rootNode.region.containsRect(visibleRect)) {
            this.rootNode = this.rootNode.generateParent();
        }
        this.rootNode = this.rootNode.smallestChildContaining(visibleRect) ?? this.rootNode;
        const visibleLeaves = image.getLeavesIntersectingRegion(viewport.visibleRect, (rect) => screenRenderer.isTooSmallToRender(rect));
        let approxVisibleRenderTime = 0;
        for (const leaf of visibleLeaves) {
            approxVisibleRenderTime += leaf.getContent().getProportionalRenderingTime();
        }
        if (approxVisibleRenderTime > this.sharedState.props.minProportionalRenderTimeToUseCache) {
            this.rootNode.renderItems(screenRenderer, [image], viewport);
        }
        else {
            image.render(screenRenderer, visibleRect);
        }
    }
    getDebugInfo() {
        return this.recordManager.getDebugInfo();
    }
    setIsDebugMode(debugMode) {
        this.sharedState.debugMode = debugMode;
    }
}
exports.default = RenderingCache;
