import AbstractRenderer from './renderers/AbstractRenderer';
import { Editor } from '../Editor';
import { Point2, Color4 } from '@js-draw/math';
import RenderingCache from './caching/RenderingCache';
export declare enum RenderingMode {
    DummyRenderer = 0,
    CanvasRenderer = 1
}
/**
 * Handles `HTMLCanvasElement`s (or other drawing surfaces if being used) used to display the editor's contents.
 *
 * @example
 * ```
 * const editor = new Editor(document.body);
 * const w = editor.display.width;
 * const h = editor.display.height;
 * const center = Vec2.of(w / 2, h / 2);
 * const colorAtCenter = editor.display.getColorAt(center);
 * ```
 */
export default class Display {
    private editor;
    private parent;
    private dryInkRenderer;
    private wetInkRenderer;
    private textRenderer;
    private textRerenderOutput;
    private cache;
    private devicePixelRatio;
    private resizeSurfacesCallback?;
    private flattenCallback?;
    /** @internal */
    constructor(editor: Editor, mode: RenderingMode, parent: HTMLElement | null);
    /**
     * @returns the visible width of the display (e.g. how much
     * space the display's element takes up in the x direction
     * in the DOM).
     */
    get width(): number;
    /** @returns the visible height of the display. See {@link width}. */
    get height(): number;
    /** @internal */
    getCache(): RenderingCache;
    /**
     * @returns the color at the given point on the dry ink renderer, or `null` if `screenPos`
     * 	is not on the display.
     */
    getColorAt: (_screenPos: Point2) => Color4 | null;
    private initializeCanvasRendering;
    private initializeTextRendering;
    /**
     * Sets the device-pixel-ratio.
     *
     * Intended for debugging. Users do not need to call this manually.
     *
     * @internal
     */
    setDevicePixelRatio(dpr: number): Promise<void> | undefined;
    /** @internal */
    getDevicePixelRatio(): number;
    /** @internal -- used for internal performance improvements. */
    onPointerEvent(event: PointerEvent): void;
    /**
     * Rerenders the text-based display.
     * The text-based display is intended for screen readers and can be navigated to by pressing `tab`.
     */
    rerenderAsText(): void;
    /**
     * Clears the main drawing surface and otherwise prepares for a rerender.
     *
     * @returns the dry ink renderer.
     */
    startRerender(): AbstractRenderer;
    /**
     * If `draftMode`, the dry ink renderer is configured to render
     * low-quality output.
     */
    setDraftMode(draftMode: boolean): void;
    /** @internal */
    getDryInkRenderer(): AbstractRenderer;
    /**
     * @returns The renderer used for showing action previews (e.g. an unfinished stroke).
     * The `wetInkRenderer`'s surface is stacked above the `dryInkRenderer`'s.
     */
    getWetInkRenderer(): AbstractRenderer;
    /** Re-renders the contents of the wetInkRenderer onto the dryInkRenderer. */
    flatten(): void;
}
