"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeOutlinedRectangleBuilder = exports.makeFilledRectangleBuilder = void 0;
const math_1 = require("@js-draw/math");
const RenderablePathSpec_1 = require("../../rendering/RenderablePathSpec");
const Stroke_1 = __importDefault(require("../Stroke"));
const makeSnapToGridAutocorrect_1 = __importDefault(require("./autocorrect/makeSnapToGridAutocorrect"));
/**
 * Creates filled rectangles with sharp corners.
 *
 * Example:
 * [[include:doc-pages/inline-examples/changing-pen-types.md]]
 */
exports.makeFilledRectangleBuilder = (0, makeSnapToGridAutocorrect_1.default)((initialPoint, viewport) => {
    return new RectangleBuilder(initialPoint, true, viewport);
});
/**
 * Creates outlined rectangles with sharp corners.
 *
 * Example:
 * [[include:doc-pages/inline-examples/changing-pen-types.md]]
 */
exports.makeOutlinedRectangleBuilder = (0, makeSnapToGridAutocorrect_1.default)((initialPoint, viewport) => {
    return new RectangleBuilder(initialPoint, false, viewport);
});
class RectangleBuilder {
    constructor(startPoint, filled, viewport) {
        this.startPoint = startPoint;
        this.filled = filled;
        this.viewport = viewport;
        // Initially, the start and end points are the same.
        this.endPoint = startPoint;
    }
    getBBox() {
        const preview = this.buildPreview();
        return preview.getBBox();
    }
    buildPreview() {
        const canvasAngle = this.viewport.getRotationAngle();
        const rotationMat = math_1.Mat33.zRotation(-canvasAngle);
        // Adjust startPoint and endPoint such that applying [rotationMat] to them
        // brings them to this.startPoint and this.endPoint.
        const startPoint = rotationMat.inverse().transformVec2(this.startPoint.pos);
        const endPoint = rotationMat.inverse().transformVec2(this.endPoint.pos);
        const rect = math_1.Rect2.fromCorners(startPoint, endPoint);
        const path = math_1.Path.fromRect(rect, this.filled ? null : this.endPoint.width)
            .transformedBy(
        // Rotate the canvas rectangle so that its rotation matches the screen
        rotationMat)
            .mapPoints((point) => this.viewport.roundPoint(point));
        const preview = new Stroke_1.default([
            (0, RenderablePathSpec_1.pathToRenderable)(path, {
                fill: this.endPoint.color,
            }),
        ]);
        return preview;
    }
    build() {
        return this.buildPreview();
    }
    preview(renderer) {
        this.buildPreview().render(renderer);
    }
    addPoint(point) {
        this.endPoint = point;
    }
}
exports.default = RectangleBuilder;
