import SerializableCommand from '../commands/SerializableCommand';
import Editor from '../Editor';
import { LineSegment2, Rect2, Mat33 } from '@js-draw/math';
import AbstractRenderer from '../rendering/renderers/AbstractRenderer';
import { TextRenderingStyle } from '../rendering/TextRenderingStyle';
import AbstractComponent from './AbstractComponent';
import { ImageComponentLocalization } from './localization';
import RestyleableComponent, { ComponentStyle } from './RestylableComponent';
export declare enum TextTransformMode {
    /** Absolutely positioned in both the X and Y dimensions. */
    ABSOLUTE_XY = 0,
    /** Relatively positioned in both the X and Y dimensions. */
    RELATIVE_XY = 1,
    /**Relatively positioned in the X direction, absolutely positioned in the Y direction. */
    RELATIVE_X_ABSOLUTE_Y = 2,
    /**Relatively positioned in the Y direction, absolutely positioned in the X direction. */
    RELATIVE_Y_ABSOLUTE_X = 3
}
type TextElement = TextComponent | string;
/**
 * Displays text.
 *
 * A `TextComponent` is a collection of `TextElement`s (`string`s or {@link TextComponent}s).
 *
 * **Example**:
 *
 * ```ts,runnable
 * import { Editor, TextComponent, Mat33, Vec2, Color4, TextRenderingStyle } from 'js-draw';
 * const editor = new Editor(document.body);
 * editor.dispatch(editor.setBackgroundStyle({ color: Color4.black, autoresize: true ));
 * ---visible---
 * /// Adding a simple TextComponent
 * ///------------------------------
 *
 * const positioning1 = Mat33.translation(Vec2.of(10, 10));
 * const style: TextRenderingStyle = {
 *     fontFamily: 'sans', size: 12, renderingStyle: { fill: Color4.green },
 * };
 *
 * editor.dispatch(
 *     editor.image.addComponent(new TextComponent(['Hello, world'], positioning1, style)),
 * );
 *
 *
 * /// Adding nested TextComponents
 * ///-----------------------------
 *
 * // Add another TextComponent that contains text and a TextComponent. Observe that '[Test]'
 * // is placed directly after 'Test'.
 * const positioning2 = Mat33.translation(Vec2.of(10, 50));
 * editor.dispatch(
 *     editor.image.addComponent(
 *         new TextComponent([ new TextComponent(['Test'], positioning1, style), '[Test]' ], positioning2, style)
 *     ),
 * );
 * ```
 */
export default class TextComponent extends AbstractComponent implements RestyleableComponent {
    protected readonly textObjects: Array<TextElement>;
    private transform;
    private style;
    private transformMode;
    protected contentBBox: Rect2;
    readonly isRestylableComponent: true;
    /**
     * Creates a new text object from a list of component text or child TextComponents.
     *
     * @see {@link fromLines}
     */
    constructor(textObjects: Array<TextElement>, transform: Mat33, style?: TextRenderingStyle, transformMode?: TextTransformMode);
    static applyTextStyles(ctx: CanvasRenderingContext2D, style: TextRenderingStyle): void;
    private static textMeasuringCtx;
    private static estimateTextDimens;
    private static getTextMetrics;
    private static getTextDimens;
    private static getFontHeight;
    private computeUntransformedBBoxOfPart;
    private recomputeBBox;
    /**
     * Renders a TextComponent or a TextComponent child onto a `canvas`.
     *
     * `visibleRect` can be provided as a performance optimization. If not the top-level
     * text node, `baseTransform` (specifies the transformation of the parent text component
     * in canvas space) should also be provided.
     *
     * Note that passing a `baseTransform` is preferable to transforming `visibleRect`. At high
     * zoom levels, transforming `visibleRect` by the inverse of the parent transform can lead to
     * inaccuracy due to precision loss.
     */
    private renderInternal;
    render(canvas: AbstractRenderer, visibleRect?: Rect2): void;
    getProportionalRenderingTime(): number;
    intersects(lineSegment: LineSegment2): boolean;
    getStyle(): ComponentStyle;
    updateStyle(style: ComponentStyle): SerializableCommand;
    forceStyle(style: ComponentStyle, editor: Editor | null): void;
    getTextStyle(): TextRenderingStyle;
    getBaselinePos(): import("@js-draw/math").Vec3;
    getTransform(): Mat33;
    protected applyTransformation(affineTransfm: Mat33): void;
    protected createClone(): AbstractComponent;
    getText(): string;
    description(localizationTable: ImageComponentLocalization): string;
    protected serializeToJSON(): Record<string, any>;
    static deserializeFromString(json: any): TextComponent;
    /**
     * Creates a `TextComponent` from `lines`.
     *
     * @example
     * ```ts
     * const textStyle = {
     *   size: 12,
     *   fontFamily: 'serif',
     *   renderingStyle: { fill: Color4.black },
     * };
     *
     * const text = TextComponent.fromLines('foo\nbar'.split('\n'), Mat33.identity, textStyle);
     * ```
     */
    static fromLines(lines: string[], transform: Mat33, style: TextRenderingStyle): AbstractComponent;
    private static TextCursor;
}
export {};
