import SerializableCommand from '../commands/SerializableCommand';
import { Mat33, Path, Rect2, LineSegment2, Color4 } from '@js-draw/math';
import Editor from '../Editor';
import AbstractRenderer from '../rendering/renderers/AbstractRenderer';
import { StrokeStyle } from '../rendering/RenderingStyle';
import AbstractComponent from './AbstractComponent';
import { ImageComponentLocalization } from './localization';
import RestyleableComponent, { ComponentStyle } from './RestylableComponent';
import RenderablePathSpec, { RenderablePathSpecWithPath } from '../rendering/RenderablePathSpec';
import Viewport from '../Viewport';
/**
 * Represents an {@link AbstractComponent} made up of one or more {@link Path}s.
 *
 * @example
 * For some {@link Editor} editor and `Stroke` stroke,
 *
 * **Restyling**:
 * ```ts
 * editor.dispatch(stroke.updateStyle({ color: Color4.red }));
 * ```
 *
 * **Transforming**:
 * ```ts
 * editor.dispatch(stroke.transformBy(Mat33.translation(Vec2.of(10, 0))));
 * ```
 *
 * **Adding**:
 * [[include:doc-pages/inline-examples/adding-a-stroke.md]]
 */
export default class Stroke extends AbstractComponent implements RestyleableComponent {
    private parts;
    protected contentBBox: Rect2;
    readonly isRestylableComponent: true;
    private approximateRenderingTime;
    /**
     * Creates a `Stroke` from the given `parts`. All parts should have the
     * same color.
     *
     * @example
     * ```ts
     * // A path that starts at (1,1), moves to the right by (2, 0),
     * // then moves down and right by (3, 3)
     * const path = Path.fromString('m1,1 2,0 3,3');
     *
     * const stroke = new Stroke([
     *     // Fill with red
     *     pathToRenderable(path, { fill: Color4.red })
     * ]);
     * ```
     */
    constructor(parts: RenderablePathSpec[], initialZIndex?: number);
    /**
     * Creates a new `Stroke` from a {@link Path} and `style`. Strokes created
     * with this method have transparent fill.
     *
     * Example:
     * ```ts,runnable
     * import { Editor, Stroke, Color4 } from 'js-draw';
     * const editor = new Editor(document.body);
     * ---visible---
     * const stroke = Stroke.fromStroked('m0,0 l10,10', { width: 10, color: Color4.red });
     * editor.dispatch(editor.image.addComponent(stroke));
     * ```
     * Notice that `path` can be a string that specifies an SVG path
     *
     * @see fromFilled
     */
    static fromStroked(path: Path | string, style: StrokeStyle): Stroke;
    /** @see fromStroked */
    static fromFilled(path: Path | string, fill: Color4): Stroke;
    getStyle(): ComponentStyle;
    updateStyle(style: ComponentStyle): SerializableCommand;
    forceStyle(style: ComponentStyle, editor: Editor | null): void;
    /** @beta -- May fail for concave `path`s */
    withRegionErased(eraserPath: Path, viewport: Viewport): Stroke[];
    intersects(line: LineSegment2): boolean;
    keyPoints(): import("@js-draw/math").Vec3[];
    intersectsRect(rect: Rect2): boolean;
    private simplifiedPath;
    private computeSimplifiedPathFor;
    occludesEverythingBelowWhenRenderedInRect(rect: Rect2): boolean;
    render(canvas: AbstractRenderer, visibleRect?: Rect2): void;
    getProportionalRenderingTime(): number;
    private bboxForPart;
    getExactBBox(): Rect2;
    protected applyTransformation(affineTransfm: Mat33): void;
    /**
     * @returns A list of the parts that make up this path. Many paths only have one part.
     *
     * Each part (a {@link RenderablePathSpec}) contains information about the style and geometry
     * of that part of the stroke. Use the `.path` property to do collision detection and other
     * operations involving the stroke's geometry.
     *
     * Note that many of {@link Path}'s methods (e.g. {@link Path.intersection}) take a
     * `strokeWidth` parameter that can be gotten from {@link RenderablePathSpec.style} `.stroke.width`.
     */
    getParts(): Readonly<RenderablePathSpecWithPath>[];
    /**
     * @returns the {@link Path.union} of all paths that make up this stroke.
     */
    getPath(): Path;
    description(localization: ImageComponentLocalization): string;
    protected createClone(): AbstractComponent;
    protected serializeToJSON(): {
        style: {
            fill: string;
            stroke: {
                color: string;
                width: number;
            } | undefined;
        };
        path: string;
    }[];
    /** @internal */
    static deserializeFromJSON(this: void, json: any): Stroke;
}
