"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.isRestylableComponent = exports.createRestyleComponentCommand = void 0;
const math_1 = require("@js-draw/math");
const SerializableCommand_1 = __importDefault(require("../commands/SerializableCommand"));
const UnresolvedCommand_1 = __importDefault(require("../commands/UnresolvedCommand"));
const TextRenderingStyle_1 = require("../rendering/TextRenderingStyle");
const serializeComponentStyle = (style) => {
    const result = {};
    if (style.color) {
        result.color = style.color.toHexString();
    }
    if (style.textStyle) {
        result.textStyle = (0, TextRenderingStyle_1.textStyleToJSON)(style.textStyle);
    }
    return result;
};
const deserializeComponentStyle = (json) => {
    const color = json.color ? math_1.Color4.fromHex(json.color) : undefined;
    const textStyle = json.textStyle ? (0, TextRenderingStyle_1.textStyleFromJSON)(json.textStyle) : undefined;
    return {
        color,
        textStyle,
    };
};
// For internal use by Components implementing `updateStyle`:
const createRestyleComponentCommand = (initialStyle, newStyle, component) => {
    return new DefaultRestyleComponentCommand(initialStyle, newStyle, component.getId(), component);
};
exports.createRestyleComponentCommand = createRestyleComponentCommand;
// Returns true if `component` is a `RestylableComponent`.
const isRestylableComponent = (component) => {
    const hasMethods = 'getStyle' in component && 'updateStyle' in component && 'forceStyle' in component;
    if (!hasMethods) {
        return false;
    }
    if (!('isRestylableComponent' in component) || !component['isRestylableComponent']) {
        return false;
    }
    return true;
};
exports.isRestylableComponent = isRestylableComponent;
const defaultRestyleComponentCommandId = 'default-restyle-element';
class DefaultRestyleComponentCommand extends UnresolvedCommand_1.default {
    constructor(originalStyle, newStyle, componentID, component) {
        super(defaultRestyleComponentCommandId, componentID, component);
        this.originalStyle = originalStyle;
        this.newStyle = newStyle;
    }
    getComponent(editor) {
        this.resolveComponent(editor.image);
        const component = this.component;
        if (!component || !component['forceStyle'] || !component['updateStyle']) {
            throw new Error('this.component is missing forceStyle and/or updateStyle methods!');
        }
        return component;
    }
    apply(editor) {
        this.getComponent(editor).forceStyle(this.newStyle, editor);
    }
    unapply(editor) {
        this.getComponent(editor).forceStyle(this.originalStyle, editor);
    }
    description(editor, localizationTable) {
        return localizationTable.restyledElement(this.getComponent(editor).description(localizationTable));
    }
    serializeToJSON() {
        return {
            id: this.componentID,
            originalStyle: serializeComponentStyle(this.originalStyle),
            newStyle: serializeComponentStyle(this.newStyle),
        };
    }
}
(() => {
    SerializableCommand_1.default.register(defaultRestyleComponentCommandId, (json, _editor) => {
        const origStyle = deserializeComponentStyle(json.originalStyle);
        const newStyle = deserializeComponentStyle(json.newStyle);
        const id = json.id;
        if (typeof json.id !== 'string') {
            throw new Error(`json.id is of type ${typeof json.id}, not string.`);
        }
        return new DefaultRestyleComponentCommand(origStyle, newStyle, id);
    });
})();
