"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const Command_1 = __importDefault(require("./Command"));
const SerializableCommand_1 = __importDefault(require("./SerializableCommand"));
// Returns a command that does the opposite of the given command --- `result.apply()` calls
// `command.unapply()` and `result.unapply()` calls `command.apply()`.
const invertCommand = (command) => {
    if (command instanceof SerializableCommand_1.default) {
        // SerializableCommand that does the inverse of [command]
        return new (class extends SerializableCommand_1.default {
            constructor() {
                super(...arguments);
                // For debugging
                this._command = command;
            }
            serializeToJSON() {
                return command.serialize();
            }
            apply(editor) {
                command.unapply(editor);
            }
            unapply(editor) {
                command.apply(editor);
            }
            onDrop(editor) {
                command.onDrop(editor);
            }
            description(editor, localizationTable) {
                return localizationTable.inverseOf(command.description(editor, localizationTable));
            }
        })('inverse');
    }
    else {
        // Command that does the inverse of [command].
        const result = new (class extends Command_1.default {
            apply(editor) {
                command.unapply(editor);
            }
            unapply(editor) {
                command.apply(editor);
            }
            onDrop(editor) {
                command.onDrop(editor);
            }
            description(editor, localizationTable) {
                return localizationTable.inverseOf(command.description(editor, localizationTable));
            }
        })();
        // We know that T does not extend SerializableCommand, and thus returning a Command
        // is appropriate.
        return result;
    }
};
SerializableCommand_1.default.register('inverse', (data, editor) => {
    return invertCommand(SerializableCommand_1.default.deserialize(data, editor));
});
exports.default = invertCommand;
