import { unique } from './array.js';
import { notEmpty } from './type-check.js';
const COMMENT_RE = /^\s*(?<header>eslint-disable|eslint-disable-next-line)\s+(?<ruleList>[@a-z0-9\-_$/]*(?:\s*,\s*[@a-z0-9\-_$/]*)*(?:\s*,)?)(?:\s+--\s+(?<description>.*\S))?\s*$/u;
const SHEBANG_PATTERN = /^#!.+?\r?\n/u;
/** `results` 内で使われているプラグインの名前のリストを洗い出して返す */
export function scanUsedPluginsFromResults(results) {
    const plugins = results
        .flatMap((result) => result.messages) // messages: Linter.LintMessage[]
        .map((message) => message.ruleId) // ruleIds: (string | undefined)[]
        .filter(notEmpty) // ruleIds: string[]
        .map((ruleId) => {
        const parts = ruleId.split('/');
        if (parts.length === 1)
            return undefined; // ex: 'rule-a'
        if (parts.length === 2)
            return parts[0]; // ex: 'plugin/rule-a'
        if (parts.length === 3)
            return `${parts[0]}/${parts[1]}`; // ex: '@scoped/plugin/rule-a'
        return undefined; // invalid ruleId
    }) // plugins: string[]
        .filter(notEmpty);
    return unique(plugins);
}
/**
 * Parses the comment as an ESLint disable comment.
 * Returns undefined if the comment cannot be parsed as a disable comment.
 *
 * ## Reference: Structure of a disable comment
 * /* eslint-disable-next-line rule-a, rule-b, rule-c, rule-d -- I'm the rules.
 *    ^^^^^^^^^^^^^^^^^^^^^^^^ ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ ^^ ^^^^^^^^^^^^^^
 *    |                        |                              |  |
 *    header                   |                              |  |
 *                             ruleList                       |  |
 *                                            descriptionHeader  |
 *                                                               description
 */
export function parseDisableComment(comment) {
    // NOTE: Comment nodes should always have range and loc, but they are optional in the types.
    // If range or loc is missing, consider the parsing failed.
    if (!comment.range || !comment.loc)
        return undefined;
    const result = COMMENT_RE.exec(comment.value);
    if (!result)
        return undefined;
    if (!result.groups)
        return undefined;
    const { header, ruleList, description } = result.groups;
    const ruleIds = ruleList
        .split(',')
        .map((r) => r.trim())
        // Exclude empty strings
        .filter((ruleId) => ruleId !== '');
    const scope = header === 'eslint-disable-next-line' ? 'next-line' : 'file';
    // A file scope comment must be block-style.
    if (scope === 'file' && comment.type === 'Line')
        return undefined;
    return {
        type: comment.type,
        scope: header === 'eslint-disable-next-line' ? 'next-line' : 'file',
        ruleIds: ruleIds,
        // description is optional
        ...(description === '' || description === undefined ? {} : { description }),
        range: comment.range,
        loc: comment.loc,
    };
}
/**
 * Convert text to comment text.
 */
export function toCommentText(args) {
    const { type, text } = args;
    if (type === 'Line') {
        return `// ${text}`;
    }
    else {
        return `/* ${text} */`;
    }
}
/**
 * Convert `DisableComment` to comment text.
 */
export function toDisableCommentText({ type, scope, ruleIds, description, }) {
    const header = scope === 'next-line' ? 'eslint-disable-next-line' : 'eslint-disable';
    const ruleList = unique(ruleIds).join(', ');
    const footer = description === undefined ? '' : ` -- ${description}`;
    return toCommentText({ type, text: `${header} ${ruleList}${footer}` });
}
function getIndentFromLine(sourceCode, line) {
    const headNodeIndex = sourceCode.getIndexFromLoc({ line: line, column: 0 });
    // Extract the same indent as the line we want to fix
    const indent = sourceCode.text.slice(headNodeIndex, headNodeIndex +
        sourceCode.text
            .slice(headNodeIndex)
            // ref: https://tc39.es/ecma262/#sec-white-space
            // eslint-disable-next-line no-control-regex
            .search(/[^\u{0009}\u{000B}\u{000C}\u{FEFF}\p{gc=Space_Separator}]/u));
    return indent;
}
function isLineInJSXText(sourceCode, line) {
    const headNodeIndex = sourceCode.getIndexFromLoc({ line: line, column: 0 });
    const headNode = sourceCode.getNodeByRangeIndex(headNodeIndex);
    return (headNode === null || headNode === void 0 ? void 0 : headNode.type) === 'JSXText';
}
/**
 * Merge the ruleIds of the disable comments.
 * @param a The ruleIds of first disable comment
 * @param b The ruleIds of second disable comment
 * @returns The ruleIds of merged disable comment
 */
export function mergeRuleIds(a, b) {
    return unique([...a, ...b]);
}
/**
 * Merge the description of the disable comments.
 * @param a The description of first disable comment
 * @param b The description of second disable comment
 * @returns The description of merged disable comment
 */
export function mergeDescription(a, b) {
    if (a === undefined && b === undefined)
        return undefined;
    if (a === undefined)
        return b;
    if (b === undefined)
        return a;
    return `${a}, ${b}`;
}
export function insertDescriptionCommentStatementBeforeLine(args) {
    const { fixer, sourceCode, line, description } = args;
    const indent = getIndentFromLine(sourceCode, line);
    const headNodeIndex = sourceCode.getIndexFromLoc({ line, column: 0 });
    if (isLineInJSXText(sourceCode, line)) {
        const commentText = toCommentText({ type: 'Block', text: description });
        return fixer.insertTextBeforeRange([headNodeIndex, headNodeIndex], `${indent}{${commentText}}\n`);
    }
    else {
        const commentText = toCommentText({ type: 'Line', text: description });
        return fixer.insertTextBeforeRange([headNodeIndex, headNodeIndex], `${indent}${commentText}\n`);
    }
}
/**
 * Update existing disable comment.
 * @returns The eslint's fix object
 */
export function updateDisableComment(args) {
    const { fixer, disableComment: existingDisableComment, newRules, newDescription } = args;
    const newDisableCommentText = toDisableCommentText({
        type: existingDisableComment.type,
        scope: existingDisableComment.scope,
        ruleIds: newRules,
        description: newDescription,
    });
    return fixer.replaceTextRange(existingDisableComment.range, newDisableCommentText);
}
export function insertDisableCommentStatementBeforeLine(args) {
    const { fixer, sourceCode, line, scope, ruleIds, description } = args;
    const indent = getIndentFromLine(sourceCode, line);
    const headNodeIndex = sourceCode.getIndexFromLoc({ line: line, column: 0 });
    const isInJSXText = isLineInJSXText(sourceCode, line);
    const type = isInJSXText || scope === 'file' ? 'Block' : 'Line';
    const disableCommentText = toDisableCommentText({
        type,
        scope,
        ruleIds,
        description,
    });
    if (isInJSXText) {
        return fixer.insertTextBeforeRange([headNodeIndex, headNodeIndex], `${indent}{${disableCommentText}}\n`);
    }
    else {
        return fixer.insertTextBeforeRange([headNodeIndex, headNodeIndex], `${indent}${disableCommentText}\n`);
    }
}
/**
 * Convert `InlineConfigComment` to comment text.
 */
export function toInlineConfigCommentText({ rulesRecord, description }) {
    const header = 'eslint';
    const rulesRecordText = Object.entries(rulesRecord)
        .map(([ruleId, ruleEntry]) => {
        // TODO: Inherit options of the rule set by the user in eslintrc if the option exists.
        return `${ruleId}: ${JSON.stringify(ruleEntry)}`;
    })
        .join(', ');
    if (description === undefined) {
        return `/* ${header} ${rulesRecordText} */`;
    }
    else {
        return `/* ${header} ${rulesRecordText} -- ${description} */`;
    }
}
/**
 * Create the results with only messages with the specified rule ids.
 * @param results The lint results.
 * @param ruleIds The rule ids.
 * @returns The results with only messages with the specified rule ids
 */
export function filterResultsByRuleId(results, ruleIds) {
    return results
        .map((result) => {
        return {
            ...result,
            messages: result.messages.filter((message) => ruleIds.includes(message.ruleId)),
        };
    })
        .filter((result) => result.messages.length > 0);
}
/**
 * Find shebang from the first line of the file.
 * @param sourceCodeText The source code text of the file.
 * @returns The information of shebang. If the file does not have shebang, return null.
 */
export function findShebang(sourceCodeText) {
    const result = SHEBANG_PATTERN.exec(sourceCodeText);
    if (!result)
        return null;
    return { range: [0, result[0].length] };
}
//# sourceMappingURL=eslint.js.map