import yargs from 'yargs';
import { DEFAULT_BASE_CONFIG } from '../core.js';
import { VERSION } from './package.js';
/** Parse argv into the config object of eslint-interactive */
export function parseArgv(argv) {
    var _a, _b;
    const parsedArgv = yargs(argv.slice(2))
        .wrap(Math.min(140, process.stdout.columns))
        .scriptName('eslint-interactive')
        .version(VERSION)
        .usage('$0 [file.js] [dir]')
        .detectLocale(false)
        // NOTE: yargs doesn't support negative only option. So we use `--eslintrc` instead of `--no-eslintrc`.
        .option('eslintrc', {
        type: 'boolean',
        describe: 'Enable use of configuration from .eslintrc.*',
        default: DEFAULT_BASE_CONFIG.useEslintrc,
    })
        .option('config', {
        alias: 'c',
        type: 'string',
        describe: 'Use this configuration, overriding .eslintrc.* config options if present',
    })
        .option('ext', {
        type: 'array',
        describe: 'Specify JavaScript file extensions',
    })
        .nargs('ext', 1)
        .option('resolve-plugins-relative-to', {
        type: 'string',
        describe: 'A folder where plugins should be resolved from, CWD by default',
    })
        .option('rulesdir', {
        type: 'array',
        describe: 'Use additional rules from this directory',
    })
        .nargs('rulesdir', 1)
        // Following ESLint, --ignore-path accepts only one path. However, this limitation may be relaxed in the future.
        // ref: https://github.com/eslint/eslint/issues/9794
        .option('ignore-path', {
        type: 'string',
        describe: 'Specify path of ignore file',
    })
        .option('format', {
        type: 'string',
        describe: 'Specify the format to be used for the `Display problem messages` action',
        default: DEFAULT_BASE_CONFIG.formatterName,
    })
        .option('quiet', {
        type: 'boolean',
        describe: 'Report errors only',
        default: DEFAULT_BASE_CONFIG.quiet,
    })
        .option('cache', {
        type: 'boolean',
        describe: 'Only check changed files',
        default: DEFAULT_BASE_CONFIG.cache,
    })
        .option('cache-location', {
        type: 'string',
        describe: `Path to the cache file or directory`,
        default: DEFAULT_BASE_CONFIG.cacheLocation,
    })
        .example('$0 ./src', 'Lint ./src/ directory')
        .example('$0 ./src ./test', 'Lint multiple directories')
        .example("$0 './src/**/*.{ts,tsx,vue}'", 'Lint with glob pattern')
        .example('$0 ./src --ext .ts,.tsx,.vue', 'Lint with custom extensions')
        .example('$0 ./src --rulesdir ./rules', 'Lint with custom rules')
        .example('$0 ./src --no-eslintrc --config ./.eslintrc.ci.js', 'Lint with custom config')
        .parseSync();
    // NOTE: convert `string` type because yargs convert `'10'` (`string` type) into `10` (`number` type)
    // and `lintFiles` only accepts `string[]`.
    const patterns = parsedArgv._.map((pattern) => pattern.toString());
    const rulePaths = (_a = parsedArgv.rulesdir) === null || _a === void 0 ? void 0 : _a.map((rulePath) => rulePath.toString());
    const extensions = (_b = parsedArgv.ext) === null || _b === void 0 ? void 0 : _b.map((extension) => extension.toString()).flatMap((extension) => extension.split(','));
    const formatterName = parsedArgv.format;
    const resolvePluginsRelativeTo = parsedArgv['resolve-plugins-relative-to'];
    return {
        patterns,
        useEslintrc: parsedArgv.eslintrc,
        overrideConfigFile: parsedArgv.config,
        extensions,
        rulePaths,
        ignorePath: parsedArgv.ignorePath,
        formatterName,
        quiet: parsedArgv.quiet,
        cache: parsedArgv.cache,
        cacheLocation: parsedArgv['cache-location'],
        resolvePluginsRelativeTo,
    };
}
//# sourceMappingURL=parse-argv.js.map