"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.load = exports.loadContext = void 0;
const tslib_1 = require("tslib");
const path_1 = tslib_1.__importDefault(require("path"));
const lodash_1 = tslib_1.__importDefault(require("lodash"));
const utils_1 = require("@docusaurus/utils");
const config_1 = require("./config");
const clientModules_1 = require("./clientModules");
const plugins_1 = require("./plugins");
const routes_1 = require("./routes");
const htmlTags_1 = require("./htmlTags");
const siteMetadata_1 = require("./siteMetadata");
const i18n_1 = require("./i18n");
const translations_1 = require("./translations/translations");
/**
 * Loading context is the very first step in site building. Its options are
 * directly acquired from CLI options. It mainly loads `siteConfig` and the i18n
 * context (which includes code translations). The `LoadContext` will be passed
 * to plugin constructors.
 */
async function loadContext(options) {
    const { siteDir, outDir: baseOutDir = utils_1.DEFAULT_BUILD_DIR_NAME, locale, config: customConfigFilePath, } = options;
    const generatedFilesDir = path_1.default.resolve(siteDir, utils_1.GENERATED_FILES_DIR_NAME);
    const { siteConfig: initialSiteConfig, siteConfigPath } = await (0, config_1.loadSiteConfig)({
        siteDir,
        customConfigFilePath,
    });
    const i18n = await (0, i18n_1.loadI18n)(initialSiteConfig, { locale });
    const baseUrl = (0, utils_1.localizePath)({
        path: initialSiteConfig.baseUrl,
        i18n,
        options,
        pathType: 'url',
    });
    const outDir = (0, utils_1.localizePath)({
        path: path_1.default.resolve(siteDir, baseOutDir),
        i18n,
        options,
        pathType: 'fs',
    });
    const siteConfig = { ...initialSiteConfig, baseUrl };
    const localizationDir = path_1.default.resolve(siteDir, i18n.path, i18n.localeConfigs[i18n.currentLocale].path);
    const codeTranslationFileContent = (await (0, translations_1.readCodeTranslationFileContent)({ localizationDir })) ?? {};
    // We only need key->message for code translations
    const codeTranslations = lodash_1.default.mapValues(codeTranslationFileContent, (value) => value.message);
    return {
        siteDir,
        generatedFilesDir,
        localizationDir,
        siteConfig,
        siteConfigPath,
        outDir,
        baseUrl,
        i18n,
        codeTranslations,
    };
}
exports.loadContext = loadContext;
/**
 * This is the crux of the Docusaurus server-side. It reads everything it needs—
 * code translations, config file, plugin modules... Plugins then use their
 * lifecycles to generate content and other data. It is side-effect-ful because
 * it generates temp files in the `.docusaurus` folder for the bundler.
 */
async function load(options) {
    const { siteDir } = options;
    const context = await loadContext(options);
    const { generatedFilesDir, siteConfig, siteConfigPath, outDir, baseUrl, i18n, localizationDir, codeTranslations: siteCodeTranslations, } = context;
    const { plugins, pluginsRouteConfigs, globalData } = await (0, plugins_1.loadPlugins)(context);
    const clientModules = (0, clientModules_1.loadClientModules)(plugins);
    const { headTags, preBodyTags, postBodyTags } = (0, htmlTags_1.loadHtmlTags)(plugins);
    const { registry, routesChunkNames, routesConfig, routesPaths } = (0, routes_1.loadRoutes)(pluginsRouteConfigs, baseUrl, siteConfig.onDuplicateRoutes);
    const codeTranslations = {
        ...(await (0, translations_1.getPluginsDefaultCodeTranslationMessages)(plugins)),
        ...siteCodeTranslations,
    };
    const siteMetadata = await (0, siteMetadata_1.loadSiteMetadata)({ plugins, siteDir });
    // === Side-effects part ===
    const genWarning = (0, utils_1.generate)(generatedFilesDir, 
    // cSpell:ignore DONT
    'DONT-EDIT-THIS-FOLDER', `This folder stores temp files that Docusaurus' client bundler accesses.

DO NOT hand-modify files in this folder because they will be overwritten in the
next build. You can clear all build artifacts (including this folder) with the
\`docusaurus clear\` command.
`);
    const genSiteConfig = (0, utils_1.generate)(generatedFilesDir, `${utils_1.DEFAULT_CONFIG_FILE_NAME}.mjs`, `/*
 * AUTOGENERATED - DON'T EDIT
 * Your edits in this file will be overwritten in the next build!
 * Modify the docusaurus.config.js file at your site's root instead.
 */
export default ${JSON.stringify(siteConfig, null, 2)};
`);
    const genClientModules = (0, utils_1.generate)(generatedFilesDir, 'client-modules.js', `export default [
${clientModules
        // Use `require()` because `import()` is async but client modules can have CSS
        // and the order matters for loading CSS.
        .map((clientModule) => `  require('${(0, utils_1.escapePath)(clientModule)}'),`)
        .join('\n')}
];
`);
    const genRegistry = (0, utils_1.generate)(generatedFilesDir, 'registry.js', `export default {
${Object.entries(registry)
        .sort((a, b) => a[0].localeCompare(b[0]))
        .map(([chunkName, modulePath]) => `  '${chunkName}': [() => import(/* webpackChunkName: '${chunkName}' */ '${modulePath}'), '${modulePath}', require.resolveWeak('${modulePath}')],`)
        .join('\n')}};
`);
    const genRoutesChunkNames = (0, utils_1.generate)(generatedFilesDir, 'routesChunkNames.json', JSON.stringify(routesChunkNames, null, 2));
    const genRoutes = (0, utils_1.generate)(generatedFilesDir, 'routes.js', routesConfig);
    const genGlobalData = (0, utils_1.generate)(generatedFilesDir, 'globalData.json', JSON.stringify(globalData, null, 2));
    const genI18n = (0, utils_1.generate)(generatedFilesDir, 'i18n.json', JSON.stringify(i18n, null, 2));
    const genCodeTranslations = (0, utils_1.generate)(generatedFilesDir, 'codeTranslations.json', JSON.stringify(codeTranslations, null, 2));
    const genSiteMetadata = (0, utils_1.generate)(generatedFilesDir, 'site-metadata.json', JSON.stringify(siteMetadata, null, 2));
    await Promise.all([
        genWarning,
        genClientModules,
        genSiteConfig,
        genRegistry,
        genRoutesChunkNames,
        genRoutes,
        genGlobalData,
        genSiteMetadata,
        genI18n,
        genCodeTranslations,
    ]);
    return {
        siteConfig,
        siteConfigPath,
        siteMetadata,
        siteDir,
        outDir,
        baseUrl,
        i18n,
        localizationDir,
        generatedFilesDir,
        routes: pluginsRouteConfigs,
        routesPaths,
        plugins,
        headTags,
        preBodyTags,
        postBodyTags,
        codeTranslations,
    };
}
exports.load = load;
