"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.start = void 0;
const tslib_1 = require("tslib");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const path_1 = tslib_1.__importDefault(require("path"));
const lodash_1 = tslib_1.__importDefault(require("lodash"));
const logger_1 = tslib_1.__importDefault(require("@docusaurus/logger"));
const utils_1 = require("@docusaurus/utils");
const chokidar_1 = tslib_1.__importDefault(require("chokidar"));
const html_webpack_plugin_1 = tslib_1.__importDefault(require("html-webpack-plugin"));
const openBrowser_1 = tslib_1.__importDefault(require("react-dev-utils/openBrowser"));
const WebpackDevServerUtils_1 = require("react-dev-utils/WebpackDevServerUtils");
const evalSourceMapMiddleware_1 = tslib_1.__importDefault(require("react-dev-utils/evalSourceMapMiddleware"));
const webpack_1 = tslib_1.__importDefault(require("webpack"));
const webpack_dev_server_1 = tslib_1.__importDefault(require("webpack-dev-server"));
const webpack_merge_1 = tslib_1.__importDefault(require("webpack-merge"));
const server_1 = require("../server");
const client_1 = tslib_1.__importDefault(require("../webpack/client"));
const utils_2 = require("../webpack/utils");
const getHostPort_1 = require("../server/getHostPort");
async function start(siteDirParam = '.', cliOptions = {}) {
    // Temporary workaround to unlock the ability to translate the site config
    // We'll remove it if a better official API can be designed
    // See https://github.com/facebook/docusaurus/issues/4542
    process.env.DOCUSAURUS_CURRENT_LOCALE = cliOptions.locale;
    const siteDir = await fs_extra_1.default.realpath(siteDirParam);
    logger_1.default.info('Starting the development server...');
    function loadSite() {
        return (0, server_1.load)({
            siteDir,
            config: cliOptions.config,
            locale: cliOptions.locale,
            localizePath: undefined, // Should this be configurable?
        });
    }
    // Process all related files as a prop.
    const props = await loadSite();
    const protocol = process.env.HTTPS === 'true' ? 'https' : 'http';
    const { host, port } = await (0, getHostPort_1.getHostPort)(cliOptions);
    if (port === null) {
        process.exit();
    }
    const { baseUrl, headTags, preBodyTags, postBodyTags } = props;
    const urls = (0, WebpackDevServerUtils_1.prepareUrls)(protocol, host, port);
    const openUrl = (0, utils_1.normalizeUrl)([urls.localUrlForBrowser, baseUrl]);
    logger_1.default.success `Docusaurus website is running at: url=${openUrl}`;
    // Reload files processing.
    const reload = lodash_1.default.debounce(() => {
        loadSite()
            .then(({ baseUrl: newBaseUrl }) => {
            const newOpenUrl = (0, utils_1.normalizeUrl)([urls.localUrlForBrowser, newBaseUrl]);
            if (newOpenUrl !== openUrl) {
                logger_1.default.success `Docusaurus website is running at: url=${newOpenUrl}`;
            }
        })
            .catch((err) => {
            logger_1.default.error(err.stack);
        });
    }, 500);
    const { siteConfig, plugins, localizationDir } = props;
    const normalizeToSiteDir = (filepath) => {
        if (filepath && path_1.default.isAbsolute(filepath)) {
            return (0, utils_1.posixPath)(path_1.default.relative(siteDir, filepath));
        }
        return (0, utils_1.posixPath)(filepath);
    };
    const pluginPaths = plugins
        .flatMap((plugin) => plugin.getPathsToWatch?.() ?? [])
        .filter(Boolean)
        .map(normalizeToSiteDir);
    const pathsToWatch = [...pluginPaths, props.siteConfigPath, localizationDir];
    const pollingOptions = {
        usePolling: !!cliOptions.poll,
        interval: Number.isInteger(cliOptions.poll)
            ? cliOptions.poll
            : undefined,
    };
    const httpsConfig = await (0, utils_2.getHttpsConfig)();
    const fsWatcher = chokidar_1.default.watch(pathsToWatch, {
        cwd: siteDir,
        ignoreInitial: true,
        ...{ pollingOptions },
    });
    ['add', 'change', 'unlink', 'addDir', 'unlinkDir'].forEach((event) => fsWatcher.on(event, reload));
    let config = (0, webpack_merge_1.default)(await (0, client_1.default)(props, cliOptions.minify), {
        watchOptions: {
            ignored: /node_modules\/(?!@docusaurus)/,
            poll: cliOptions.poll,
        },
        infrastructureLogging: {
            // Reduce log verbosity, see https://github.com/facebook/docusaurus/pull/5420#issuecomment-906613105
            level: 'warn',
        },
        plugins: [
            // Generates an `index.html` file with the <script> injected.
            new html_webpack_plugin_1.default({
                template: path_1.default.join(__dirname, '../webpack/templates/index.html.template.ejs'),
                // So we can define the position where the scripts are injected.
                inject: false,
                filename: 'index.html',
                title: siteConfig.title,
                headTags,
                preBodyTags,
                postBodyTags,
            }),
        ],
    });
    // Plugin Lifecycle - configureWebpack and configurePostCss.
    plugins.forEach((plugin) => {
        const { configureWebpack, configurePostCss } = plugin;
        if (configurePostCss) {
            config = (0, utils_2.applyConfigurePostCss)(configurePostCss.bind(plugin), config);
        }
        if (configureWebpack) {
            config = (0, utils_2.applyConfigureWebpack)(configureWebpack.bind(plugin), // The plugin lifecycle may reference `this`.
            config, false, props.siteConfig.webpack?.jsLoader, plugin.content);
        }
    });
    const compiler = (0, webpack_1.default)(config);
    if (process.env.E2E_TEST) {
        compiler.hooks.done.tap('done', (stats) => {
            if (stats.hasErrors()) {
                logger_1.default.error('E2E_TEST: Project has compiler errors.');
                process.exit(1);
            }
            logger_1.default.success('E2E_TEST: Project can compile.');
            process.exit(0);
        });
    }
    // https://webpack.js.org/configuration/dev-server
    const defaultDevServerConfig = {
        hot: cliOptions.hotOnly ? 'only' : true,
        liveReload: false,
        client: {
            progress: true,
            overlay: {
                warnings: false,
                errors: true,
            },
        },
        headers: {
            'access-control-allow-origin': '*',
        },
        devMiddleware: {
            publicPath: baseUrl,
            // Reduce log verbosity, see https://github.com/facebook/docusaurus/pull/5420#issuecomment-906613105
            stats: 'summary',
        },
        static: siteConfig.staticDirectories.map((dir) => ({
            publicPath: baseUrl,
            directory: path_1.default.resolve(siteDir, dir),
            watch: {
                // Useful options for our own monorepo using symlinks!
                // See https://github.com/webpack/webpack/issues/11612#issuecomment-879259806
                followSymlinks: true,
                ignored: /node_modules\/(?!@docusaurus)/,
                ...{ pollingOptions },
            },
        })),
        ...(httpsConfig && {
            server: typeof httpsConfig === 'object'
                ? {
                    type: 'https',
                    options: httpsConfig,
                }
                : 'https',
        }),
        historyApiFallback: {
            rewrites: [{ from: /\/*/, to: baseUrl }],
        },
        allowedHosts: 'all',
        host,
        port,
        setupMiddlewares: (middlewares, devServer) => {
            // This lets us fetch source contents from webpack for the error overlay.
            middlewares.unshift((0, evalSourceMapMiddleware_1.default)(devServer));
            return middlewares;
        },
    };
    // Allow plugin authors to customize/override devServer config
    const devServerConfig = (0, webpack_merge_1.default)([defaultDevServerConfig, config.devServer].filter(Boolean));
    const devServer = new webpack_dev_server_1.default(devServerConfig, compiler);
    devServer.startCallback(() => {
        if (cliOptions.open) {
            (0, openBrowser_1.default)(openUrl);
        }
    });
    ['SIGINT', 'SIGTERM'].forEach((sig) => {
        process.on(sig, () => {
            devServer.stop();
            process.exit();
        });
    });
}
exports.start = start;
