package net.canvoki.carburoid.model

import android.content.Context
import android.text.format.DateUtils
import android.util.TypedValue
import androidx.annotation.AttrRes
import androidx.annotation.ColorInt
import androidx.annotation.DrawableRes
import androidx.annotation.StringRes
import androidx.compose.material3.ColorScheme
import androidx.compose.ui.graphics.Color
import net.canvoki.carburoid.R
import java.time.DayOfWeek
import java.time.Instant
import java.time.LocalDateTime
import java.time.LocalTime
import java.time.ZoneId
import java.time.temporal.ChronoUnit
import java.util.Locale
import androidx.appcompat.R as AppCompatR
import com.google.android.material.R as MaterialR

val END_OF_DAY = LocalTime.of(23, 59)
typealias TimeSpec = Pair<Int, Int>
typealias Interval = Pair<TimeSpec, TimeSpec>
typealias Intervals = List<Interval>
typealias DayRange = List<DayOfWeek>
typealias ScheduleEntry = Pair<DayRange, Intervals>

data class OpeningStatus(
    val isOpen: Boolean,
    val until: Instant?,
) {
    val defaultThresholdMinutes: Long = 24 * 60

    enum class UiValues(
        @field:StringRes val stringRes: Int,
        @field:AttrRes val colorAttr: Int,
        @field:DrawableRes val iconRes: Int,
        val colorSelector: (ColorScheme) -> Color,
        val usesRelativeTime: Boolean = false,
    ) {
        PERMANENTLY_CLOSED(
            stringRes = R.string.station_status_permanently_closed,
            colorAttr = AppCompatR.attr.colorError,
            colorSelector = { s -> s.error },
            iconRes = R.drawable.ic_block,
        ) {
            override fun matches(
                status: OpeningStatus,
                deadline: Instant,
            ): Boolean = !status.isOpen && status.until == null
        },
        OPENS_AT(
            stringRes = R.string.station_status_opens_at,
            colorAttr = MaterialR.attr.colorOnSurfaceVariant,
            colorSelector = { s -> s.onSurfaceVariant },
            iconRes = R.drawable.ic_lock_clock,
            usesRelativeTime = true,
        ) {
            override fun matches(
                status: OpeningStatus,
                deadline: Instant,
            ): Boolean = !status.isOpen && status.until != null
        },
        OPEN_24H(
            stringRes = R.string.station_status_247,
            colorAttr = AppCompatR.attr.colorPrimary,
            colorSelector = { s -> s.primary },
            iconRes = R.drawable.ic_schedule,
        ) {
            override fun matches(
                status: OpeningStatus,
                deadline: Instant,
            ): Boolean = status.isOpen && status.until == null
        },
        CLOSES_SOON(
            stringRes = R.string.station_status_closes_at,
            colorAttr = AppCompatR.attr.colorError,
            colorSelector = { s -> s.error },
            iconRes = R.drawable.ic_warning,
            usesRelativeTime = true,
        ) {
            override fun matches(
                status: OpeningStatus,
                deadline: Instant,
            ): Boolean = status.isOpen && status.until != null && status.until < deadline
        },
        OPEN(
            stringRes = R.string.station_status_open,
            colorAttr = AppCompatR.attr.colorPrimary,
            colorSelector = { s -> s.primary },
            iconRes = R.drawable.ic_check_circle,
        ) {
            override fun matches(
                status: OpeningStatus,
                deadline: Instant,
            ): Boolean = status.isOpen && status.until != null && status.until >= deadline
        }, ;

        abstract fun matches(
            status: OpeningStatus,
            deadline: Instant,
        ): Boolean

        fun forHumans(
            context: Context,
            status: OpeningStatus,
        ): String =
            if (usesRelativeTime) {
                val relative = status.getRelativeTimeSpan(status.until!!, Instant.now())
                context.getString(stringRes, relative)
            } else {
                context.getString(stringRes)
            }
    }

    private fun resolveState(thresholdMinutes: Long): UiValues {
        val deadline = getDeadline(thresholdMinutes)
        return UiValues.entries.first { it.matches(this, deadline) }
    }

    fun forHumans(
        context: Context,
        thresholdMinutes: Long = defaultThresholdMinutes,
    ): String = resolveState(thresholdMinutes).forHumans(context, this)

    @ColorInt
    fun color(
        context: Context,
        thresholdMinutes: Long = defaultThresholdMinutes,
    ): Int {
        val typedValue = TypedValue()

        @AttrRes val colorAttr = resolveState(thresholdMinutes).colorAttr
        context.theme.resolveAttribute(colorAttr, typedValue, true)
        return typedValue.data
    }

    fun color(
        colorScheme: ColorScheme,
        thresholdMinutes: Long = defaultThresholdMinutes,
    ): Color = resolveState(thresholdMinutes).colorSelector(colorScheme)

    @DrawableRes
    fun icon(thresholdMinutes: Long = defaultThresholdMinutes): Int = resolveState(thresholdMinutes).iconRes

    private fun getDeadline(thresholdMinutes: Long): Instant = Instant.now().plus(thresholdMinutes, ChronoUnit.MINUTES)

    private fun getRelativeTimeSpan(
        until: Instant,
        now: Instant,
    ): String {
        val untilMillis = until.toEpochMilli()
        val nowMillis = now.toEpochMilli()

        return DateUtils
            .getRelativeTimeSpanString(
                untilMillis,
                nowMillis,
                DateUtils.MINUTE_IN_MILLIS,
                DateUtils.FORMAT_ABBREV_RELATIVE,
            ).toString()
            .lowercase()
    }
}

fun toLocal(
    instant: Instant,
    zoneId: ZoneId = ZoneId.of("Europe/Madrid"),
): Pair<DayOfWeek, LocalTime> {
    val localDateTime = instant.atZone(zoneId).toLocalDateTime()
    val truncatedTime = localDateTime.toLocalTime().truncatedTo(ChronoUnit.MINUTES)
    return localDateTime.dayOfWeek to truncatedTime
}

fun toInstant(
    reference: Instant,
    day: DayOfWeek,
    time: LocalTime,
    zoneId: ZoneId,
): Instant {
    val refZoned = reference.atZone(zoneId)
    val refDate = refZoned.toLocalDate()
    val refDay = refZoned.dayOfWeek
    val refDayValue = refDay.value
    val refTime = refZoned.toLocalTime()
    val targetDayValue = day.value

    val daysToAdd =
        if (day == refDay && time < refTime) {
            7
        } else {
            (targetDayValue - refDayValue + 7) % 7
        }

    val targetDate = refDate.plusDays(daysToAdd.toLong())
    val targetLocal = LocalDateTime.of(targetDate, time)
    return targetLocal.atZone(zoneId).toInstant()
}

class OpeningHours {
    private var currentDay: DayOfWeek = DayOfWeek.MONDAY
    private val dayIntervals = mutableMapOf<DayOfWeek, MutableList<Pair<LocalTime, LocalTime>>>()

    fun add(
        day: DayOfWeek,
        startHour: Int,
        startMinute: Int,
        endHour: Int,
        endMinute: Int,
    ) {
        val interval = LocalTime.of(startHour, startMinute) to LocalTime.of(endHour, endMinute)
        val intervals = dayIntervals.getOrPut(day) { mutableListOf() }

        val mergeStart =
            intervals
                .indexOfFirst {
                    interval.first <= it.second
                }.let {
                    if (it < 0) intervals.size else it
                }
        val mergeEnd =
            intervals
                .indexOfFirst {
                    interval.second < it.first
                }.let {
                    if (it < 0) intervals.size else it
                }

        if (mergeStart == mergeEnd) {
            intervals.add(mergeStart, interval)
            return
        }

        // merge
        val merged = (
            minOf(intervals[mergeStart].first, interval.first) to
                maxOf(intervals[mergeEnd - 1].second, interval.second)
        )
        intervals.subList(mergeStart, mergeEnd).clear()
        intervals.add(mergeStart, merged)
    }

    fun getStatus(
        instant: Instant,
        zoneId: ZoneId,
    ): OpeningStatus {
        val (day, time) = toLocal(instant, zoneId)

        fun openUntil(until: Pair<DayOfWeek, LocalTime>?): OpeningStatus {
            if (until == null) return OpeningStatus(isOpen = true, null)

            var (untilDay, untilTime) = until

            if (untilTime == LocalTime.of(23, 59)) {
                untilDay = untilDay + 1
                untilTime = LocalTime.MIDNIGHT
            }

            val untilInstant = toInstant(instant, untilDay, untilTime, zoneId)
            return OpeningStatus(isOpen = true, until = untilInstant)
        }

        fun closedUntil(until: Pair<DayOfWeek, LocalTime>?): OpeningStatus {
            if (until == null) return OpeningStatus(isOpen = false, null)
            val (untilDay, untilTime) = until
            val untilInstant = toInstant(instant, untilDay, untilTime, zoneId)
            return OpeningStatus(isOpen = false, until = untilInstant)
        }

        var closingAt: LocalTime? = null
        for ((start, end) in getDayIntervals(day)) {
            if (end < time) continue // ignore intervals in the past

            if (closingAt == null) { // no enclosing interval found yet
                if (start > time) {
                    // Next interval is in the future
                    // We are closed until that interval
                    return closedUntil(day to start)
                }
            } else { // Looking for closure
                // A gap is there
                if (start > closingAt.plusMinutes(1)) {
                    return openUntil(day to closingAt)
                }
            }
            closingAt = end
        }
        if (closingAt == null) {
            var nextOpening = searchNextOpening(day)
            return closedUntil(nextOpening)
        }
        var nextClosing = searchNextOpeningGap(day, closingAt)
        return openUntil(nextClosing)
    }

    private fun getDayIntervals(day: DayOfWeek): List<Pair<LocalTime, LocalTime>> =
        dayIntervals.getOrDefault(
            day,
            emptyList(),
        )

    private fun searchNextOpening(day: DayOfWeek): Pair<DayOfWeek, LocalTime>? {
        for (i in 1L..7L) {
            val nextDay = day + i
            for ((start, end) in getDayIntervals(nextDay)) {
                return nextDay to start
            }
        }
        return null
    }

    private fun searchNextOpeningGap(
        day: DayOfWeek,
        time: LocalTime,
    ): Pair<DayOfWeek, LocalTime>? {
        var closingAt = time
        var closingDay = day
        for (dayOffset in 1L..8L) {
            val nextDay = day + dayOffset
            val dayIntervals = getDayIntervals(nextDay)
            if (dayIntervals.isEmpty()) {
                return closingDay to closingAt
            }

            for ((start, end) in dayIntervals) {
                if (start != closingAt && start != closingAt?.plusMinutes(1)) {
                    // Gap detected
                    return closingDay to closingAt
                }
                closingAt = end
                closingDay = nextDay
            }
        }
        return null
    }

    override fun toString(): String {
        val dayStrings: List<Pair<String, String>> =
            DayOfWeek.values().map { day ->
                spanishWeekDayShort(day) to formatIntervals(dayIntervals[day] ?: emptyList())
            }

        var pivot = ""
        val result = mutableListOf<Pair<String, String>>()
        for (window in dayStrings.windowed(2, partialWindows = true)) {
            val (currentDay, currentInterval) = window[0]
            val nextInterval = window.getOrNull(1)?.second
            if (currentInterval.isEmpty()) continue
            if (nextInterval == currentInterval) {
                // repeated interval detected
                // set pivot if not set
                if (pivot.isEmpty()) {
                    pivot = "$currentDay-"
                }
                // Do not output yet
                continue
            }
            result.add(pivot + currentDay to currentInterval)
            pivot = "" // reset pivot
        }

        return result
            .map { (day, str) -> "$day: $str" }
            .joinToString("; ")
    }

    private fun formatIntervals(intervals: List<Pair<LocalTime, LocalTime>>): String =
        intervals
            .map {
                formatInterval(it)
            }.joinToString(" y ")

    private fun formatInterval(interval: Pair<LocalTime, LocalTime>): String {
        val (start, end) = interval
        if (start == LocalTime.MIDNIGHT && end == END_OF_DAY) {
            return "24H"
        }
        return "${formatTime(start)}-${formatTime(end)}"
    }

    private fun formatTime(time: LocalTime): String = String.format(Locale.ROOT, "%02d:%02d", time.hour, time.minute)

    private fun spanishWeekDayShort(day: DayOfWeek): String =
        when (day) {
            DayOfWeek.MONDAY -> "L"
            DayOfWeek.TUESDAY -> "M"
            DayOfWeek.WEDNESDAY -> "X"
            DayOfWeek.THURSDAY -> "J"
            DayOfWeek.FRIDAY -> "V"
            DayOfWeek.SATURDAY -> "S"
            DayOfWeek.SUNDAY -> "D"
        }

    companion object {
        fun parseTime(intervalStr: String): TimeSpec? {
            val parts = intervalStr.split(":")
            if (parts.size != 2) return null

            val hours = parts[0].toIntOrNull() ?: return null
            val minutes = parts[1].toIntOrNull() ?: return null

            if (hours !in 0..23) return null
            if (minutes !in 0..59) return null

            return hours to minutes
        }

        fun parseInterval(intervalStr: String): Interval? {
            if (intervalStr == "24H") return ((0 to 0) to (23 to 59))
            val parts = intervalStr.split("-")
            if (parts.size != 2) return null

            val start = parseTime(parts[0]) ?: return null
            val end = parseTime(parts[1]) ?: return null

            return start to end
        }

        fun parseIntervals(spec: String): Intervals? {
            val parts = spec.split(" y ")
            val intervals = parts.map { parseInterval(it) }
            if (intervals.any { it == null }) return null
            return intervals.map { it!! }
        }

        fun parseDayShort(spec: String): DayOfWeek? =
            when (spec) {
                "L" -> DayOfWeek.MONDAY
                "M" -> DayOfWeek.TUESDAY
                "X" -> DayOfWeek.WEDNESDAY
                "J" -> DayOfWeek.THURSDAY
                "V" -> DayOfWeek.FRIDAY
                "S" -> DayOfWeek.SATURDAY
                "D" -> DayOfWeek.SUNDAY
                else -> null
            }

        fun parseDayRange(spec: String): DayRange? {
            val parts = spec.split("-")
            val start = parts.getOrNull(0)?.let { parseDayShort(it) }
            if (start == null) return null

            if (parts.size == 1) return listOf(start)

            val end = parts[1]?.let { parseDayShort(it) }
            val allDays = DayOfWeek.values().toList()
            val startIndex = allDays.indexOf(start)
            val endIndex = allDays.indexOf(end)
            if (startIndex < endIndex) {
                return allDays.subList(startIndex, endIndex + 1)
            }
            // Crossed? cycle through end
            return allDays.subList(startIndex, allDays.size) + allDays.subList(0, endIndex + 1)
        }

        fun parseScheduleEntry(spec: String): ScheduleEntry? {
            val parts = spec.split(": ")
            if (parts.size != 2) return null
            val (daysStr, timesStr) = parts
            val days = parseDayRange(daysStr) ?: return null
            val times = parseIntervals(timesStr) ?: return null
            return days to times
        }

        fun parse(spec: String): OpeningHours? {
            val oh = OpeningHours()
            val entries = spec.split("; ")
            for (entrySpec in entries) {
                // println("Entry: '$entrySpec'")
                val (days, times) = parseScheduleEntry(entrySpec) ?: return null
                for (day in days) {
                    for (time in times) {
                        val (start, end) = time
                        val (sh, sm) = start
                        val (eh, em) = end
                        if (sh < eh || (sh == eh && sm <= em)) {
                            oh.add(day, sh, sm, eh, em)
                            continue
                        }
                        oh.add(day, sh, sm, 23, 59)
                        if ((eh to em) == (0 to 0)) {
                            continue
                        }
                        val nextDay = day + 1
                        oh.add(nextDay, 0, 0, eh, em)
                    }
                }
            }
            return oh
        }
    }
}
