//-----------------------------------------------------------------------
// <copyright file="FilesizeTests.cs" company="aliasvault">
// Copyright (c) aliasvault. All rights reserved.
// Licensed under the AGPLv3 license. See LICENSE.md file in the project root for full license information.
// </copyright>
//-----------------------------------------------------------------------

namespace AliasVault.UnitTests.Vault;

using AliasVault.Api.Helpers;

/// <summary>
/// Tests for the Vault FileSize calculation.
/// </summary>
public class FilesizeTests
{
    /// <summary>
    /// Test the FileSize calculation for small strings.
    /// </summary>
    [Test]
    public void FileSizeSmallTest()
    {
        var smallString = "dGVzdA==";
        Assert.That(FileHelper.Base64StringToKilobytes(smallString), Is.EqualTo(1));
    }

    /// <summary>
    /// Test the FileSize calculation for medium strings.
    /// </summary>
    [Test]
    public void FileSizeMediumTest()
    {
        var mediumString = "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";
        Assert.That(FileHelper.Base64StringToKilobytes(mediumString), Is.EqualTo(3));
    }

    /// <summary>
    /// Test the FileSize calculation for large strings.
    /// </summary>
    [Test]
    public void FileSizeLargeTest()
    {
        var largeString = "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";
        Assert.That(FileHelper.Base64StringToKilobytes(largeString), Is.EqualTo(32));
    }
}
