//-----------------------------------------------------------------------
// <copyright file="SrpArgonEncryptionTests.cs" company="aliasvault">
// Copyright (c) aliasvault. All rights reserved.
// Licensed under the AGPLv3 license. See LICENSE.md file in the project root for full license information.
// </copyright>
//-----------------------------------------------------------------------

namespace AliasVault.UnitTests.Utilities;

using System.Security.Cryptography;
using AliasVault.Cryptography.Client;
using SecureRemotePassword;

/// <summary>
/// Tests for the SrpArgonEncryption class.
/// </summary>
public class SrpArgonEncryptionTests
{
    /// <summary>
    /// Test basic encryption and decryption using default encryption logic (Argon2id and AES-256).
    /// </summary>
    /// <returns>Task.</returns>
    [Test]
    public async Task TestBasicEncrypt()
    {
        var stopwatch = System.Diagnostics.Stopwatch.StartNew();
        string password = "your-password";
        string salt = "your-salt"; // Use a secure random salt in production

        string plaintext = "Hello, World!";

        // Derive a key from the password using Argon2id
        byte[] key = await Encryption.DeriveKeyFromPasswordAsync(password, salt);
        Console.WriteLine($"Derived key: {key.Length} bytes (hex: {Convert.ToHexString(key).Replace("-", string.Empty)})");

        // SymmetricEncrypt the plaintext
        string encrypted = Cryptography.Server.Encryption.SymmetricEncrypt(plaintext, key);
        Console.WriteLine($"Encrypted: {encrypted}");

        Assert.That(encrypted, Is.Not.Null);
        Assert.That(encrypted, Is.Not.Empty);
        Assert.That(encrypted, Is.Not.EqualTo(plaintext));

        // SymmetricDecrypt the ciphertext
        string decrypted = Cryptography.Server.Encryption.SymmetricDecrypt(encrypted, key);
        Console.WriteLine($"Decrypted: {decrypted}");
        Assert.That(decrypted, Is.EqualTo(plaintext));

        stopwatch.Stop();
        var elapsedMilliseconds = stopwatch.ElapsedMilliseconds;
        Console.WriteLine($"Execution time: {elapsedMilliseconds} ms");
    }

    /// <summary>
    /// Test that the used JS encryption implementation is compatible with the C# encryption implementation.
    /// </summary>
    [Test]
    public void TestJsEncryption()
    {
        string jsEncryptionKeyBase64 = "vtIsIn3D9oZcQ2ssfXLwM6EduYbW3b1tFSZPzmhhy+Y=";
        string jsEncryptedBase64Contents = "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";
        var encryptionKey = Convert.FromBase64String(jsEncryptionKeyBase64);

        // Try to decrypt it from .NET.
        var decrypted = Cryptography.Server.Encryption.SymmetricDecrypt(jsEncryptedBase64Contents, encryptionKey);

        // Assert that its equal as the original what we expect.
        var originalUnencrypted = "U1FMaXRlIGZvcm1hdCAzABAAAQEAQCAgAAAAAQAAAAMAAAAAAAAAAAAAAAIAAAAEAAAAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABAC52iQ0P+AACDvUADvUPxQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAIFNAQcXHx8Bgml0YWJsZVBhc3N3b3Jkc1Bhc3N3b3JkcwJDUkVBVEUgVEFCTEUgIlBhc3N3b3JkcyIgKAogICAgIklkIiBURVhUIE5PVCBOVUxMIENPTlNUUkFJTlQgIlBLX1Bhc3N3b3JkcyIgUFJJTUFSWSBLRVksCiAgICAiVmFsdWUiIFRFWFQgTlVMTCwKICAgICJDcmVhdGVkQXQiIFRFWFQgTk9UIE5VTEwsCiAgICAiVXBkYXRlZEF0IiBURVhUIE5PVCBOVUxMCikxAgYXRR8BAGluZGV4c3FsaXRlX2F1dG9pbmRleF9QYXNzd29yZHNfMVBhc3N3b3JkcwMAAAAIAAAAAA0AAAADDrMAD5EPIg6zAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABtAwVVQTszNTVDMzNFQzYtQ0NDOC00QTI1LUI4MjgtNTJCQTQ5RUE1RDE3VGVzdCBmYWN0b3J5IGluc2VydCBTUUxpdGUyMDI0LTA2LTI0IDE0OjEwOjUxLjI3MjAwMDEtMDEtMDEgMDA6MDA6MDBtAgVVQTszRTk3ODAyREItOUVCMy00QzZCLUE3QjktNzQ2RUNGMzc0ODg4VGVzdCBmYWN0b3J5IGluc2VydCBTUUxpdGUyMDI0LTA2LTI0IDE0OjEwOjUxLjA0OTAwMDEtMDEtMDEgMDA6MDA6MDBtAQVVQTszNjk2M0I1RTktREE1RS00ODAwLTlDMjAtM0QwOTdCRUJCNDIyVGVzdCBmYWN0b3J5IGluc2VydCBTUUxpdGUyMDI0LTA2LTI0IDE0OjEwOjQ5LjY3MzAwMDEtMDEtMDEgMDA6MDA6MDAKAAAAAw+GAA/XD68PhgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACgDVQFFOTc4MDJEQi05RUIzLTRDNkItQTdCOS03NDZFQ0YzNzQ4ODgCJwNVCTY5NjNCNUU5LURBNUUtNDgwMC05QzIwLTNEMDk3QkVCQjQyMigDVQE1NUMzM0VDNi1DQ0M4LTRBMjUtQjgyOC01MkJBNDlFQTVEMTcD";
        Assert.That(decrypted, Is.EqualTo(originalUnencrypted));
    }

    /// <summary>
    /// Test basic encryption and decryption using default encryption logic (Argon2id and AES-256).
    /// </summary>
    /// <returns>Task.</returns>
    [Test]
    public async Task TestNotEqualsPassword()
    {
        string password = "your-password";
        string salt = "your-salt"; // Use a secure random salt in production

        string plaintext = "Hello, World!";

        // Derive a key from the password using Argon2id
        byte[] key = await Cryptography.Client.Encryption.DeriveKeyFromPasswordAsync(password, salt);

        // SymmetricEncrypt the plaintext
        string encrypted = Cryptography.Server.Encryption.SymmetricEncrypt(plaintext, key);

        // SymmetricDecrypt the ciphertext using a different key
        byte[] key2 = await Cryptography.Client.Encryption.DeriveKeyFromPasswordAsync("your-password2", salt);

        Assert.Throws<AuthenticationTagMismatchException>(() => Cryptography.Server.Encryption.SymmetricDecrypt(encrypted, key2));
    }

    /// <summary>
    /// Test the SRP authentication flow to ensure it works correctly.
    /// </summary>
    /// <returns>Async task.</returns>
    [Test]
    public async Task TestSrpAuthentication()
    {
        var email = "test@example.com";
        var password = "myPassword";

        // Signup -----------------------------
        // Client generates a salt and verifier.
        var client = new SrpClient();
        var salt = client.GenerateSalt();

        byte[] passwordHash = await Cryptography.Client.Encryption.DeriveKeyFromPasswordAsync(password, salt);
        var passwordHashString = Convert.ToHexString(passwordHash).Replace("-", string.Empty);
        var srpSignup = Srp.PasswordChangeAsync(client, salt, email, passwordHashString);

        var privateKey = srpSignup.PrivateKey;
        var verifier = srpSignup.Verifier;

        // Login -----------------------------------
        // 1. Client generates an ephemeral value.
        var clientEphemeral = Srp.GenerateEphemeralClient();

        // --> Then client sends request to server.

        // 2. Server retrieves salt and verifier from database.
        // Then server generates an ephemeral value as well.
        var serverEphemeral = Srp.GenerateEphemeralServer(verifier);

        // --> Send serverEphemeral.Public to client.

        // 3. Client derives shared session key.
        var clientSession = Srp.DeriveSessionClient(privateKey, clientEphemeral.Secret, serverEphemeral.Public, salt, email);

        // --> send session.Proof to server.

        // 4. Server verifies the proof.
        var serverSession = Srp.DeriveSessionServer(serverEphemeral.Secret, clientEphemeral.Public, salt, email, verifier, clientSession.Proof);

        // --> send serverSession.Proof to client.
        Assert.That(serverSession, Is.Not.Null, "Server session is null, likely indicating that the provided password is incorrect.");

        // 5. Client verifies the proof.
        Srp.VerifySession(clientEphemeral.Public, clientSession, serverSession.Proof);

        // Ensure the keys match.
        Assert.That(clientSession.Key, Is.EqualTo(serverSession.Key));
    }

    /// <summary>
    /// Test byte array encryption and decryption overload methods.
    /// </summary>
    /// <returns>Task.</returns>
    [Test]
    public async Task TestByteArrayEncryption()
    {
        string password = "your-password";
        string salt = "your-salt";

        // Derive a key from the password using Argon2id
        byte[] key = await Cryptography.Client.Encryption.DeriveKeyFromPasswordAsync(password, salt);

        // Generate random byte array of 1000 bytes
        byte[] plainBytes = new byte[1000];
        new Random().NextBytes(plainBytes);

        // SymmetricEncrypt the plainBytes
        var cipherBytes = Cryptography.Server.Encryption.SymmetricEncrypt(plainBytes, key);

        Assert.That(cipherBytes, Is.Not.Null);
        Assert.That(cipherBytes, Is.Not.Empty);
        Assert.That(cipherBytes, Is.Not.EqualTo(plainBytes));

        // SymmetricDecrypt the cipherBytes
        var decrypted = Cryptography.Server.Encryption.SymmetricDecrypt(cipherBytes, key);
        Assert.That(decrypted, Is.EqualTo(plainBytes));
    }
}
