import { type BinaryLike, type BufferLike, type TypedArray } from './Utils';
import type { KeyObjectHandle } from './NativeQuickCrypto/webcrypto';
import type { KeyPairKey } from './Cipher';
export declare const kNamedCurveAliases: {
    readonly 'P-256': "prime256v1";
    readonly 'P-384': "secp384r1";
    readonly 'P-521': "secp521r1";
};
export type NamedCurve = 'P-256' | 'P-384' | 'P-521';
export type ImportFormat = 'raw' | 'pkcs8' | 'spki' | 'jwk';
export type AnyAlgorithm = DigestAlgorithm | HashAlgorithm | KeyPairAlgorithm | SecretKeyAlgorithm | SignVerifyAlgorithm | DeriveBitsAlgorithm | EncryptDecryptAlgorithm | AESAlgorithm | 'PBKDF2' | 'HKDF' | 'unknown';
export type DigestAlgorithm = 'SHA-1' | 'SHA-256' | 'SHA-384' | 'SHA-512';
export type HashAlgorithm = DigestAlgorithm | 'SHA-224' | 'RIPEMD-160';
export type KeyPairType = 'rsa' | 'rsa-pss' | 'ec';
export type RSAKeyPairAlgorithm = 'RSASSA-PKCS1-v1_5' | 'RSA-PSS' | 'RSA-OAEP';
export type ECKeyPairAlgorithm = 'ECDSA' | 'ECDH';
export type CFRGKeyPairAlgorithm = 'Ed25519' | 'Ed448' | 'X25519' | 'X448';
export type AESAlgorithm = 'AES-CTR' | 'AES-CBC' | 'AES-GCM' | 'AES-KW';
export type KeyPairAlgorithm = RSAKeyPairAlgorithm | ECKeyPairAlgorithm | CFRGKeyPairAlgorithm;
export type SecretKeyAlgorithm = 'HMAC' | AESAlgorithm;
export type SecretKeyType = 'hmac' | 'aes';
export type SignVerifyAlgorithm = 'RSASSA-PKCS1-v1_5' | 'RSA-PSS' | 'ECDSA' | 'HMAC' | 'Ed25519' | 'Ed448';
export type DeriveBitsAlgorithm = 'PBKDF2' | 'HKDF' | 'ECDH' | 'X25519' | 'X448';
export type RsaOaepParams = {
    name: 'RSA-OAEP';
    label?: BufferLike;
};
export type AesCbcParams = {
    name: 'AES-CBC';
    iv: BufferLike;
};
export type AesCtrParams = {
    name: 'AES-CTR';
    counter: TypedArray;
    length: number;
};
export type AesGcmParams = {
    name: 'AES-GCM';
    iv: BufferLike;
    tagLength?: TagLength;
    additionalData?: BufferLike;
};
export type AesKwParams = {
    name: 'AES-KW';
    wrappingKey?: BufferLike;
};
export type AesKeyGenParams = {
    length: AESLength;
    name?: AESAlgorithm;
};
export type TagLength = 32 | 64 | 96 | 104 | 112 | 120 | 128;
export type AESLength = 128 | 192 | 256;
export type EncryptDecryptParams = AesCbcParams | AesCtrParams | AesGcmParams | RsaOaepParams;
export type EncryptDecryptAlgorithm = 'RSA-OAEP' | 'AES-CTR' | 'AES-CBC' | 'AES-GCM';
export type SubtleAlgorithm = {
    name: AnyAlgorithm;
    salt?: string;
    iterations?: number;
    hash?: HashAlgorithm | HashAlgorithmIdentifier;
    namedCurve?: NamedCurve;
    length?: number;
    modulusLength?: number;
    publicExponent?: number | Uint8Array;
};
export type HashAlgorithmIdentifier = {
    name: HashAlgorithm;
};
export type KeyUsage = 'encrypt' | 'decrypt' | 'sign' | 'verify' | 'deriveKey' | 'deriveBits' | 'wrapKey' | 'unwrapKey';
export declare enum KFormatType {
    kKeyFormatDER = 0,
    kKeyFormatPEM = 1,
    kKeyFormatJWK = 2
}
export type KFormat = 'der' | 'pem' | 'jwk';
export declare enum KeyType {
    Secret = 0,
    Public = 1,
    Private = 2
}
export type KTypePrivate = 'pkcs1' | 'pkcs8' | 'sec1';
export type KTypePublic = 'pkcs1' | 'spki';
export type KType = KTypePrivate | KTypePublic;
export declare enum KWebCryptoKeyFormat {
    kWebCryptoKeyFormatRaw = 0,
    kWebCryptoKeyFormatPKCS8 = 1,
    kWebCryptoKeyFormatSPKI = 2,
    kWebCryptoKeyFormatJWK = 3
}
export declare enum WebCryptoKeyExportStatus {
    OK = 0,
    INVALID_KEY_TYPE = 1,
    FAILED = 2
}
export declare enum KeyEncoding {
    kKeyEncodingPKCS1 = 0,
    kKeyEncodingPKCS8 = 1,
    kKeyEncodingSPKI = 2,
    kKeyEncodingSEC1 = 3
}
export type DSAEncoding = 'der' | 'ieee-p1363';
export type EncodingOptions = {
    key?: any;
    type?: KType;
    encoding?: string;
    dsaEncoding?: DSAEncoding;
    format?: KFormat;
    padding?: number;
    cipher?: string;
    passphrase?: BinaryLike;
    saltLength?: number;
    oaepHash?: string;
    oaepLabel?: BinaryLike;
};
export type AsymmetricKeyType = 'rsa' | 'rsa-pss' | 'dsa' | 'ec' | undefined;
export type JWK = {
    kty?: 'AES' | 'RSA' | 'EC' | 'oct';
    use?: 'sig' | 'enc';
    key_ops?: KeyUsage[];
    alg?: string;
    crv?: string;
    kid?: string;
    x5u?: string;
    x5c?: string[];
    x5t?: string;
    'x5t#256'?: string;
    n?: string;
    e?: string;
    d?: string;
    p?: string;
    q?: string;
    x?: string;
    y?: string;
    k?: string;
    dp?: string;
    dq?: string;
    qi?: string;
    ext?: boolean;
};
export type CryptoKeyPair = {
    publicKey: KeyPairKey;
    privateKey: KeyPairKey;
};
export declare enum CipherOrWrapMode {
    kWebCryptoCipherEncrypt = 0,
    kWebCryptoCipherDecrypt = 1
}
export declare function preparePrivateKey(key: BinaryLike | EncodingOptions): {
    format: KFormatType;
    data: ArrayBuffer;
    type?: KeyEncoding;
    passphrase?: BinaryLike;
};
export declare function preparePublicOrPrivateKey(key: BinaryLike | EncodingOptions): {
    format: KFormatType;
    data: ArrayBuffer;
    type?: KeyEncoding;
    passphrase?: BinaryLike;
};
export declare function parsePublicKeyEncoding(enc: EncodingOptions, keyType: string | undefined, objName?: string): {
    format: KFormatType;
    type: KeyEncoding | undefined;
    cipher: string | undefined;
    passphrase: ArrayBuffer | undefined;
};
export declare function parsePrivateKeyEncoding(enc: EncodingOptions, keyType: string | undefined, objName?: string): {
    format: KFormatType;
    type: KeyEncoding | undefined;
    cipher: string | undefined;
    passphrase: ArrayBuffer | undefined;
};
export declare function createSecretKey(key: BinaryLike, encoding?: string): SecretKeyObject;
export declare function createPublicKey(key: BinaryLike | EncodingOptions): PublicKeyObject;
export declare const createPrivateKey: (key: BinaryLike | EncodingOptions) => PrivateKeyObject;
export declare class CryptoKey {
    keyObject: KeyObject;
    keyAlgorithm: SubtleAlgorithm;
    keyUsages: KeyUsage[];
    keyExtractable: boolean;
    constructor(keyObject: KeyObject, keyAlgorithm: SubtleAlgorithm, keyUsages: KeyUsage[], keyExtractable: boolean);
    inspect(_depth: number, _options: any): any;
    get type(): "public" | "private" | "secret" | "unknown";
    get extractable(): boolean;
    get algorithm(): SubtleAlgorithm;
    get usages(): KeyUsage[];
}
declare class KeyObject {
    handle: KeyObjectHandle;
    type: 'public' | 'secret' | 'private' | 'unknown';
    export(_options?: EncodingOptions): ArrayBuffer;
    constructor(type: string, handle: KeyObjectHandle);
}
export declare class SecretKeyObject extends KeyObject {
    constructor(handle: KeyObjectHandle);
    export(options?: EncodingOptions): ArrayBuffer;
}
declare class AsymmetricKeyObject extends KeyObject {
    constructor(type: string, handle: KeyObjectHandle);
    private _asymmetricKeyType?;
    get asymmetricKeyType(): AsymmetricKeyType;
}
export declare class PublicKeyObject extends AsymmetricKeyObject {
    constructor(handle: KeyObjectHandle);
    export(options: EncodingOptions): ArrayBuffer;
}
export declare class PrivateKeyObject extends AsymmetricKeyObject {
    constructor(handle: KeyObjectHandle);
    export(options: EncodingOptions): ArrayBuffer;
}
export declare const isCryptoKey: (obj: any) => boolean;
export {};
//# sourceMappingURL=keys.d.ts.map