"use strict";

import { NativeQuickCrypto } from './NativeQuickCrypto/NativeQuickCrypto';
import { toArrayBuffer, binaryLikeToArrayBuffer } from './Utils';
import Stream from 'readable-stream';
import { Buffer } from '@craftzdog/react-native-buffer';
const createInternalHmac = NativeQuickCrypto.createHmac;
export function createHmac(algorithm, key, options) {
  return new Hmac(algorithm, key, options);
}
class Hmac extends Stream.Transform {
  isFinalized = false;
  constructor(algorithm, key,
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  _options) {
    super();
    const keyAsString = binaryLikeToArrayBuffer(key);
    if (keyAsString === undefined) {
      throw 'Wrong key type';
    }
    this.internalHmac = createInternalHmac(algorithm, keyAsString);
  }

  /**
   * Updates the `Hmac` content with the given `data`, the encoding of which
   * is given in `inputEncoding`.
   * If `encoding` is not provided, and the `data` is a string, an
   * encoding of `'utf8'` is enforced. If `data` is a `Buffer`, `TypedArray`, or`DataView`, then `inputEncoding` is ignored.
   *
   * This can be called many times with new data as it is streamed.
   * @since v0.1.94
   * @param inputEncoding The `encoding` of the `data` string.
   */
  update(data, inputEncoding) {
    if (data instanceof ArrayBuffer) {
      this.internalHmac.update(data);
      return this;
    }
    if (typeof data === 'string') {
      const buffer = Buffer.from(data, inputEncoding);
      this.internalHmac.update(toArrayBuffer(buffer));
      return this;
    }
    this.internalHmac.update(binaryLikeToArrayBuffer(data));
    return this;
  }
  _transform(chunk, encoding, callback) {
    this.update(chunk, encoding);
    callback();
  }
  _flush(callback) {
    this.push(this.digest());
    callback();
  }

  /**
   * Calculates the HMAC digest of all of the data passed using `hmac.update()`.
   * If `encoding` is
   * provided a string is returned; otherwise a `Buffer` is returned;
   *
   * The `Hmac` object can not be used again after `hmac.digest()` has been
   * called. Multiple calls to `hmac.digest()` will result in an error being thrown.
   * @since v0.1.94
   * @param encoding The `encoding` of the return value.
   */

  digest(encoding) {
    const result = this.isFinalized ? new ArrayBuffer(0) : this.internalHmac.digest();
    this.isFinalized = true;
    if (encoding && encoding !== 'buffer') {
      return Buffer.from(result).toString(encoding);
    }
    return Buffer.from(result);
  }
}
//# sourceMappingURL=Hmac.js.map