import { CombinedFieldPatterns, CombinedGenderOptionPatterns, CombinedStopWords } from "./FieldPatterns";
import { FormFields } from "./types/FormFields";

/**
 * Form detector.
 */
export class FormDetector {
  private readonly document: Document;
  private readonly clickedElement: HTMLElement | null;
  private readonly visibilityCache: Map<HTMLElement, boolean>;

  /**
   * Constructor.
   */
  public constructor(document: Document, clickedElement?: HTMLElement) {
    this.document = document;
    this.clickedElement = clickedElement ?? null;
    this.visibilityCache = new Map();
  }

  /**
   * Detect login forms on the page based on the clicked element.
   */
  public containsLoginForm(): boolean {
    let formWrapper = this.getFormWrapper();
    if (formWrapper?.getAttribute('role') === 'dialog') {
      // If we hit a dialog, search for form only within the dialog
      formWrapper = formWrapper.querySelector('form') as HTMLElement | null ?? formWrapper;
    }

    if (!formWrapper) {
      // If no form or dialog found, fallback to document.body
      formWrapper = this.document.body as HTMLElement;
    }

    /**
     * Sanity check: if form contains more than 150 inputs, don't process as this is likely not a login form.
     * This is a simple way to prevent processing large forms that are not login forms and making the browser page unresponsive.
     */
    const inputCount = formWrapper.querySelectorAll('input').length;
    if (inputCount > 200) {
      return false;
    }

    // Check if the wrapper contains a password or likely username field before processing.
    if (this.containsPasswordField(formWrapper) || this.containsLikelyUsernameOrEmailField(formWrapper)) {
      return true;
    }

    return false;
  }

  /**
   * Detect login forms on the page based on the clicked element.
   */
  public getForm(): FormFields | null {
    if (!this.clickedElement) {
      return null;
    }

    const formWrapper = this.getFormWrapper();
    return this.detectFormFields(formWrapper);
  }

  /**
   * Get suggested service names from the page title and URL.
   * Returns an array with two suggestions: the primary name and the domain name as an alternative.
   */
  public static getSuggestedServiceName(document: Document, location: Location): string[] {
    const title = document.title;
    const maxWords = 4;
    const maxLength = 50;

    /**
     * We apply a limit to the length and word count of the title to prevent
     * the service name from being too long or containing too many words which
     * is not likely to be a good service name.
     */
    const validLength = (text: string): boolean => {
      const validLength = text.length >= 3 && text.length <= maxLength;
      const validWordCount = text.split(/[\s|\-—/\\]+/).length <= maxWords;
      return validLength && validWordCount;
    };

    /**
     * Filter out common words from prefix/suffix until no more matches found
     */
    const getMeaningfulTitleParts = (title: string): string[] => {
      const words = title.toLowerCase().split(' ').map(word => word.toLowerCase());

      // Strip stopwords from start until no more matches
      let startIndex = 0;
      while (startIndex < words.length && CombinedStopWords.has(words[startIndex].toLowerCase())) {
        startIndex++;
      }

      // Strip stopwords from end until no more matches
      let endIndex = words.length - 1;
      while (endIndex > startIndex && CombinedStopWords.has(words[endIndex].toLowerCase())) {
        endIndex--;
      }

      // Return remaining words
      return words.slice(startIndex, endIndex + 1);
    };

    /**
     * Get original case version of meaningful words
     */
    const getOriginalCase = (text: string, meaningfulParts: string[]): string => {
      return text
        .split(/[\s|]+/)
        .filter(word => meaningfulParts.includes(word.toLowerCase()))
        .join(' ');
    };

    // Domain name suggestion (always included as fallback or first suggestion)
    const domainSuggestion = location.hostname.replace(/^www\./, '');

    // First try to extract meaningful parts based on the divider
    const dividerRegex = /[|\-—/\\:]/;
    const dividerMatch = dividerRegex.exec(title);
    if (dividerMatch) {
      const dividerIndex = dividerMatch.index;
      const beforeDivider = title.substring(0, dividerIndex).trim();
      const afterDivider = title.substring(dividerIndex + 1).trim();

      // Count meaningful words on each side
      const beforeWords = getMeaningfulTitleParts(beforeDivider);
      const afterWords = getMeaningfulTitleParts(afterDivider);

      // Get both parts in original case
      const beforePart = getOriginalCase(beforeDivider, beforeWords);
      const afterPart = getOriginalCase(afterDivider, afterWords);

      // Check if both parts are valid
      const beforeValid = validLength(beforePart);
      const afterValid = validLength(afterPart);

      // If both parts are valid, return both as suggestions
      if (beforeValid && afterValid) {
        return [beforePart, afterPart, domainSuggestion];
      }

      // If only one part is valid, return it
      if (beforeValid) {
        return [beforePart, domainSuggestion];
      }
      if (afterValid) {
        return [afterPart, domainSuggestion];
      }
    }

    // If no meaningful parts found after divider, try the full title
    const meaningfulParts = getMeaningfulTitleParts(title);
    const serviceName = getOriginalCase(title, meaningfulParts);
    if (validLength(serviceName)) {
      return [serviceName, domainSuggestion];
    }

    // Fall back to domain name
    return [domainSuggestion];
  }

  /**
   * Get the form wrapper element.
   */
  private getFormWrapper(): HTMLElement | null {
    return this.clickedElement?.closest('form, [role="dialog"]') as HTMLElement | null;
  }

  /**
   * Get the actual input element from a potentially custom element.
   * This handles any element with shadow DOM containing input elements.
   * @param element The element to check (could be a custom element or regular input)
   * @returns The actual input element, or the original element if no nested input is found
   */
  private getActualInputElement(element: HTMLElement): HTMLElement {
    // If it's already an input, return it
    if (element.tagName.toLowerCase() === 'input') {
      return element;
    }

    // Check for shadow DOM input (generic approach)
    const elementWithShadow = element as HTMLElement & { shadowRoot?: ShadowRoot };
    if (elementWithShadow.shadowRoot) {
      const shadowInput = elementWithShadow.shadowRoot.querySelector('input, textarea') as HTMLElement;
      if (shadowInput) {
        return shadowInput;
      }
    }

    // Check for regular child input (non-shadow DOM)
    const childInput = element.querySelector('input, textarea') as HTMLElement;
    if (childInput) {
      return childInput;
    }

    // Return the original element if no nested input found
    return element;
  }

  /**
   * Check if an element and all its parents are visible.
   * This checks for display:none, visibility:hidden, and opacity:0
   * Uses a cache to avoid redundant checks of the same elements.
   */
  private isElementVisible(element: HTMLElement | null): boolean {
    if (!element) {
      return false;
    }

    // Check cache first
    if (this.visibilityCache.has(element)) {
      return this.visibilityCache.get(element)!;
    }

    let current: HTMLElement | null = element;
    while (current) {
      try {
        const style = this.document.defaultView?.getComputedStyle(current);
        if (!style) {
          // Cache and return true for this element and all its parents
          let parent: HTMLElement | null = current;
          while (parent) {
            this.visibilityCache.set(parent, true);
            parent = parent.parentElement;
          }
          return true;
        }

        // Check for display:none
        if (style.display === 'none') {
          // Cache and return false for this element and all its parents
          let parent: HTMLElement | null = current;
          while (parent) {
            this.visibilityCache.set(parent, false);
            parent = parent.parentElement;
          }
          return false;
        }

        // Check for visibility:hidden
        if (style.visibility === 'hidden') {
          // Cache and return false for this element and all its parents
          let parent: HTMLElement | null = current;
          while (parent) {
            this.visibilityCache.set(parent, false);
            parent = parent.parentElement;
          }
          return false;
        }

        // Check for opacity:0
        if (parseFloat(style.opacity) === 0) {
          // Cache and return false for this element and all its parents
          let parent: HTMLElement | null = current;
          while (parent) {
            this.visibilityCache.set(parent, false);
            parent = parent.parentElement;
          }
          return false;
        }
      } catch {
        // If we can't get computed style, cache and return true for this element and all its parents
        let parent: HTMLElement | null = current;
        while (parent) {
          this.visibilityCache.set(parent, true);
          parent = parent.parentElement;
        }
        return true;
      }

      current = current.parentElement;
    }

    // Cache and return true for the original element
    this.visibilityCache.set(element, true);
    return true;
  }

  /**
   * Find all input/select elements matching patterns and types, ordered by best match.
   */
  private findAllInputFields(
    form: HTMLFormElement | null,
    patterns: string[],
    types: string[],
    excludeElements: HTMLInputElement[] = []
  ): HTMLInputElement[] {
    // Query for standard input elements, select elements, and elements with type attributes
    const standardCandidates = form
      ? Array.from(form.querySelectorAll<HTMLElement>('input, select, [type]'))
      : Array.from(this.document.querySelectorAll<HTMLElement>('input, select, [type]'));

    /**
     * Also find any custom elements that might contain shadow DOM inputs
     * Look for elements with shadow roots that contain input elements
     */
    const allElements = form
      ? Array.from(form.querySelectorAll<HTMLElement>('*'))
      : Array.from(this.document.querySelectorAll<HTMLElement>('*'));

    const shadowDOMCandidates = allElements.filter(el => {
      // Check if element has shadow DOM with input elements
      const elementWithShadow = el as HTMLElement & { shadowRoot?: ShadowRoot };
      if (elementWithShadow.shadowRoot) {
        const shadowInput = elementWithShadow.shadowRoot.querySelector('input, textarea');
        return shadowInput !== null;
      }
      return false;
    });

    // Combine and deduplicate candidates
    const allCandidates = [...standardCandidates, ...shadowDOMCandidates];
    const candidates = allCandidates.filter((el, index, arr) => arr.indexOf(el) === index);

    const matches: { input: HTMLInputElement; score: number }[] = [];

    for (const input of Array.from(candidates)) {
      if (excludeElements.includes(input as HTMLInputElement)) {
        continue;
      }

      if (!this.isElementVisible(input)) {
        continue;
      }

      // Get type from either the element's type property or its type attribute
      const tagName = input.tagName.toLowerCase();
      let type = tagName === 'select'
        ? 'select'
        : (input as HTMLInputElement).type?.toLowerCase() || input.getAttribute('type')?.toLowerCase() || '';

      // Check if element has shadow DOM with input elements (generic detection)
      const elementWithShadow = input as HTMLElement & { shadowRoot?: ShadowRoot };
      const hasShadowDOMInput = elementWithShadow.shadowRoot &&
        elementWithShadow.shadowRoot.querySelector('input, textarea');

      // For elements with shadow DOM, get the type from the actual input inside
      if (hasShadowDOMInput && !type) {
        const shadowInput = elementWithShadow.shadowRoot!.querySelector('input, textarea') as HTMLInputElement;
        if (shadowInput) {
          type = shadowInput.type?.toLowerCase() || 'text';
        }
      }

      // Check if this element should be considered based on type matching
      if (!types.includes(type)) {
        // For shadow DOM elements, allow if we're looking for text and it contains an input
        if (hasShadowDOMInput && types.includes('text') && !type) {
          // This is a shadow DOM element without explicit type, treat as text input
        } else {
          continue;
        }
      }

      if (types.includes('email') && type === 'email') {
        matches.push({ input: input as HTMLInputElement, score: -1 });
        continue;
      }

      // Collect all text attributes to check
      const attributesToCheck = [
        input.id,
        input.getAttribute('name'),
        input.getAttribute('placeholder')
      ]
        .map(a => a?.toLowerCase() ?? '');

      // Check for associated labels if input has an ID or name
      if (input.id || input.getAttribute('name')) {
        const label = this.document.querySelector(`label[for="${input.id || input.getAttribute('name')}"]`);
        if (label) {
          attributesToCheck.push(label.textContent?.toLowerCase() ?? '');
        }
      }

      /**
       * Check for slot-based labels (e.g., <span slot="label">Email or username</span>)
       * Look for slot elements within the input's parent hierarchy
       */
      let slotParent: HTMLElement | null = input;
      for (let depth = 0; depth < 3 && slotParent; depth++) {
        const slotElements = slotParent.querySelectorAll('[slot="label"], [slot="helper-text"]');
        for (const slotEl of Array.from(slotElements)) {
          const slotText = slotEl.textContent?.toLowerCase() ?? '';
          if (slotText) {
            attributesToCheck.push(slotText);
          }
        }
        /** Also check if the parent itself is a custom element with slots */
        if (slotParent.shadowRoot) {
          const shadowSlots = slotParent.shadowRoot.querySelectorAll('slot[name="label"], slot[name="helper-text"]');
          for (const slot of Array.from(shadowSlots)) {
            const assignedNodes = (slot as HTMLSlotElement).assignedNodes();
            for (const node of assignedNodes) {
              if (node.textContent) {
                attributesToCheck.push(node.textContent.toLowerCase());
              }
            }
          }
        }
        slotParent = slotParent.parentElement;
      }

      // Check for sibling elements with class containing "label"
      const parent = input.parentElement;
      if (parent) {
        for (const sib of Array.from(parent.children)) {
          if (
            sib !== input &&
            Array.from(sib.classList).some(c => c.toLowerCase().includes('label'))
          ) {
            attributesToCheck.push(sib.textContent?.toLowerCase() ?? '');
          }
        }
      }

      // Check for parent label and table cell structure
      let currentElement: HTMLElement | null = input;
      for (let depth = 0; depth < 5 && currentElement; depth++) {
        // Stop if we have too many child elements (near body)
        if (currentElement.children.length > 15) {
          break;
        }

        // Check for label - search both parent and child elements
        const childLabel = currentElement.querySelector('label');
        if (childLabel) {
          attributesToCheck.push(childLabel.textContent?.toLowerCase() ?? '');
          break;
        }

        // Check for table cell structure
        const td = currentElement.closest('td');
        if (td) {
          // Get the parent row
          const row = td.closest('tr');
          if (row) {
            // Check all sibling cells in the row
            for (const cell of Array.from(row.querySelectorAll('td'))) {
              if (cell !== td) {
                attributesToCheck.push(cell.textContent?.toLowerCase() ?? '');
                break;
              }
            }
          }
          break;
        }

        currentElement = currentElement.parentElement;
      }

      let bestIndex = patterns.length;
      for (let i = 0; i < patterns.length; i++) {
        if (attributesToCheck.some(a => a.includes(patterns[i]))) {
          bestIndex = i;
          break;
        }
      }
      if (bestIndex < patterns.length) {
        matches.push({ input: input as HTMLInputElement, score: bestIndex });
      }
    }

    return matches
      .sort((a, b) => a.score - b.score)
      .map(m => m.input);
  }

  /**
   * Find a single input/select element based on common patterns in its attributes.
   */
  private findInputField(
    form: HTMLFormElement | null,
    patterns: string[],
    types: string[],
    excludeElements: HTMLInputElement[] = []
  ): HTMLInputElement | null {
    const all = this.findAllInputFields(form, patterns, types, excludeElements);

    // Filter out parent-child duplicates and fields overlapping with excludeElements
    const filtered = this.filterOutNestedDuplicates(all, excludeElements);

    // if email type explicitly requested, prefer actual <input type="email">
    if (types.includes('email')) {
      const emailMatch = filtered.find(i => (i.type || '').toLowerCase() === 'email');
      if (emailMatch) {
        return emailMatch;
      }
    }
    return filtered.length > 0 ? filtered[0] : null;
  }

  /**
   * Find the email field in the form.
   */
  private findEmailField(form: HTMLFormElement | null): {
    primary: HTMLInputElement | null,
    confirm: HTMLInputElement | null
  } {
    // Find all email fields first
    const emailFields = this.findAllInputFields(
      form,
      CombinedFieldPatterns.email,
      ['text', 'email']
    );

    // Filter out parent-child relationships
    const filteredEmailFields = this.filterOutNestedDuplicates(emailFields);
    const primaryEmail = filteredEmailFields[0] ?? null;

    /*
     * Find confirmation email field if primary exists
     * and ensure it's not the same as the primary email field.
     */
    const confirmEmailFields = primaryEmail
      ? this.findAllInputFields(
        form,
        CombinedFieldPatterns.emailConfirm,
        ['text', 'email'],
        [primaryEmail]
      )
      : [];

    const filteredConfirmFields = this.filterOutNestedDuplicates(confirmEmailFields);
    const confirmEmail = filteredConfirmFields[0] ?? null;

    return {
      primary: primaryEmail,
      confirm: confirmEmail
    };
  }

  /**
   * Find the birthdate fields in the form.
   */
  private findBirthdateFields(form: HTMLFormElement | null, excludeElements: HTMLInputElement[] = []): FormFields['birthdateField'] {
    // First try to find a single date input
    const singleDateField = this.findInputField(form, CombinedFieldPatterns.birthdate, ['date', 'text'], excludeElements);

    // Detect date format by searching all text content in the form
    let format = 'yyyy-mm-dd'; // default format
    if (form && singleDateField) {
      // Get the parent container
      const container = singleDateField.closest('div');
      if (container) {
        // Collect text from all relevant elements
        const elements = [
          ...Array.from(container.getElementsByTagName('label')),
          ...Array.from(container.getElementsByTagName('span')),
          container
        ];

        const allText = elements
          .map(el => el.textContent?.toLowerCase() ?? '')
          .join(' ')
          // Normalize different types of spaces and separators
          .replace(/[\s\u00A0]/g, '')
          // Don't replace separators yet to detect the preferred one
          .toLowerCase();

        // Check for date format patterns with either slash or dash
        if (/dd[-/]mm[-/]jj/i.test(allText) || /dd[-/]mm[-/]yyyy/i.test(allText)) {
          // Determine separator style from the matched pattern
          format = allText.includes('/') ? 'dd/mm/yyyy' : 'dd-mm-yyyy';
        } else if (/mm[-/]dd[-/]yyyy/i.test(allText)) {
          format = allText.includes('/') ? 'mm/dd/yyyy' : 'mm-dd-yyyy';
        } else if (/yyyy[-/]mm[-/]dd/i.test(allText)) {
          format = allText.includes('/') ? 'yyyy/mm/dd' : 'yyyy-mm-dd';
        }

        // Check placeholder as fallback
        if (format === 'yyyy-mm-dd' && singleDateField.placeholder) {
          const placeholder = singleDateField.placeholder.toLowerCase();
          if (/dd[-/]mm/i.test(placeholder)) {
            format = placeholder.includes('/') ? 'dd/mm/yyyy' : 'dd-mm-yyyy';
          } else if (/mm[-/]dd/i.test(placeholder)) {
            format = placeholder.includes('/') ? 'mm/dd/yyyy' : 'mm-dd-yyyy';
          }
        }
      }
    }

    if (singleDateField) {
      return {
        single: singleDateField,
        format,
        day: null,
        month: null,
        year: null
      };
    }

    // Look for separate day/month/year fields
    const dayField = this.findInputField(form, CombinedFieldPatterns.birthDateDay, ['text', 'number', 'select'], excludeElements);
    const monthField = this.findInputField(form, CombinedFieldPatterns.birthDateMonth, ['text', 'number', 'select'], excludeElements);
    const yearField = this.findInputField(form, CombinedFieldPatterns.birthDateYear, ['text', 'number', 'select'], excludeElements);

    return {
      single: null,
      format: 'yyyy-mm-dd', // Default format for separate fields
      day: dayField,
      month: monthField,
      year: yearField
    };
  }

  /**
   * Find the gender field in the form.
   */
  private findGenderField(form: HTMLFormElement | null, excludeElements: HTMLInputElement[] = []): FormFields['genderField'] {
    // Try to find select or input element using the shared method
    const genderField = this.findInputField(
      form,
      CombinedFieldPatterns.gender,
      ['select'],
      excludeElements
    );

    if (genderField?.tagName.toLowerCase() === 'select') {
      return {
        type: 'select',
        field: genderField
      };
    }

    // Try to find radio buttons
    const radioButtons = form
      ? form.querySelectorAll<HTMLInputElement>('input[type="radio"][name*="gender"], input[type="radio"][name*="sex"]')
      : null;

    if (radioButtons && radioButtons.length > 0) {
      /**
       * Find a radio button by patterns.
       */
      const findRadioByPatterns = (patterns: string[], isOther: boolean = false) : HTMLInputElement | null => {
        return Array.from(radioButtons).find(radio => {
          const attributes = [
            radio.value,
            radio.id,
            radio.name,
            radio.labels?.[0]?.textContent ?? ''
          ].map(attr => attr?.toLowerCase() ?? '');

          // For "other" patterns, skip if it matches male or female patterns
          if (isOther && (
            CombinedGenderOptionPatterns.male.some(pattern => attributes.some(attr => attr.includes(pattern))) ||
            CombinedGenderOptionPatterns.female.some(pattern => attributes.some(attr => attr.includes(pattern)))
          )) {
            return false;
          }

          return patterns.some(pattern =>
            attributes.some(attr => attr.includes(pattern))
          );
        }) ?? null;
      };

      return {
        type: 'radio',
        field: null, // Set to null since we're providing specific mappings
        radioButtons: {
          male: findRadioByPatterns(CombinedGenderOptionPatterns.male),
          female: findRadioByPatterns(CombinedGenderOptionPatterns.female),
          other: findRadioByPatterns(CombinedGenderOptionPatterns.other)
        }
      };
    }

    // Fall back to regular text input
    const textField = this.findInputField(form, CombinedFieldPatterns.gender, ['text'], excludeElements);

    return {
      type: 'text',
      field: textField
    };
  }

  /**
   * Filter out nested duplicates where a parent element and its child are both detected.
   * This happens with custom elements that contain actual input elements.
   * We prefer the innermost actual input element over the parent custom element.
   * Also excludes fields that overlap with already-detected fields.
   */
  private filterOutNestedDuplicates(fields: HTMLInputElement[], excludeElements: HTMLInputElement[] = []): HTMLInputElement[] {
    if (fields.length === 0) {
      return fields;
    }

    const filtered: HTMLInputElement[] = [];

    for (const field of fields) {
      let shouldInclude = true;

      // Check if this field overlaps with any excluded element
      for (const excluded of excludeElements) {
        // Skip if field is the same as excluded
        if (field === excluded) {
          shouldInclude = false;
          break;
        }

        // Skip if field is a child of excluded element
        if (excluded.contains(field)) {
          shouldInclude = false;
          break;
        }

        // Skip if field is a parent of excluded element
        if (field.contains(excluded)) {
          shouldInclude = false;
          break;
        }

        // Check shadow DOM relationships
        const fieldWithShadow = field as HTMLElement & { shadowRoot?: ShadowRoot };
        const excludedWithShadow = excluded as HTMLElement & { shadowRoot?: ShadowRoot };

        // Skip if excluded element's shadow DOM contains this field
        if (excludedWithShadow.shadowRoot && excludedWithShadow.shadowRoot.contains(field)) {
          shouldInclude = false;
          break;
        }

        // Skip if this field's shadow DOM contains the excluded element
        if (fieldWithShadow.shadowRoot && fieldWithShadow.shadowRoot.contains(excluded)) {
          shouldInclude = false;
          break;
        }

        // Get actual input elements and compare those
        const actualField = this.getActualInputElement(field);
        const actualExcluded = this.getActualInputElement(excluded);

        // Skip if the actual inputs are the same
        if (actualField === actualExcluded) {
          shouldInclude = false;
          break;
        }
      }

      if (!shouldInclude) {
        continue;
      }

      // Check if this field is a parent of any other field in the list
      for (const otherField of fields) {
        if (field !== otherField) {
          // Check if field contains otherField (field is parent)
          if (field.contains(otherField)) {
            shouldInclude = false;
            break;
          }

          // Check if field's shadow DOM contains otherField
          const fieldWithShadow = field as HTMLElement & { shadowRoot?: ShadowRoot };
          if (fieldWithShadow.shadowRoot && fieldWithShadow.shadowRoot.contains(otherField)) {
            shouldInclude = false;
            break;
          }
        }
      }

      if (shouldInclude) {
        // Also check if this field is not already represented by its actual input
        const actualInput = this.getActualInputElement(field);
        if (actualInput !== field) {
          // If the actual input is also in the list, skip the parent
          if (fields.includes(actualInput as HTMLInputElement)) {
            continue;
          }
        }

        filtered.push(field);
      }
    }

    return filtered;
  }

  /**
   * Find the password field in a form.
   */
  private findPasswordField(form: HTMLFormElement | null): {
    primary: HTMLInputElement | null,
    confirm: HTMLInputElement | null
  } {
    const passwordFields = this.findAllInputFields(form, CombinedFieldPatterns.password, ['password']);

    // Filter out parent-child relationships to avoid detecting the same field twice
    const filteredFields = this.filterOutNestedDuplicates(passwordFields);

    return {
      primary: filteredFields[0] ?? null,
      confirm: filteredFields[1] ?? null
    };
  }

  /**
   * Check if a form contains a password field.
   */
  private containsPasswordField(wrapper: HTMLElement): boolean {
    const passwordFields = this.findPasswordField(wrapper as HTMLFormElement | null);
    if (passwordFields.primary && this.isElementVisible(passwordFields.primary)) {
      return true;
    }

    return false;
  }

  /**
   * Check if a form contains a likely username or email field.
   */
  private containsLikelyUsernameOrEmailField(wrapper: HTMLElement): boolean {
    // Check if the form contains an email field.
    const emailFields = this.findEmailField(wrapper as HTMLFormElement | null);
    if (emailFields.primary && this.isElementVisible(emailFields.primary)) {
      return true;
    }

    // Check if the form contains a username field.
    const usernameField = this.findInputField(wrapper as HTMLFormElement | null, CombinedFieldPatterns.username, ['text'], []);
    if (usernameField && this.isElementVisible(usernameField)) {
      return true;
    }

    // Check if the form contains a first name field.
    const firstNameField = this.findInputField(wrapper as HTMLFormElement | null, CombinedFieldPatterns.firstName, ['text'], []);
    if (firstNameField && this.isElementVisible(firstNameField)) {
      return true;
    }

    // Check if the form contains a last name field.
    const lastNameField = this.findInputField(wrapper as HTMLFormElement | null, CombinedFieldPatterns.lastName, ['text'], []);
    if (lastNameField && this.isElementVisible(lastNameField)) {
      return true;
    }

    return false;
  }

  /**
   * Check if a field is an autofill-triggerable field (username, email, or password).
   */
  public isAutofillTriggerableField(): boolean {
    // Check if it's a username, email or password field by reusing the existing detection logic
    const formWrapper = this.getFormWrapper();

    if (!this.clickedElement) {
      return false;
    }

    // Get the actual input element (handles shadow DOM)
    const actualElement = this.getActualInputElement(this.clickedElement);

    // Check both the clicked element and the actual input element
    const elementsToCheck = [this.clickedElement, actualElement].filter((el, index, arr) =>
      el && arr.indexOf(el) === index // Remove duplicates
    );

    // Check if any of the elements is a username field
    const usernameFields = this.findAllInputFields(formWrapper as HTMLFormElement | null, CombinedFieldPatterns.username, ['text']);
    if (usernameFields.some(input => elementsToCheck.includes(input))) {
      return true;
    }

    // Check if any of the elements is a password field
    const passwordField = this.findPasswordField(formWrapper as HTMLFormElement | null);
    if ((passwordField.primary && elementsToCheck.includes(passwordField.primary)) ||
        (passwordField.confirm && elementsToCheck.includes(passwordField.confirm))) {
      return true;
    }

    // Check if any of the elements is an email field
    const emailFields = this.findAllInputFields(formWrapper as HTMLFormElement | null, CombinedFieldPatterns.email, ['text', 'email']);
    if (emailFields.some(input => elementsToCheck.includes(input))) {
      return true;
    }

    return false;
  }

  /**
   * Create a form entry.
   */
  private detectFormFields(wrapper: HTMLElement | null): FormFields {
    // Keep track of detected fields to prevent overlap
    const detectedFields: HTMLInputElement[] = [];

    // Find fields in priority order (most specific to least specific).
    const emailFields = this.findEmailField(wrapper as HTMLFormElement | null);
    if (emailFields.primary) {
      detectedFields.push(emailFields.primary);
    }
    if (emailFields.confirm) {
      detectedFields.push(emailFields.confirm);
    }

    const passwordFields = this.findPasswordField(wrapper as HTMLFormElement | null);
    if (passwordFields.primary) {
      detectedFields.push(passwordFields.primary);
    }
    if (passwordFields.confirm) {
      detectedFields.push(passwordFields.confirm);
    }

    const usernameField = this.findInputField(wrapper as HTMLFormElement | null, CombinedFieldPatterns.username, ['text'], detectedFields);
    if (usernameField) {
      detectedFields.push(usernameField);
    }

    const fullNameField = this.findInputField(wrapper as HTMLFormElement | null, CombinedFieldPatterns.fullName, ['text'], detectedFields);
    if (fullNameField) {
      detectedFields.push(fullNameField);
    }

    const lastNameField = this.findInputField(wrapper as HTMLFormElement | null, CombinedFieldPatterns.lastName, ['text'], detectedFields);
    if (lastNameField) {
      detectedFields.push(lastNameField);
    }

    const firstNameField = this.findInputField(wrapper as HTMLFormElement | null, CombinedFieldPatterns.firstName, ['text'], detectedFields);
    if (firstNameField) {
      detectedFields.push(firstNameField);
    }

    const birthdateField = this.findBirthdateFields(wrapper as HTMLFormElement | null, detectedFields);
    if (birthdateField.single) {
      detectedFields.push(birthdateField.single);
    }
    if (birthdateField.day) {
      detectedFields.push(birthdateField.day);
    }
    if (birthdateField.month) {
      detectedFields.push(birthdateField.month);
    }
    if (birthdateField.year) {
      detectedFields.push(birthdateField.year);
    }

    const genderField = this.findGenderField(wrapper as HTMLFormElement | null, detectedFields);
    if (genderField.field) {
      detectedFields.push(genderField.field as HTMLInputElement);
    }

    return {
      form: wrapper as HTMLFormElement,
      emailField: emailFields.primary,
      emailConfirmField: emailFields.confirm,
      usernameField,
      passwordField: passwordFields.primary,
      passwordConfirmField: passwordFields.confirm,
      fullNameField,
      firstNameField,
      lastNameField,
      birthdateField,
      genderField
    };
  }
}
