import org.jetbrains.kotlin.gradle.dsl.JvmTarget
import java.util.Properties

plugins {
    alias(libs.plugins.android.application)
    alias(libs.plugins.kotlin.android)
    alias(libs.plugins.kotlin.compose)
    alias(libs.plugins.kotlin.serialization)
    alias(libs.plugins.baselineprofile)
    alias(libs.plugins.sentry.android.gradle)
    id("version-tasks")
}

val versionPropsFile = rootProject.file("version.properties")
val versionProps = Properties().apply { load(versionPropsFile.inputStream()) }

val localPropsFile = rootProject.file("local.properties")
    .takeIf { it.exists() }
    ?.inputStream()
    ?.use { Properties().apply { load(it) } }

fun prop(name: String, default: String): String {
    return System.getenv(name)
        ?: localPropsFile?.getProperty(name)
        ?: default
}

val sentryPropsFile = rootProject.file("sentry.properties")
val sentryProps = Properties()
if (sentryPropsFile.exists()) {
    sentryProps.load(sentryPropsFile.inputStream())
}
fun sentryProp(name: String): String {
    return sentryProps.getProperty(name) ?: ""
}

android {
    namespace = "me.proton.android.lumo"
    compileSdk = 36

    defaultConfig {
        minSdk = 29
        targetSdk = 36
        versionName = "1.2.9"
        versionCode = 40

        testInstrumentationRunner = "androidx.test.runner.AndroidJUnitRunner"

        // Default production environment
        buildConfigField("String", "ENV_NAME", "\"\"")
        buildConfigField("String", "SENTRY_DSN", "\"${sentryProp("dsn")}\"")

        ndk {
            abiFilters += listOf(
                "armeabi-v7a",
                "arm64-v8a",
                "x86",
                "x86_64"
            )
        }
    }


    buildTypes {
        debug {
        }
        release {
            isMinifyEnabled = true
            isShrinkResources = true
            proguardFiles(
                getDefaultProguardFile("proguard-android-optimize.txt"),
                "proguard-rules.pro"
            )
        }
        create("alpha") {
            isMinifyEnabled = true
            isShrinkResources = true
            proguardFiles(
                getDefaultProguardFile("proguard-android-optimize.txt"),
                "proguard-rules.pro"
            )
        }
        create("benchmarkRelease") {
        }
        create("nonMinifiedRelease") {
        }
    }

    flavorDimensions += listOf("env", "services")
    productFlavors {
        create("gms") {
            dimension = "services"
            versionNameSuffix = "-gms"
        }
        create("noGms") {
            dimension = "services"
            versionNameSuffix = "-nogms"
        }

        create("production") {
            dimension = "env"
            applicationId = "me.proton.android.lumo"
            buildConfigField(
                "String",
                "BASE_DOMAIN",
                "\"${prop("BASE_DOMAIN_PRODUCTION", "proton.me")}\""
            )
            buildConfigField("String", "OFFER_ID", "\"${prop("OFFER_ID_PRODUCTION", "")}\"")
        }

        create("noble") {
            dimension = "env"
            applicationId = "me.proton.lumo"
            buildConfigField("String", "BASE_DOMAIN", "\"${prop("BASE_DOMAIN_NOBLE", "")}\"")
            buildConfigField("String", "OFFER_ID", "\"${prop("OFFER_ID_NOBLE", "")}\"")
        }
    }


    compileOptions {
        sourceCompatibility = JavaVersion.VERSION_17
        targetCompatibility = JavaVersion.VERSION_17
    }

    kotlin {
        compilerOptions {
            jvmTarget.set(JvmTarget.JVM_17)
        }
    }

    buildFeatures {
        compose = true
        buildConfig = true
    }

    // Optimize build performance
    packaging {
        resources {
            excludes += "/META-INF/{AL2.0,LGPL2.1}"
            excludes += "/META-INF/DEPENDENCIES"
        }
    }

    // Custom APK naming
    applicationVariants.all {
        val variant = this
        variant.outputs.all {
            val output = this as com.android.build.gradle.internal.api.BaseVariantOutputImpl
            val appName = "lumo"
            val versionName = variant.versionName
            val buildType = variant.buildType.name
            val flavor = variant.flavorName

            // Format: lumo-v0.1.2-production-debug.apk
            output.outputFileName = "${appName}-v${versionName}-${flavor}-${buildType}.apk"
        }
    }

    dependenciesInfo {
        // Disables dependency metadata when building APKs.
        includeInApk = false
        // Disables dependency metadata when building Android App Bundles.
        includeInBundle = false
    }
}

androidComponents {
    onVariants { variant ->
        val name = variant.name

        if (name == "productionGmsRelease" || name == "productionNoGmsRelease") {
            project.dependencies.add(
                "${name}BaselineProfile",
                project(":baselineprofile")
            )
        }
    }
}

dependencies {
    implementation(libs.androidx.core.ktx)
    implementation(libs.androidx.lifecycle.runtime.ktx)
    implementation(libs.androidx.activity.compose)
    implementation(platform(libs.androidx.compose.bom))
    implementation("androidx.lifecycle:lifecycle-viewmodel-ktx:${libs.versions.lifecycleRuntimeKtx.get()}")
    implementation("androidx.lifecycle:lifecycle-viewmodel-compose:${libs.versions.lifecycleRuntimeKtx.get()}")
    implementation("androidx.lifecycle:lifecycle-runtime-compose:${libs.versions.lifecycleRuntimeKtx.get()}")
    implementation(libs.androidx.ui)
    implementation(libs.androidx.ui.graphics)
    implementation(libs.androidx.ui.tooling.preview)
    implementation(libs.androidx.material3)
    implementation("androidx.compose.material:material-icons-core")
    implementation("androidx.compose.material:material-icons-extended")
    implementation("com.google.android.material:material:1.13.0")
    implementation("io.coil-kt:coil-compose:2.7.0")
    implementation("com.airbnb.android:lottie-compose:6.6.9")
    implementation(libs.androidx.navigation.compose)
    implementation(libs.kotlinx.serialization.json)
    implementation(libs.androidx.profileinstaller)
    implementation(libs.android.startup.runtime)

    implementation("com.alphacephei:vosk-android:0.3.70@aar")
    implementation("net.java.dev.jna:jna:5.18.1@aar")

    implementation(project(":vosk-model"))

//    "baselineProfile"(project(":baselineprofile"))

    "gmsImplementation"(libs.billing.ktx)
    "gmsImplementation"(libs.sentry)
    "gmsImplementation"(libs.sentry.android)

    // Hilt removed - using lightweight DependencyProvider instead

    testImplementation(libs.junit)
    testImplementation("org.jetbrains.kotlinx:kotlinx-coroutines-test:1.8.0")
    testImplementation("app.cash.turbine:turbine:1.1.0")
    testImplementation("io.mockk:mockk:1.13.11")
    testImplementation("io.mockk:mockk-android:1.13.11")
    testImplementation("androidx.arch.core:core-testing:2.2.0")
    androidTestImplementation(libs.androidx.junit)
    androidTestImplementation(libs.androidx.espresso.core)
    androidTestImplementation(platform(libs.androidx.compose.bom))
    androidTestImplementation(libs.androidx.ui.test.junit4)
    debugImplementation(libs.androidx.ui.tooling)
    debugImplementation(libs.androidx.ui.test.manifest)
}

sentry {
    autoInstallation {
        enabled = !isNoGms()
        sentryVersion = libs.versions.sentry.asProvider()
        autoUploadProguardMapping = isSentryAutoUploadEnabled()
        uploadNativeSymbols = isSentryAutoUploadEnabled()
    }

    tracingInstrumentation {
        enabled = false
    }

    ignoredVariants.set(
        listOf(
            "productionNoGmsDebug",
            "nobleNoGmsDebug",
            "productionNoGmsAlpha",
            "nobleNoGmsAlpha",
            "productionNoGmsRelease",
            "nobleNoGmsRelease",
        )
    )
}

fun isNoGms(): Boolean = gradle.startParameter.taskNames.any {
    it.contains("noGms", true)
}

fun isSentryAutoUploadEnabled(): Boolean = gradle.startParameter.taskNames.any {
    it.contains("productionGmsRelease", true)
}