package me.lucky.silence.ui

import android.Manifest
import android.app.role.RoleManager
import android.content.Context
import android.os.Build
import android.widget.Toast
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.navigation.compose.rememberNavController
import me.lucky.silence.Modem
import me.lucky.silence.Preferences
import me.lucky.silence.R
import me.lucky.silence.Utils
import me.lucky.silence.ui.common.ClickablePreference
import me.lucky.silence.ui.common.ClickableSwitchPreference
import me.lucky.silence.ui.common.Route
import me.lucky.silence.ui.common.SwitchPreference
import me.lucky.silence.ui.common.ToggleableButton

@Composable
fun ModuleList(modules: List<Module>) {
    Column {
        modules.forEach { module ->
            if ((module.getPreference != null) && (module.setPreference != null) && (module.navigation != null)) {
                ClickableSwitchPreference(
                    name = stringResource(module.name),
                    description = stringResource(module.description),
                    getIsEnabled = module.getPreference,
                    setIsEnabled = module.setPreference,
                    onModuleClick = { module.navigation.invoke() }
                )
            } else if (module.getPreference != null && module.setPreference != null) {
                SwitchPreference(
                    name = stringResource(module.name),
                    description = stringResource(module.description),
                    getIsEnabled = module.getPreference,
                    setIsEnabled = module.setPreference,
                )
            } else if (module.navigation != null) {
                ClickablePreference(
                    name = stringResource(module.name),
                    description = stringResource(module.description),
                    onModuleClick = { module.navigation.invoke() }
                )
            }
        }
    }
}

data class Module(
    val name: Int,
    val description: Int,
    val getPreference: (() -> Boolean)? = null,
    val setPreference: ((Boolean) -> Unit)? = null,
    val navigation: (() -> Unit)? = null,
)

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun MainScreen(
    ctx: Context,
    prefs: Preferences,
    onNavigateToContacted: () -> Unit,
    onNavigateToGroups: () -> Unit,
    onNavigateToRepeated: () -> Unit,
    onNavigateToMessages: () -> Unit,
    onNavigateToSim: () -> Unit,
    onNavigateToExtra: () -> Unit,
    onNavigateToSettings: () -> Unit,
    onNavigateToRegex: () -> Unit,
) {
    val scrollState = rememberScrollState()
    val blockEnabledToast = stringResource(R.string.block_enabled)
    val blockDisabledToast = stringResource(R.string.block_disabled)
    val registerForRepeatedPermissions =
        rememberLauncherForActivityResult(ActivityResultContracts.RequestPermission()) {}

    fun requestRepeatedPermissions() =
        registerForRepeatedPermissions.launch(Manifest.permission.READ_CALL_LOG)

    val roleManager: RoleManager by lazy { ctx.getSystemService(RoleManager::class.java) }
    val registerForCallScreeningRole =
        rememberLauncherForActivityResult(ActivityResultContracts.StartActivityForResult()) {}

    fun requestCallScreeningRole() =
        registerForCallScreeningRole
            .launch(roleManager.createRequestRoleIntent(RoleManager.ROLE_CALL_SCREENING))

    val modules = listOf(
        Module(
            name = R.string.repeated_main,
            description = R.string.repeated_description,
            getPreference = { prefs.isRepeatedEnabled },
            setPreference = { isChecked ->
                prefs.isRepeatedEnabled = isChecked
                if (isChecked) requestRepeatedPermissions()
            },
            navigation = onNavigateToRepeated,
        ),
        Module(
            name = R.string.regex_main,
            description = R.string.regex_description,
            getPreference = { prefs.isRegexEnabled },
            setPreference = { prefs.isRegexEnabled = it },
            navigation = onNavigateToRegex,
        ),
        Module(
            name = R.string.contacted_main,
            description = R.string.contacted_description,
            navigation = onNavigateToContacted,
        ),
        Module(
            name = R.string.groups_main,
            description = R.string.groups_description,
            navigation = onNavigateToGroups,
        ),
        Module(
            name = R.string.messages_main,
            description = R.string.messages_description,
            navigation = onNavigateToMessages,
        ),
        *(if (Utils.getModemCount(ctx, Modem.SUPPORTED) >= 2
            && Build.VERSION.SDK_INT >= Build.VERSION_CODES.S)
        {
            arrayOf(
                Module(
                    name = R.string.sim,
                    description = R.string.sim_description,
                    navigation = onNavigateToSim,
                ),
            )
        } else {
            emptyArray()
        }),
        Module(
            name = R.string.extra,
            description = R.string.extra_description,
            navigation = onNavigateToExtra,
        ),
    )
    var isBlockEnabled by remember { mutableStateOf(prefs.isBlockEnabled) }
    Scaffold(topBar = {
        TopAppBar(title = { Text(text = stringResource(R.string.app_name)) }, actions = {
            IconButton(onClick = {
                isBlockEnabled = !isBlockEnabled
                prefs.isBlockEnabled = isBlockEnabled
                Toast
                    .makeText(
                        ctx,
                        if (isBlockEnabled) blockEnabledToast else blockDisabledToast,
                        Toast.LENGTH_SHORT
                    )
                    .show()
            }) {
                Icon(
                    painter = painterResource(
                        id = if (isBlockEnabled) R.drawable.ic_baseline_do_not_disturb_on_24
                            else R.drawable.ic_baseline_do_not_disturb_off_24
                    ),
                    contentDescription = stringResource(R.string.block_main)
                )
            }
            IconButton(onClick = onNavigateToSettings) {
                Icon(
                    painter = painterResource(id = R.drawable.ic_baseline_settings_24),
                    contentDescription = stringResource(R.string.settings)
                )
            }
        })
    }, content = { padding ->
        Column(
            modifier = Modifier
                .padding(padding)
                .fillMaxSize()
        ) {
            Column(
                modifier = Modifier
                    .weight(1f, fill = true)
                    .verticalScroll(scrollState)
            ) {
                ModuleList(modules)
            }
            ToggleableButton(
                getPreference = { prefs.isEnabled },
                setPreference = { isChecked ->
                    prefs.isEnabled = isChecked
                    if (isChecked) requestCallScreeningRole()
                    Utils.updateMessagesEnabled(ctx)
                }
            )
        }
    })
}

@Preview
@Composable
fun ModuleScreenPreview() {
    val navController = rememberNavController()
    MainScreen(
        ctx = LocalContext.current,
        prefs = Preferences(LocalContext.current),
        onNavigateToContacted = { navController.navigate(Route.CONTACTED) },
        onNavigateToGroups = { navController.navigate(Route.GROUPS) },
        onNavigateToRepeated = { navController.navigate(Route.REPEATED) },
        onNavigateToMessages = { navController.navigate(Route.MESSAGES) },
        onNavigateToSim = { navController.navigate(Route.SIM) },
        onNavigateToExtra = { navController.navigate(Route.EXTRA) },
        onNavigateToSettings = { navController.navigate(Route.SETTINGS) },
        onNavigateToRegex = { navController.navigate(Route.REGEX) },
    )
}
