import { getHistory } from "./gameOver";
import { appVersion, icons, upgrades } from "./loadGameData";
import { t } from "./i18n/i18n";
import { asyncAlert } from "./asyncAlert";
import { getSettingValue, setSettingValue } from "./settings";

export function runHistoryViewerMenuEntry() {
  const history = getHistory();

  return {
    icon: icons["icon:history"],
    text: t("history.title"),
    disabled: history.length < 10,
    help:
      history.length < 10
        ? t("history.locked")
        : t("history.help", { count: history.length }),
    async value() {
      let sort = 0;
      let sortDir = -1;
      let columns = [
        {
          label: t("history.columns.started"),
          field: (r) => r.started,
          render(v) {
            return new Date(v).toISOString().slice(0, 10);
          },
        },
        {
          label: t("history.columns.score"),
          field: (r) => r.score,
        },
        ...upgrades.map((u) => ({
          label: icons["icon:" + u.id],
          tooltip: u.name,
          field: (r) => r.perks?.[u.id] || 0,
          render(v) {
            if (!v) return "-";
            return v;
          },
        })),
      ];
      while (true) {
        const hasCurrentVersion = history.find(
          (r) => r.appVersion === appVersion,
        );
        const hasPastVersion = history.find((r) => r.appVersion !== appVersion);

        const header = columns
          .map(
            (c, ci) =>
              `<th data-tooltip="${c.tooltip || ""}" data-resolve-to="sort:${ci}">${c.label}</th>`,
          )
          .join("");
        const toString = (v) => v.toString();
        const tbody = history
          .filter(
            (r) =>
              !hasCurrentVersion ||
              r.appVersion === appVersion ||
              getSettingValue("show_old_versions_in_stats", false),
          )
          .sort(
            (a, b) =>
              sortDir * (columns[sort].field(a) - columns[sort].field(b)),
          )
          .map(
            (h) =>
              "<tr>" +
              columns
                .map((c) => {
                  const value = c.field(h) ?? 0;
                  const render = c.render || toString;
                  return "<td>" + render(value) + "</td>";
                })
                .join("") +
              "</tr>",
          )
          .join("");

        const result = await asyncAlert({
          title: t("history.title"),
          className: "history",
          content: [
            `
<table>
<thead><tr>${header}</tr></thead>
<tbody>${tbody}</tbody>
</table>
                    `,
            hasPastVersion &&
              hasCurrentVersion && {
                icon: getSettingValue("show_old_versions_in_stats", false)
                  ? icons["icon:checkmark_checked"]
                  : icons["icon:checkmark_unchecked"],
                value: "toggle",
                text: t("history.include_past_versions"),
              },
          ],
        });
        if (!result) return;
        if (result.startsWith("sort:")) {
          const newSort = parseInt(result.split(":")[1]);
          if (newSort == sort) {
            sortDir *= -1;
          } else {
            sortDir = -1;
            sort = newSort;
          }
        }
        if (result === "toggle") {
          setSettingValue(
            "show_old_versions_in_stats",
            !getSettingValue("show_old_versions_in_stats", false),
          );
        }
      }
    },
  };
}
