#!/bin/python

import logging
import sys
import argparse

from enums import Actions, FileType
from strings import (
    process as process_strings,
    get_files as strings_files,
    move as move_strings
)
from plurals import (
    process as process_plurals,
    get_files as plurals_files,
    move as move_plurals
)
from utils import write_to_files, get_type 


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="A simple script to modify strings.xml or plurals.xml files in res directory."
    )

    parser.add_argument(
        'res', 
        nargs='+', # Accepts 1 or more arguments
        help='One or more files to process. The first file is considered the primary file.'
    )

    parser.add_argument(
        '--move', '-m', 
        action='store_true',
        help='Enable move mode. Requires 2 res directories.'
    )

    parser.add_argument(
        '--verbose', '-v', 
        action='store_true',
        help='Print more values for debug puposes.'
    )

    return parser.parse_args()


def main( args: argparse.Namespace ) -> None:
    logging.basicConfig(
        level=logging.DEBUG if args.verbose else logging.INFO
    )
    
    if args.move:
        if len(args.res) != 2:
            logging.error(f'Requires 2 res directories to move')
            exit(1)
        if args.res[0] == args.res[1]:
            logging.error(f'Requires 2 differennt res directories')
            exit(1)

    while True:
        file_type = get_type()
        logging.debug(f'File type: {file_type}')

        match (file_type):
            case FileType.STRINGS:
                if args.move:
                    move_strings( args.res[0], args.res[1] )
                else:
                    process_strings( args.res[0] )
            case FileType.PLURALS:
                if args.move:
                    move_plurals( args.res[0], args.res[1] )
                else:
                    process_plurals( args.res[0] )
        # end match

        print()
        print('Do you wish to continue?')
        print('\'Y\' or \'Yes\' brings to back to step 1 while keeping your changes')
        continue_choice: str = input('(y/N): ').strip().capitalize()
        if continue_choice != "Y" and continue_choice != "Yes":
            if strings_files:
                write_to_files(strings_files())
            if plurals_files:
                write_to_files(plurals_files())

            logging.info('Closing app...')
            break


if __name__ == '__main__':
    try:
        args = parse_args()
        main( args )
    except KeyboardInterrupt:
        print("")
        exit(0)

