package me.knighthat.discord.payload

import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject


/**
 * The ready event is dispatched when a client has completed the initial handshake with the gateway (for new sessions).
 * The ready event can be the largest and most complex event the gateway will send,
 * as it contains all the state required for a client to begin interacting with the rest of the platform.
 *
 * `guilds` are the guilds of which your bot is a member.
 * They start out as unavailable when you connect to the gateway.
 * As they become available, your bot will be notified via Guild Create events.
 *
 * [shard] and [guilds] can never be `null`. If response doesn't contain any of these keys,
 * an empty list will be assigned.
 *
 * @param apiVersion API version
 * @param user Information about the user including email
 * @param guilds Guilds the user is in
 * @param sessionId Used for resuming connections
 * @param resumeUrl Gateway URL for resuming connections
 * @param shard Shard information associated with this session, if sent when identifying
 * @param application Contains `id` and `flags`
 *
 * @see <a href='https://discord.com/developers/docs/reference#api-versioning-api-versions'>API version</a>
 * @see <a href='https://discord.com/developers/docs/events/gateway#sharding'>Shard information</a>
 */
@Serializable
internal data class Ready(
    @SerialName("v") val apiVersion: Int,
    val user: JsonObject,
    @SerialName("session_id") val sessionId: String,
    @SerialName("resume_gateway_url") val resumeUrl: String,
    val application: JsonObject? = null,
    val shard: List<Int> = emptyList(),
    val guilds: List<JsonObject> = emptyList()
)