package me.knighthat.discord.payload

import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import me.knighthat.discord.Intent
import me.knighthat.discord.Status
import org.intellij.lang.annotations.MagicConstant


/**
 * Used to trigger the initial handshake with the gateway.
 *
 * @param token Authentication token
 * @param properties Connection properties
 * @param compress Whether this connection supports compression of packets
 * @param largeThreshold Value between 50 and 250, total number of members where the gateway will stop sending offline members in the guild member list
 * @param shard Used for Guild Sharding
 * @param presence Presence structure for initial presence information
 * @param intents Gateway Intents you wish to receive
 *
 * @see <a href='https://discord.com/developers/docs/events/gateway#identifying'>Identifying</a>
 * @see Properties
 * @see <a href='https://discord.com/developers/docs/events/gateway#sharding'>Guild sharding</a>
 * @see Intent
 */
@Serializable
data class Identify(
    val token: String,
    val properties: Properties,
    @param:MagicConstant(valuesFromClass = Intent::class)
    val intents: Int,
    val compress: Boolean? = false,
    @SerialName("large_threshold")
    val largeThreshold: Int? = null,
    val shard: Pair<Int, Int>? = null,
    val presence: Presence? = null
) {

    /**
     * @param os Your operating system
     * @param browser Your library name
     * @param device Your library name
     */
    @Serializable
    data class Properties(val os: String, val browser: String, val device: String)

    class Builder internal constructor() {

        var token: String? = null
        var properties: Properties? = null
        @MagicConstant(valuesFromClass = Intent::class)
        var intents: Int? = null
        var compress: Boolean? = false
        var largeThreshold: Int? = null
        var shard: Pair<Int, Int>? = null
        var presence: Presence? = null

        internal fun build(): Identify {
            val token = requireNotNull( token ) { "Identify token is required!" }
            val properties = requireNotNull( properties ) { "Identify properties is required!" }
            val intents = requireNotNull( intents ) { "Identify intents is required!" }

            return Identify(token, properties, intents, compress, largeThreshold, shard, presence)
        }

        fun initPresence( vararg activities: Activity ) {
            this.presence = Presence(null, activities.toList(), Status.ONLINE, false)
        }
    }
}