package me.knighthat.discord

import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.websocket.DefaultClientWebSocketSession
import io.ktor.client.plugins.websocket.sendSerialized
import io.ktor.client.plugins.websocket.webSocketSession
import io.ktor.client.request.get
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.http.ContentType
import io.ktor.http.HttpHeaders
import io.ktor.http.URLBuilder
import io.ktor.websocket.CloseReason
import io.ktor.websocket.Frame
import io.ktor.websocket.close
import io.ktor.websocket.readText
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.filterIsInstance
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.receiveAsFlow
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.add
import kotlinx.serialization.json.boolean
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.decodeFromJsonElement
import kotlinx.serialization.json.encodeToJsonElement
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.long
import kotlinx.serialization.json.putJsonArray
import me.knighthat.discord.payload.Identify
import me.knighthat.discord.payload.Payload
import me.knighthat.discord.payload.Presence
import me.knighthat.discord.payload.Ready
import me.knighthat.discord.payload.Reconnect
import me.knighthat.exception.CancellationException
import me.knighthat.logging.Logger
import java.net.URI
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicInteger
import java.util.concurrent.atomic.AtomicLong
import kotlin.random.Random
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.Duration.Companion.seconds

internal class DiscordImpl: Discord {

    companion object {
        private const val LOGGING_TAG = "discord-kotlin"
        private const val GATEWAY_API_URL = "https://discord.com/api/gateway"
        private const val GATEWAY_VERSION = "10"
        private const val ENCODING = "json"
    }

    private val shouldReconnect = AtomicBoolean(true)
    private val sequence = AtomicLong(-1)
    private val payloadFlow = MutableSharedFlow<Payload>()
    private val json = Json { ignoreUnknownKeys = true }

    private val heartbeatAcknowledge = AtomicBoolean(true)
    private val heartbeatRetries = AtomicInteger(0)

    @Volatile private lateinit var client: HttpClient
    @Volatile private lateinit var session: DefaultClientWebSocketSession

    @Volatile private var isReady = false
    @Volatile private var resumeUrl: String? = null
    @Volatile private var sessionId: String? = null
    @Volatile private var token: String? = null

    private fun send( payload: Payload, requireSessionReady: Boolean = true ) {
        require( !requireSessionReady || isReady ) {
            "Session isn't ready to send payloads!"
        }

        CoroutineScope(Dispatchers.Default ).launch {
            payloadFlow.emit( payload )
        }
    }

    private suspend fun getWsUrl(): String {
        Logger.v( LOGGING_TAG, "Getting websocket url" )

        if( resumeUrl == null )
            resumeUrl = client.get( GATEWAY_API_URL )
                              .body<JsonObject>()["url"]!!
                              .jsonPrimitive
                              .content

        val builder = URLBuilder( this.resumeUrl!! )
        builder.parameters["v"] = GATEWAY_VERSION
        builder.parameters["encoding"] = ENCODING

        return builder.buildString()
                      .also { Logger.d( LOGGING_TAG, "websocket url: $it" ) }
    }

    private fun sendHeartbeat() =
        runCatching {
            if( !heartbeatAcknowledge.get() && heartbeatRetries.get() >= 3 )
                throw CancellationException("heartbeat not acknowledged!", true)

            val payload = Payload(Code.HEARTBEAT, JsonPrimitive(sequence.get()))
            send( payload, false )
        }.onSuccess {
            heartbeatAcknowledge.set( false )
            heartbeatRetries.incrementAndGet()

            Logger.d(  LOGGING_TAG, "Heartbeat sent!" )
        }.onFailure { err ->
            Logger.e( LOGGING_TAG, err,"Heartbeat failed: ${err.message}")
        }

    private fun DefaultClientWebSocketSession.startSendingJob() = launch {
        payloadFlow.collect { payload ->
            // Don't log this payload because it contains user's token
            if( payload.opCode != Code.IDENTIFY )
                Logger.v( LOGGING_TAG, "Sending payload: ${json.encodeToString(payload)}" )

            session.sendSerialized( payload )
        }
    }

    private fun DefaultClientWebSocketSession.startHeartbeat( interval: Long ) = launch {
        // Initial heartbeat has a little offset
        // https://discord.com/developers/docs/events/gateway#heartbeat-interval
        val jitter = Random(System.currentTimeMillis()).nextDouble()

        Logger.d( LOGGING_TAG, "jitter: $jitter" )

        delay( (interval * jitter).milliseconds )

        while( isActive ) {
            // TODO: Before sending another heartbeat, check if old heartbeat is acknowledged
            Logger.v( LOGGING_TAG, "Sending heartbeat to Discord..." )

            sendHeartbeat().onSuccess { delay( interval.milliseconds ) }
                           // TODO: Add a simple loop here to resend another
                           //  heartbeat if failed by network, reconnect otherwise.
                           .onFailure {
                               if( it is CancellationException )
                                   this@startHeartbeat.incoming.cancel( it )

                               break
                           }
        }
    }

    private suspend fun DefaultClientWebSocketSession.startListening( onHello: (Payload) -> Unit = {} ) =
        // Convert the incoming channel to a Flow and process each frame.
        incoming.receiveAsFlow()
                // TODO: Handle [Frame.Close] as well
                .filterIsInstance<Frame.Text>()
                .map(Frame.Text::readText )
                .onEach { Logger.v( LOGGING_TAG,"Received message: $it" ) }
                .map { json.decodeFromString<Payload>( it ) }
                .onEach { payload ->
                    if( payload.opCode == Code.HELLO ) {
                        onHello( payload )
                        return@onEach
                    }

                    handleGatewayPayload(payload)
                }
                .catch { err ->
                    if( err is CancellationException && !err.canReconnect ) {
                        resumeUrl = null
                        sessionId = null
                        token = null
                    }

                    val message = err.message ?: "Error occurs while processing payload from Discord!"
                    Logger.e( LOGGING_TAG, err, message )
                }
                .collect() // Start collecting the flow to make it run.

    private fun handleGatewayPayload( payload: Payload ) {
        when ( payload.opCode ) {
            Code.ACK -> {
                heartbeatAcknowledge.set( true )
                heartbeatRetries.set( 0 )

                Logger.v( LOGGING_TAG,"Received HEARTBEAT_ACK.")
            }

            Code.HEARTBEAT -> {
                Logger.d( LOGGING_TAG, "Received HEARTBEAT request from Discord. Sending HEARTBEAT." )
                sendHeartbeat()
            }

            Code.DISPATCH -> {
                sequence.set( payload.sequence ?: -1 )

                val eventName = requireNotNull( payload.name ) {
                    "Dispatch with `null` event name"
                }
                Logger.d( LOGGING_TAG, "Received event: $eventName with sequence: ${sequence.get()}" )

                // You would handle different events here, e.g., "READY", "MESSAGE_CREATE", etc.
                when ( eventName ) {
                    "READY" -> {
                        val ready = json.decodeFromJsonElement<Ready>( payload.data!! )
                        sessionId = ready.sessionId
                        resumeUrl = ready.resumeUrl
                        isReady = true

                        Logger.i( LOGGING_TAG, "Session is ready to be used!" )
                    }
                    // Handle other events...
                }
            }

            Code.RECONNECT -> throw CancellationException("Received RECONNECT request from Discord.", true)

            Code.INVALID_SESSION -> throw CancellationException(
                message = "Received INVALID session",
                canReconnect = payload.data?.jsonPrimitive?.boolean ?: false
            )

            else -> Logger.w( LOGGING_TAG, "Unhandled opcode: ${payload.opCode}" )
        }
    }

    override fun isReady(): Boolean = isReady

    override fun setClient( client: HttpClient ) { this.client = client }

    override suspend fun login( builder: Identify.Builder.() -> Unit ) {
        if( !shouldReconnect.get() ) {
            Logger.d( LOGGING_TAG, "shouldReconnect is `false`" )

            shouldReconnect.set( true )
            return
        } else {
            Logger.v( LOGGING_TAG, "Starting new websocket session..." )
        }

        session = client.webSocketSession( getWsUrl() )
        session.startSendingJob()

        // FIXME: Place this after [startListening] seems to make
        //  [client.webSocketSession] go into an infinite loop
        //  when connectivity is unavailable.
        isReady = false

        Logger.v( LOGGING_TAG, "Websocket session to Discord started successfully! Starting handshake protocol..." )

        session.startListening { payload ->
            val interval = requireNotNull(
                value = payload.data
                               ?.jsonObject
                               ?.get("heartbeat_interval")
                               ?.jsonPrimitive
                               ?.long
                               ?.also {
                                   Logger.d( LOGGING_TAG, "Received HELLO. Heartbeat interval: $it ms" )
                               }
            ) { "Hello payload doesn't contain interval!" }
            session.startHeartbeat( interval )

            if( token != null && sessionId != null ) {
                Logger.v( LOGGING_TAG, "Attempting to reconnect..." )

                val reconnect = Reconnect(token!!, sessionId!!, sequence.get())
                send( Payload(Code.RESUME, json.encodeToJsonElement( reconnect )) )
            } else {
                Logger.v( LOGGING_TAG, "Setting up new session" )

                val identity = Identify.Builder().apply( builder ).build()

                this.token = identity.token

                Logger.v( LOGGING_TAG, "Sending Identify payload." )

                val payload = Payload(Code.IDENTIFY, json.encodeToJsonElement( identity ))
                send( payload, false )
            }
        }
        session.cancel()

        // Add delay to prevent spamming the server
        delay( 2.seconds )

        // This will make this function loop until the stack's overflown
        // But it's more common that the connectivity is
        login( builder )
    }

    override fun send( payload: Payload ) = send( payload, true )

    override fun logout() {
        if( !::session.isInitialized ) {
            Logger.i( LOGGING_TAG, "Session isn't created!" )
            return
        }

        shouldReconnect.set( false )

        session.launch {
            session.close( CloseReason(CloseReason.Codes.NORMAL, "") )
        }

        isReady = false
        resumeUrl = null
        sessionId = null
        token = null

        sequence.set( -1 )

        Logger.i( LOGGING_TAG, "Connection to Discord gateway has closed!" )
    }

    override fun updatePresence( presence: () -> Presence ) {
        val payload = Payload(Code.PRESENCE_UPDATE, Json.encodeToJsonElement( presence() ))
        send( payload )
    }

    override suspend fun getExternalImageUrl( imageUrl: String, applicationId: String ) = runCatching {
        Logger.v( LOGGING_TAG, "Posting $imageUrl to get external url" )

        if ( imageUrl.startsWith( "mp:" ) ) {
            Logger.d( LOGGING_TAG, "imageUrl already an external url" )
            return@runCatching imageUrl
        }

        val scheme = URI(imageUrl).scheme
        require(
            scheme.equals( "http", true )
                    || scheme.equals( "https", true )
        ) { "Only \"http\" and \"https\" are supported!" }

        val postUrl = "https://discord.com/api/v$GATEWAY_VERSION/applications/$applicationId/external-assets"
        val response = client.post( postUrl ) {
            header( HttpHeaders.Authorization, token )
            // For some reasons, this is required.
            // "java.lang.ClassCastException: kotlinx.serialization.json.JsonObject cannot be cast to io.ktor.http.content.OutgoingContent"
            // will be thrown otherwise
            header( HttpHeaders.ContentType, ContentType.Application.Json )

            setBody(
                // Use this to ensure syntax
                // {"urls":[imageUrl]}
                buildJsonObject {
                    putJsonArray( "urls" ) { add( imageUrl ) }
                }
            )
        }.body<JsonArray>()

        response.firstNotNullOf { it.jsonObject["external_asset_path"] }
                .jsonPrimitive
                .content
                .let { "mp:$it" }
    }.onSuccess {
        Logger.d( LOGGING_TAG, "External url: $it" )
    }.onFailure {
        Logger.e( LOGGING_TAG, it, "Error occurs while posting imageUrl for external url" )
    }
}