package me.knighthat.discord


/**
 * In order to prevent broken reconnect loops,
 * you should consider some close codes as a signal to stop reconnecting.
 *
 * This can be because your token expired, or your identification is invalid.
 *
 * This table explains what the application defined close codes for the gateway are,
 * and which close codes you should not attempt to reconnect.
 */
object CloseCode {

    /**
     * We're not sure what went wrong. Try reconnecting?
     */
    @CanReconnect
    const val UNKNOWN_ERROR = 4000

    /**
     * You sent an invalid Gateway opcode or an invalid payload for an opcode.
     * Don't do that!
     */
    @CanReconnect
    const val UNKNOWN_OPCODE = 4001

    /**
     * You sent an invalid payload to Discord. Don't do that!
     */
    @CanReconnect
    const val DECODE_ERROR = 4002

    /**
     * You sent us a payload prior to identifying,
     * or this session has been invalidated.
     */
    @CanReconnect
    const val NOT_AUTHENTICATED = 4003

    /**
     * The account token sent with your identify payload is incorrect.
     */
    const val AUTHENTICATION_FAILED = 4004

    /**
     * You sent more than one identify payload. Don't do that!
     */
    @CanReconnect
    const val ALREADY_AUTHENTICATED = 4005

    /**
     * The sequence sent when resuming the session was invalid.
     * Reconnect and start a new session.
     */
    @CanReconnect
    const val INVALID_SEQ = 4007

    /**
     * Woah nelly! You're sending payloads to us too quickly.
     * Slow it down! You will be disconnected on receiving this.
     */
    @CanReconnect
    const val RATE_LIMITED = 4008

    /**
     * Your session timed out. Reconnect and start a new one.
     */
    @CanReconnect
    const val SESSION_TIMED_OUT = 4009

    /**
     * You sent us an invalid shard when identifying.
     */
    const val INVALID_SHARD = 4010

    /**
     * The session would have handled too many guilds -
     * you are required to shard your connection in order to connect.
     */
    const val SHARDING_REQUIRED = 4011

    /**
     * You sent an invalid version for the gateway.
     */
    const val INVALID_API_VERSION = 4012

    /**
     * You sent an invalid intent for a Gateway Intent.
     * You may have incorrectly calculated the bitwise value.
     */
    const val INVALID_INTENT = 4013

    /**
     * You sent a disallowed intent for a Gateway Intent.
     * You may have tried to specify an intent that you have not enabled or are not approved for.
     */
    const val DISALLOWED_INTENT = 4014
}