package me.knighthat.discord.payload

import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonObject
import me.knighthat.discord.Flag
import me.knighthat.discord.StatusDisplayType
import me.knighthat.discord.Type
import org.intellij.lang.annotations.MagicConstant


/**
 * @param name Activity's name
 * @param type Activity type
 * @param url Stream URL, is validated when type is 1
 * @param createdAt Unix timestamp (in milliseconds) of when the activity was added to the user's session
 * @param timestamps Unix timestamps for start and/or end of the game
 * @param applicationId Application ID for the game
 * @param statusDisplayType Status display type; controls which field is displayed in the user's status text in the member list
 * @param details What the player is currently doing
 * @param detailsUrl URL that is linked when clicking on the details text
 * @param state User's current party status, or text used for a custom status
 * @param stateUrl URL that is linked when clicking on the state text
 * @param emoji Emoji used for a custom status
 * @param party Information for the current party of the player
 * @param assets Images for the presence and their hover texts
 * @param secrets Secrets for Rich Presence joining and spectating
 * @param instance Whether or not the activity is an instanced game session
 * @param flags Activity flags ORd together, describes what the payload includes
 * @param buttons Custom buttons' labels shown in the Rich Presence (max 2)
 * @param metadata Custom buttons' urls of [buttons] to redirect when clicked
 *
 * @see Type
 * @see Timestamp
 * @see Assets
 * @see Flag
 * @see metadata
 */
@Serializable
data class Activity(
    val name: String,
    @param:MagicConstant(valuesFromClass = Type::class)
    val type: Int,
    val url: String? = null,
    @SerialName("created_at")
    val createdAt: Long,
    val timestamps: Timestamp? = null,
    @SerialName("application_id")
    val applicationId: String? = null,
    @SerialName("status_display_type")
    @param:MagicConstant(valuesFromClass = StatusDisplayType::class)
    val statusDisplayType: Int? = null,
    val details: String? = null,
    @SerialName("details_url")
    val detailsUrl: String? = null,
    val state: String? = null,
    @SerialName("state_url")
    val stateUrl: String? = null,
    val emoji: JsonObject? = null,
    val party: JsonObject? = null,
    val assets: Assets? = null,
    val secrets: JsonObject? = null,
    val instance: Boolean? = null,
    @param:MagicConstant(valuesFromClass = Flag::class)
    val flags: Int? = null,
    val buttons: List<String>? = null,
    val metadata: Metadata? = null
) {

    /**
     * @param name Activity's name
     * @param type Activity type
     * @param url Stream URL, is validated when type is 1
     * @param createdAt Unix timestamp (in milliseconds) of when the activity was added to the user's session
     * @param timestamps Unix timestamps for start and/or end of the game
     * @param applicationId Application ID for the game
     * @param statusDisplayType Status display type; controls which field is displayed in the user's status text in the member list
     * @param details What the player is currently doing
     * @param detailsUrl URL that is linked when clicking on the details text
     * @param state User's current party status, or text used for a custom status
     * @param stateUrl URL that is linked when clicking on the state text
     * @param emoji Emoji used for a custom status
     * @param party Information for the current party of the player
     * @param assets Images for the presence and their hover texts
     * @param secrets Secrets for Rich Presence joining and spectating
     * @param instance Whether or not the activity is an instanced game session
     * @param flags Activity flags ORd together, describes what the payload includes
     * @param buttons Custom buttons shown in the Rich Presence (max 2)
     *
     * @see Type
     * @see Timestamp
     * @see Assets
     * @see Flag
     * @see Button
     */
    constructor(
        name: String,
        @MagicConstant(valuesFromClass = Type::class)
        type: Int,
        url: String? = null,
        createdAt: Long,
        timestamps: Timestamp? = null,
        applicationId: String? = null,
        @MagicConstant(valuesFromClass = StatusDisplayType::class)
        statusDisplayType: Int? = null,
        details: String? = null,
        detailsUrl: String? = null,
        state: String? = null,
        stateUrl: String? = null,
        emoji: JsonObject? = null,
        party: JsonObject? = null,
        assets: Assets? = null,
        secrets: JsonObject? = null,
        instance: Boolean? = null,
        @MagicConstant(valuesFromClass = Flag::class)
        flags: Int? = null,
        buttons: List<Button>? = null
    ): this(
        name = name,
        type = type,
        url = url,
        createdAt = createdAt,
        timestamps = timestamps,
        applicationId = applicationId,
        statusDisplayType = statusDisplayType,
        details = details,
        detailsUrl = detailsUrl,
        state = state,
        stateUrl = stateUrl,
        emoji = emoji,
        party = party,
        assets = assets,
        secrets = secrets,
        instance = instance,
        flags = flags,
        buttons = buttons?.map( Button::label ),
        metadata = Metadata(buttons?.map(Button::url ))
    )

    /**
     * For [Type.LISTENING] and [Type.WATCHING] activities,
     * you can include both start and end timestamps to display a time bar.
     *
     * @param start Unix time (in milliseconds) of when the activity started
     * @param end Unix time (in milliseconds) of when the activity ends
     */
    @Serializable
    data class Timestamp(val start: Long, val end: Long?)

    /**
     * @param largeImage See Activity Asset Image
     * @param largeText Text displayed when hovering over the large image of the activity
     * @param largeUrl URL that is opened when clicking on the large image
     * @param smallImage See Activity Asset Image
     * @param smallText Text displayed when hovering over the small image of the activity
     * @param smallUrl URL that is opened when clicking on the small image
     *
     * @see <a href='https://discord.com/developers/docs/events/gateway-events#activity-object-activity-asset-image'>Activity asset image</a>
     */
    @Serializable
    data class Assets(
        @SerialName("large_image") val largeImage: String?,
        @SerialName("large_text") val largeText: String?,
        @SerialName("large_url") val largeUrl: String?,
        @SerialName("small_image") val smallImage: String?,
        @SerialName("small_text") val smallText: String?,
        @SerialName("small_url") val smallUrl: String?,
    )

    /**
     * @param label Text shown on the button (1-32 characters)
     * @param url URL opened when clicking the button (1-512 characters)
     */
    @Serializable
    data class Button(val label: String, val url: String)

    /**
     * @param buttonUrls URL to redirect user to when clicked.
     * Must be of the same position of the button in [Activity.buttons]
     */
    @Serializable
    data class Metadata(
        @SerialName("button_urls") val buttonUrls: List<String>?
    )
}
