package me.danb.retherm;

import androidx.appcompat.app.AppCompatActivity;
import androidx.core.graphics.Insets;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.preference.PreferenceManager;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.net.Uri;
import android.os.Bundle;
import android.view.*;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.*;

import com.google.android.material.appbar.MaterialToolbar;
import com.google.android.material.color.DynamicColors;

import java.util.HashMap;

public class CalculatorActivity extends AppCompatActivity {
    boolean showThermocouples;
    boolean americanStandard;
    boolean useFahrenheit;
    InputMethodManager imm;

    EditText calcInput;
    RadioGroup calcRadioThermocouples;
    RadioGroup calcRadioRtd;

    HorizontalScrollView rtdHSV;
    HorizontalScrollView thermocoupleHSV;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        DynamicColors.applyToActivityIfAvailable(this);
        setContentView(R.layout.calculator_layout);

        // View setup
        MaterialToolbar toolbar = findViewById(R.id.topAppBar);
        calcInput = findViewById(R.id.calculatorETInput);

        calcRadioThermocouples = findViewById(R.id.calculatorRGthermocouples);
        calcRadioRtd = findViewById(R.id.calculatorRGrtd);
        rtdHSV = findViewById(R.id.calculatorHSVrtd);
        thermocoupleHSV = findViewById(R.id.calculatorHSVthermocouples);

        final Button calculateButton = findViewById(R.id.calculatorButtonCalculate);
        final TextView calcAnswer = findViewById(R.id.calculatorTVanswer);

        imm = (InputMethodManager) getSystemService(
                Context.INPUT_METHOD_SERVICE);

        calcInput.setOnEditorActionListener((v, actionId, event) -> {
            if (actionId == EditorInfo.IME_ACTION_DONE) {
                calculateButton.callOnClick();
            }
            return false;
        });

        // Calculate button click action
        calculateButton.setOnClickListener(v -> {
            imm.hideSoftInputFromWindow(calcInput.getWindowToken(), 0);

            double input;
            if (calcInput.getText().toString().isEmpty() || calcInput.getText().toString().equals(".") || calcInput.getText().toString().equals("-.")) {
                input = 0;
            } else {
                input = Double.parseDouble(calcInput.getText().toString());
            }

            String answer = "Answer";
            if (calcRadioRtd.getCheckedRadioButtonId() != -1) {
                answer = getRtdResult(input);
            } else if (calcRadioThermocouples.getCheckedRadioButtonId() != -1) {
                answer = getThermocoupleResult(input);
            }

            calcAnswer.setText(answer);
            calcAnswer.setAlpha(0);
            calcAnswer.animate().alpha(1);
        });

        toolbar.setOnMenuItemClickListener(this::onMenuItemClick);


        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.main), (v, insets) -> {
            Insets bars = insets.getInsets(
                    WindowInsetsCompat.Type.systemBars()
                            | WindowInsetsCompat.Type.displayCutout()
            );
            v.setPadding(bars.left, bars.top, bars.right, bars.bottom);
            return WindowInsetsCompat.CONSUMED;
        });
    }

    private String getRtdResult(double input) {
        RtdCalculator rtdCalculator = new RtdCalculator();

        int selectedPtValue;
        switch (calcRadioRtd.getCheckedRadioButtonId()) {
            case (R.id.calculatorRBpt100):
                selectedPtValue = 100;
                break;
            case (R.id.calculatorRBpt500):
                selectedPtValue = 500;
                break;
            case (R.id.calculatorRBpt1000):
                selectedPtValue = 1000;
                break;
            default:
                selectedPtValue = 0;
        }

        double answer;
        try {
            if (americanStandard) {
                double aRtdAlpha = 0.003916;
                answer = rtdCalculator.RtdLinearFitToCelsius(input, (double) selectedPtValue, aRtdAlpha);
            } else {
                double factor = (double) selectedPtValue / 100;
                answer = rtdCalculator.EuropeOhmsToCelsius(input, factor);
            }
        } catch (Exception exception) {
            return exception.getMessage();
        }

        return formatAnswer(answer);
    }

    private String getThermocoupleResult(double input) {
        ThermocoupleCalculator tcCalc = new ThermocoupleCalculator();

        HashMap<Integer, String> tcTypeByButtonId = new HashMap<>();
        tcTypeByButtonId.put(R.id.calculatorRBb, "b");
        tcTypeByButtonId.put(R.id.calculatorRBe, "e");
        tcTypeByButtonId.put(R.id.calculatorRBj, "j");
        tcTypeByButtonId.put(R.id.calculatorRBk, "k");
        tcTypeByButtonId.put(R.id.calculatorRBn, "n");
        tcTypeByButtonId.put(R.id.calculatorRBr, "r");
        tcTypeByButtonId.put(R.id.calculatorRBs, "s");
        tcTypeByButtonId.put(R.id.calculatorRBt, "t");

        String activeTcType = tcTypeByButtonId.get(calcRadioThermocouples.getCheckedRadioButtonId());
        if (activeTcType == null) {
            activeTcType = "x";
        }

        double answer;
        try {
            answer = tcCalc.MvToCelsius(activeTcType , input);
        } catch (Exception exception) {
            return exception.getMessage();
        }

        return formatAnswer(answer);
    }

    private String formatAnswer(double answerCelsius) {
        if (useFahrenheit) {
            return String.format("%.2f °F", celsiusToFahrenheit(answerCelsius));
        }

        return String.format("%.2f °C", answerCelsius);
    }

    @Override
    public void onResume() {
        super.onResume();
        SharedPreferences extras = PreferenceManager.getDefaultSharedPreferences(this);
        showThermocouples = extras.getBoolean("show_thermocouples", false);
        americanStandard = extras.getBoolean("american_standard", false);
        useFahrenheit = extras.getBoolean("use_fahrenheit", false);
        if (!showThermocouples && thermocoupleHSV != null) {
            thermocoupleHSV.setVisibility(View.GONE);
        }
        if (showThermocouples) {
            thermocoupleHSV = findViewById(R.id.calculatorHSVthermocouples);
            thermocoupleHSV.setVisibility(View.VISIBLE);
        }

    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.rtdcalulator, menu);
        return true;
    }

    private boolean onMenuItemClick(MenuItem item) {
        if (item.getItemId() == R.id.action_settings) {
            Intent intent = new Intent(CalculatorActivity.this, ExtrasActivity.class);
            startActivity(intent);
            return true;
        } else if (item.getItemId() == R.id.action_thermocouples) {
            Intent intent = getPackageManager().getLaunchIntentForPackage("com.southerntemp.thermocouples");
            if (intent != null) {
                // start the activity
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                startActivity(intent);
            } else {
                // bring user to the market
                // or let them choose an app?
                intent = new Intent(Intent.ACTION_VIEW);
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                intent.setData(Uri.parse("market://details?id=" + "com.southerntemp.thermocouples"));
                startActivity(intent);
            }
            return true;
        } else {
            return true;
        }
    }

    public void RadioButtonRtdClick(View v) {
        imm.hideSoftInputFromWindow(calcInput.getWindowToken(), 0);
        calcRadioThermocouples.clearCheck();
    }

    public void RadioButtonThermocoupleClick(View v) {
        imm.hideSoftInputFromWindow(calcInput.getWindowToken(), 0);
        calcRadioRtd.clearCheck();
    }

    private double celsiusToFahrenheit(double celsius) {
        return celsius * (9.0 / 5.0) + 32.0;
    }

}
