import 'dart:async';
import 'dart:convert';
import 'package:build/build.dart';

Builder l10nKeysBuilder(BuilderOptions options) => L10nKeysBuilder();

class L10nKeysBuilder implements Builder {
  @override
  final buildExtensions = const {
    'lib/l10n/app_en.arb': ['lib/l10n/l10n_keys.dart'],
  };

  @override
  FutureOr<void> build(BuildStep buildStep) async {
    final inputId = buildStep.inputId;
    final content = await buildStep.readAsString(inputId);
    final Map<String, dynamic> arb = json.decode(content);

    final simpleKeys = <String>[];
    final parameterizedKeys = <String, String>{};

    for (final key in arb.keys) {
      if (key.startsWith('@') || key.startsWith('_')) continue;

      final metaKey = '@$key';
      final meta = arb[metaKey];
      final value = arb[key];

      final hasMetaPlaceholders = meta != null && meta is Map && meta.containsKey('placeholders');
      final hasValuePlaceholders = value is String && RegExp(r'\{[a-zA-Z_]+\}').hasMatch(value);

      if (hasMetaPlaceholders || hasValuePlaceholders) {
        parameterizedKeys[key] = hasMetaPlaceholders ? _getParameterSignature(meta['placeholders']) : 'dynamic';
      } else {
        simpleKeys.add(key);
      }
    }

    simpleKeys.sort();

    final buffer = StringBuffer();

    buffer.writeln('// GENERATED CODE - DO NOT MODIFY BY HAND');
    buffer.writeln('// Generated by l10n_keys_builder from app_en.arb');
    buffer.writeln('// ignore_for_file: constant_identifier_names');
    buffer.writeln();
    buffer.writeln("import 'app_localizations.dart';");
    buffer.writeln();

    buffer.writeln('abstract class L10nKeys {');
    for (final key in simpleKeys) {
      buffer.writeln("  static const $key = '$key';");
    }
    buffer.writeln('}');
    buffer.writeln();

    buffer.writeln('extension L10nResolver on AppLocalizations {');
    buffer.writeln('  String resolve(String? key) {');
    buffer.writeln("    if (key == null) return '';");
    buffer.writeln('    return switch (key) {');
    for (final key in simpleKeys) {
      buffer.writeln("      '$key' => $key,");
    }
    buffer.writeln('      _ => key,');
    buffer.writeln('    };');
    buffer.writeln('  }');
    buffer.writeln('}');

    final outputId = AssetId(inputId.package, 'lib/l10n/l10n_keys.dart');
    await buildStep.writeAsString(outputId, buffer.toString());
  }

  String _getParameterSignature(Map<String, dynamic> placeholders) {
    return placeholders.keys.join(', ');
  }
}
