import 'package:lean_builder/runner.dart';
import 'package:lean_builder/src/asset/asset.dart';

/// {@template build_result}
/// Represents the result of a build operation for a chunk of assets.
///
/// This class tracks:
/// - The outputs generated by the build
/// - Any assets that failed to build
///
/// It's used primarily for aggregating results from parallel build operations
/// before finalizing a build phase.
/// {@endtemplate}
class BuildResult {
  /// Map of source assets to their generated output URIs
  final Map<Asset, Set<Uri>> outputs;

  /// List of assets that failed to build, along with their errors
  final List<FailedAsset> failedAssets;

  /// {@macro build_result}
  BuildResult(this.outputs, this.failedAssets);

  @override
  String toString() {
    final int outputCount = outputs.length;
    final int failedCount = failedAssets.length;
    return 'BuildResult(outputs: $outputCount, fieldAssets: $failedCount)';
  }
}

/// {@template failed_asset}
/// Represents an asset that failed to build.
///
/// This class captures:
/// - The asset that failed
/// - The error that occurred
/// - The stack trace at the time of failure
///
/// It's used to provide detailed error reporting and to track which
/// assets need to be reprocessed in subsequent builds.
/// {@endtemplate}
class FailedAsset {
  /// The asset that failed to build
  final Asset asset;

  /// The error that occurred during building
  final Object error;

  /// The stack trace at the time of failure
  final StackTrace? stackTrace;

  /// {@macro failed_asset}
  FailedAsset(this.asset, this.error, this.stackTrace);

  @override
  bool operator ==(Object other) =>
      identical(this, other) ||
      other is FailedAsset && runtimeType == other.runtimeType && asset == other.asset && error == other.error;

  @override
  int get hashCode => asset.hashCode ^ error.hashCode;
}

/// {@template phase_result}
/// Represents the result of a build phase.
///
/// This class tracks:
/// - The outputs generated by the phase
/// - Outputs that were deleted during cleanup
/// - Assets that failed to build
///
/// It provides methods to check for changes and errors, which is used
/// to determine whether subsequent phases need to be run.
/// {@endtemplate}
class PhaseResult {
  /// The set of URIs for outputs generated by this phase
  final Set<Uri> outputs;

  /// The set of URIs for outputs that were deleted during cleanup
  final Set<Uri> deletedOutputs;

  /// List of assets that failed to build, along with their errors
  final List<FailedAsset> failedAssets;

  /// Whether the phase encountered any errors
  bool get hasErrors => failedAssets.isNotEmpty;

  /// {@macro phase_result}
  PhaseResult({
    required this.outputs,
    required this.failedAssets,
    required this.deletedOutputs,
  });

  /// Whether the phase produced any changes
  ///
  /// This includes both new outputs and deleted outputs
  bool get containsAnyChanges => outputs.isNotEmpty || deletedOutputs.isNotEmpty;

  /// {@template phase_result.contains_changes_from_builder}
  /// Checks whether this phase contains changes from a specific builder.
  ///
  /// This method is used to determine which builders in subsequent phases
  /// need to be run based on changes produced by this phase.
  ///
  /// [entry] The builder entry to check for changes
  ///
  /// Returns true if the phase contains changes from the specified builder
  /// {@endtemplate}
  bool containsChangesFromBuilder(BuilderEntry entry) {
    bool didAnyChange = false;
    for (final String ext in entry.outputExtensions) {
      if (outputs.any((Uri e) => e.path.endsWith(ext))) {
        didAnyChange = true;
        break;
      }
      if (deletedOutputs.any((Uri e) => e.path.endsWith(ext))) {
        didAnyChange = true;
        break;
      }
    }
    return didAnyChange;
  }
}

/// {@template multi_failed_assets_exception}
/// An exception that represents multiple failed assets.
///
/// This exception is thrown when one or more assets fail to build
/// during a build phase. It encapsulates all the failed assets
/// to provide comprehensive error reporting.
/// {@endtemplate}
class MultiFailedAssetsException implements Exception {
  /// The list of assets that failed to build
  final List<FailedAsset> assets;

  /// {@macro multi_failed_assets_exception}
  MultiFailedAssetsException(this.assets);
}
