// Copyright 2013 The Flutter Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// This file specifically tests the test code generated by dartTestOut.

import 'package:flutter/services.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:shared_test_plugin_code/src/generated/core_tests.gen.dart'
    show AllNullableTypes;
import 'package:shared_test_plugin_code/src/generated/message.gen.dart';
import 'package:shared_test_plugin_code/test_types.dart';

import 'test_message.gen.dart';

class Mock implements TestHostApi {
  List<String> log = <String>[];

  @override
  void initialize() {
    log.add('initialize');
  }

  @override
  MessageSearchReply search(MessageSearchRequest arg) {
    log.add('search');
    return MessageSearchReply()..result = arg.query;
  }
}

class MockNested implements TestNestedApi {
  bool didCall = false;
  @override
  MessageSearchReply search(MessageNested arg) {
    didCall = true;
    if (arg.request == null) {
      return MessageSearchReply();
    } else {
      return MessageSearchReply()..result = arg.request?.query;
    }
  }
}

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  group('equality method', () {
    final correctList = <Object?>['a', 2, 'three'];
    final List<Object?> matchingList = correctList.toList();
    final differentList = <Object?>['a', 2, 'three', 4.0];
    final correctMap = <String, Object?>{'a': 1, 'b': 2, 'c': 'three'};
    final matchingMap = <String, Object?>{...correctMap};
    final differentKeyMap = <String, Object?>{'a': 1, 'b': 2, 'd': 'three'};
    final differentValueMap = <String, Object?>{'a': 1, 'b': 2, 'c': 'five'};
    final correctListInMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'c': correctList,
    };
    final matchingListInMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'c': matchingList,
    };
    final differentListInMap = <String, Object?>{
      'a': 1,
      'b': 2,
      'c': differentList,
    };
    final correctMapInList = <Object?>['a', 2, correctMap];
    final matchingMapInList = <Object?>['a', 2, matchingMap];
    final differentKeyMapInList = <Object?>['a', 2, differentKeyMap];
    final differentValueMapInList = <Object?>['a', 2, differentValueMap];

    test('equality method correctly checks deep equality', () {
      final AllNullableTypes generic = genericAllNullableTypes;
      final AllNullableTypes identical = AllNullableTypes.decode(
        generic.encode(),
      );
      expect(identical, generic);
    });

    test('equality method correctly identifies non-matching classes', () {
      final AllNullableTypes generic = genericAllNullableTypes;
      final allNull = AllNullableTypes();
      expect(allNull == generic, false);
    });

    test(
      'equality method correctly identifies non-matching lists in classes',
      () {
        final withList = AllNullableTypes(list: correctList);
        final withDifferentList = AllNullableTypes(list: differentList);
        expect(withList == withDifferentList, false);
      },
    );

    test(
      'equality method correctly identifies matching -but unique- lists in classes',
      () {
        final withList = AllNullableTypes(list: correctList);
        final withDifferentList = AllNullableTypes(list: matchingList);
        expect(withList, withDifferentList);
      },
    );

    test(
      'equality method correctly identifies non-matching keys in maps in classes',
      () {
        final withMap = AllNullableTypes(map: correctMap);
        final withDifferentMap = AllNullableTypes(map: differentKeyMap);
        expect(withMap == withDifferentMap, false);
      },
    );

    test(
      'equality method correctly identifies non-matching values in maps in classes',
      () {
        final withMap = AllNullableTypes(map: correctMap);
        final withDifferentMap = AllNullableTypes(map: differentValueMap);
        expect(withMap == withDifferentMap, false);
      },
    );

    test(
      'equality method correctly identifies matching -but unique- maps in classes',
      () {
        final withMap = AllNullableTypes(map: correctMap);
        final withDifferentMap = AllNullableTypes(map: matchingMap);
        expect(withMap, withDifferentMap);
      },
    );

    test(
      'equality method correctly identifies non-matching lists nested in maps in classes',
      () {
        final withListInMap = AllNullableTypes(map: correctListInMap);
        final withDifferentListInMap = AllNullableTypes(
          map: differentListInMap,
        );
        expect(withListInMap == withDifferentListInMap, false);
      },
    );

    test(
      'equality method correctly identifies matching -but unique- lists nested in maps in classes',
      () {
        final withListInMap = AllNullableTypes(map: correctListInMap);
        final withDifferentListInMap = AllNullableTypes(map: matchingListInMap);
        expect(withListInMap, withDifferentListInMap);
      },
    );

    test(
      'equality method correctly identifies non-matching keys in maps nested in lists in classes',
      () {
        final withMapInList = AllNullableTypes(list: correctMapInList);
        final withDifferentMapInList = AllNullableTypes(
          list: differentKeyMapInList,
        );
        expect(withMapInList == withDifferentMapInList, false);
      },
    );

    test(
      'equality method correctly identifies non-matching values in maps nested in lists in classes',
      () {
        final withMapInList = AllNullableTypes(list: correctMapInList);
        final withDifferentMapInList = AllNullableTypes(
          list: differentValueMapInList,
        );
        expect(withMapInList == withDifferentMapInList, false);
      },
    );

    test(
      'equality method correctly identifies matching -but unique- maps nested in lists in classes',
      () {
        final withMapInList = AllNullableTypes(list: correctMapInList);
        final withDifferentMapInList = AllNullableTypes(
          list: matchingMapInList,
        );
        expect(withMapInList, withDifferentMapInList);
      },
    );
  });
  test('simple', () async {
    final api = MessageNestedApi();
    final mock = MockNested();
    TestNestedApi.setUp(mock);
    final MessageSearchReply reply = await api.search(
      MessageNested()..request = null,
    );
    expect(mock.didCall, true);
    expect(reply.result, null);
  });

  test('nested', () async {
    final api = MessageApi();
    final mock = Mock();
    TestHostApi.setUp(mock);
    final MessageSearchReply reply = await api.search(
      MessageSearchRequest()..query = 'foo',
    );
    expect(mock.log, <String>['search']);
    expect(reply.result, 'foo');
  });

  test('no-arg calls', () async {
    final api = MessageApi();
    final mock = Mock();
    TestHostApi.setUp(mock);
    await api.initialize();
    expect(mock.log, <String>['initialize']);
  });

  test('calling methods with null', () async {
    final mock = Mock();
    TestHostApi.setUp(mock);
    expect(
      await const BasicMessageChannel<Object?>(
        'dev.flutter.pigeon.pigeon_integration_tests.MessageApi.initialize',
        StandardMessageCodec(),
      ).send(<Object?>[null]),
      isEmpty,
    );
    try {
      await const BasicMessageChannel<Object?>(
            'dev.flutter.pigeon.pigeon_integration_tests.MessageApi.search',
            StandardMessageCodec(),
          ).send(<Object?>[null])
          as List<Object?>?;
      expect(true, isFalse); // should not reach here
    } catch (error) {
      expect(error, isAssertionError);
      expect(
        error.toString(),
        contains(
          'Argument for dev.flutter.pigeon.pigeon_integration_tests.MessageApi.search was null, expected non-null MessageSearchRequest.',
        ),
      );
    }
    expect(mock.log, <String>['initialize']);
  });
}
