// Copyright (c) 2016, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'dart:async';

import 'package:analyzer/analysis_rule/rule_context.dart';
import 'package:analyzer/analysis_rule/rule_visitor_registry.dart';
import 'package:analyzer/dart/analysis/results.dart';
import 'package:analyzer/dart/ast/visitor.dart';
import 'package:analyzer/dart/element/element.dart';
import 'package:analyzer/diagnostic/diagnostic.dart';
import 'package:analyzer/error/error.dart';
import 'package:analyzer/file_system/file_system.dart';
import 'package:analyzer/src/dart/analysis/analysis_context_collection.dart';
import 'package:analyzer/src/dart/analysis/driver.dart';
import 'package:analyzer/src/dart/analysis/driver_event.dart' as driver_events;
import 'package:analyzer/src/dart/analysis/file_state.dart';
import 'package:analyzer/src/dart/analysis/status.dart';
import 'package:analyzer/src/dart/ast/ast.dart';
import 'package:analyzer/src/dart/element/element.dart';
import 'package:analyzer/src/error/codes.dart';
import 'package:analyzer/src/lint/linter.dart';
import 'package:analyzer/src/test_utilities/lint_registration_mixin.dart';
import 'package:analyzer/src/utilities/extensions/async.dart';
import 'package:analyzer/utilities/package_config_file_builder.dart';
import 'package:analyzer_testing/utilities/utilities.dart';
import 'package:analyzer_utilities/testing/tree_string_sink.dart';
import 'package:linter/src/rules.dart';
import 'package:test/test.dart';
import 'package:test_reflective_loader/test_reflective_loader.dart';

import '../../../util/diff.dart';
import '../../../util/element_printer.dart';
import '../resolution/context_collection_resolution.dart';
import '../resolution/node_text_expectations.dart';
import '../resolution/resolution.dart';
import 'result_printer.dart';

main() {
  defineReflectiveSuite(() {
    defineReflectiveTests(AnalysisDriver_PubPackageTest);
    defineReflectiveTests(AnalysisDriver_BlazeWorkspaceTest);
    defineReflectiveTests(AnalysisDriver_LintTest);
    defineReflectiveTests(FineAnalysisDriverTest);
    defineReflectiveTests(UpdateNodeTextExpectations);
  });
}

@reflectiveTest
class AnalysisDriver_BlazeWorkspaceTest extends BlazeWorkspaceResolutionTest {
  void test_nestedLib_notCanonicalUri() async {
    var outerLibPath = '$workspaceRootPath/my/outer/lib';

    var innerFile = newFile('$outerLibPath/inner/lib/b.dart', 'class B {}');
    var innerUri = Uri.parse('package:my.outer.lib.inner/b.dart');

    var analysisSession = contextFor(innerFile).currentSession;

    void assertInnerUri(ResolvedUnitResult result) {
      var innerSource = result.libraryFragment.libraryImports
          .map((import) => import.importedLibrary?.firstFragment.source)
          .nonNulls
          .where((importedSource) => importedSource.fullName == innerFile.path)
          .single;
      expect(innerSource.uri, innerUri);
    }

    // Reference "inner" using a non-canonical URI.
    {
      var a = newFile(convertPath('$outerLibPath/a.dart'), r'''
import 'inner/lib/b.dart';
''');
      var result = await analysisSession.getResolvedUnit(a.path);
      result as ResolvedUnitResult;
      assertInnerUri(result);
    }

    // Reference "inner" using the canonical URI, via relative.
    {
      var c = newFile('$outerLibPath/inner/lib/c.dart', r'''
import 'b.dart';
''');
      var result = await analysisSession.getResolvedUnit(c.path);
      result as ResolvedUnitResult;
      assertInnerUri(result);
    }

    // Reference "inner" using the canonical URI, via absolute.
    {
      var d = newFile('$outerLibPath/inner/lib/d.dart', '''
import '$innerUri';
''');
      var result = await analysisSession.getResolvedUnit(d.path);
      result as ResolvedUnitResult;
      assertInnerUri(result);
    }
  }
}

@reflectiveTest
class AnalysisDriver_LintTest extends PubPackageResolutionTest
    with LintRegistrationMixin {
  @override
  void setUp() {
    super.setUp();

    useEmptyByteStore();
    registerLintRule(_AlwaysReportedLint.instance);
    writeTestPackageAnalysisOptionsFile(
      analysisOptionsContent(rules: [_AlwaysReportedLint.code.name]),
    );
  }

  @override
  Future<void> tearDown() {
    unregisterLintRules();
    return super.tearDown();
  }

  test_getResolvedUnit_lint_existingFile() async {
    addTestFile('');
    await resolveTestFile();

    // Existing/empty file triggers the lint.
    _assertHasLintReported(result.diagnostics, _AlwaysReportedLint.code.name);
  }

  test_getResolvedUnit_lint_notExistingFile() async {
    await resolveTestFile();

    // No errors for a file that doesn't exist.
    assertErrorsInResult([]);
  }

  void _assertHasLintReported(List<Diagnostic> diagnostics, String name) {
    var matching = diagnostics.where((element) {
      var diagnosticCode = element.diagnosticCode;
      return diagnosticCode is LintCode && diagnosticCode.name == name;
    }).toList();
    expect(matching, hasLength(1));
  }
}

@reflectiveTest
class AnalysisDriver_PubPackageTest extends PubPackageResolutionTest
    with _EventsMixin {
  @override
  bool get retainDataForTesting => true;

  @override
  void setUp() {
    super.setUp();
    registerLintRules();
    useEmptyByteStore();
  }

  test_addedFiles() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var b = newFile('$testPackageLibPath/b.dart', '');

    var driver = driverFor(testFile);

    driver.addFile2(a);
    driver.addFile2(b);
    await driver.applyPendingFileChanges();
    expect(driver.addedFiles2, unorderedEquals([a, b]));

    driver.removeFile2(a);
    await driver.applyPendingFileChanges();
    expect(driver.addedFiles2, unorderedEquals([b]));
  }

  test_addFile() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var b = newFile('$testPackageLibPath/b.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(b);
    driver.addFile2(a);

    // The files are analyzed in the order of adding.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_addFile_afterRemove() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
import 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);
    driver.addFile2(a);
    driver.addFile2(b);

    // Initial analysis, `b` does not use `a`, so there is a hint.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[status] idle
''');

    // Update `b` to use `a`, no more hints.
    modifyFile2(b, r'''
import 'a.dart';
void f() {
  A;
}
''');

    // Remove and add `b`.
    driver.removeFile2(b);
    driver.addFile2(b);

    // `b` was analyzed, no more hints.
    await assertEventsText(collector, r'''
[status] working
[operation] reuseLinkedBundle SDK
[operation] reuseLinkedBundle
  package:test/a.dart
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_addFile_missingDartLibrary_async() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/async/async.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    driver.addFile2(a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
''');
  }

  test_addFile_missingDartLibrary_core() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/core/core.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    driver.addFile2(a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
''');
  }

  test_addFile_notAbsolutePath() async {
    var driver = driverFor(testFile);
    expect(() {
      driver.addFile('not_absolute.dart');
    }, throwsArgumentError);
  }

  test_addFile_priorityFiles() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var b = newFile('$testPackageLibPath/b.dart', '');
    var c = newFile('$testPackageLibPath/c.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);
    driver.addFile2(c);
    driver.priorityFiles2 = [b];

    // 1. The priority file is produced first.
    // 2. Each analyzed file produces `ResolvedUnitResult`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/c.dart
  library: /home/test/lib/c.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/c.dart
    uri: package:test/c.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_addFile_removeFile() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Add, and immediately remove.
    driver.addFile2(a);
    driver.removeFile2(a);

    // No files to analyze.
    await assertEventsText(collector, r'''
[status] working
[status] idle
''');
  }

  test_addFile_thenRemove() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var b = newFile('$testPackageLibPath/b.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);

    // Now remove `a`.
    driver.removeFile2(a);

    // We remove `a` before analysis started.
    // So, only `b` was analyzed.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_cachedPriorityResults() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a];

    // Get the result, not cached.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // Get the (cached) result, not reported to the stream.
    collector.getResolvedUnit('A2', a);
    await assertEventsText(collector, r'''
[future] getResolvedUnit A2
  ResolvedUnitResult #0
''');

    // Get the (cached) result, reported to the stream.
    collector.getResolvedUnit('A3', a, sendCachedToStream: true);
    await assertEventsText(collector, r'''
[stream]
  ResolvedUnitResult #0
[future] getResolvedUnit A3
  ResolvedUnitResult #0
''');
  }

  test_cachedPriorityResults_flush_onAnyFileChange() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var b = newFile('$testPackageLibPath/b.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a];

    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // Change a file.
    // The cache is flushed, so we get a new result.
    driver.changeFile2(a);
    collector.getResolvedUnit('A2', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[operation] reuseLinkedBundle
  package:test/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A2
  ResolvedUnitResult #1
''');

    // Add `b`.
    // The cache is flushed, so we get a new result.
    driver.addFile2(b);
    collector.getResolvedUnit('A3', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #3
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A3
  ResolvedUnitResult #2
''');

    // Remove `b`.
    // The cache is flushed, so we get a new result.
    driver.removeFile2(b);
    collector.getResolvedUnit('A4', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[operation] reuseLinkedBundle SDK
[operation] reuseLinkedBundle
  package:test/a.dart
[stream]
  ResolvedUnitResult #4
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A4
  ResolvedUnitResult #4
''');
  }

  test_cachedPriorityResults_flush_onPrioritySetChange() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var b = newFile('$testPackageLibPath/b.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a];

    // Get the result for `a`, new.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // Make `a` and `b` priority.
    // We still have the result for `a` cached.
    driver.priorityFiles2 = [a, b];
    collector.getResolvedUnit('A2', a);
    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getResolvedUnit A2
  ResolvedUnitResult #0
''');

    // Get the result for `b`, new.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');

    // Get the result for `b`, cached.
    collector.getResolvedUnit('B2', b);
    await assertEventsText(collector, r'''
[future] getResolvedUnit B2
  ResolvedUnitResult #1
''');

    // Only `b` is priority.
    // The result for `a` is flushed, so analyzed when asked.
    driver.priorityFiles2 = [b];
    collector.getResolvedUnit('A3', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A3
  ResolvedUnitResult #2
''');
  }

  test_cachedPriorityResults_notPriority() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Always analyzed the first time.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // Analyzed again, because `a` is not priority.
    collector.getResolvedUnit('A2', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A2
  ResolvedUnitResult #1
''');
  }

  test_cachedPriorityResults_wholeLibrary_priorityLibrary_askLibrary() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a];

    // Ask the result for `a`, should cache for both `a` and `b`.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // Verify that the results for `a` and `b` are cached.
    // Note, no analysis.
    collector.getResolvedUnit('A2', a);
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[future] getResolvedUnit A2
  ResolvedUnitResult #0
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');

    // Ask for resolved library.
    // Note, no analysis.
    // Note, the units are cached.
    collector.getResolvedLibrary('L1', a);
    await assertEventsText(collector, r'''
[future] getResolvedLibrary L1
  ResolvedLibraryResult #2
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
      ResolvedUnitResult #1
''');
  }

  test_cachedPriorityResults_wholeLibrary_priorityLibrary_askPart() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a];

    // Ask the result for `b`, should cache for both `a` and `b`.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');

    // Verify that the results for `a` and `b` are cached.
    // Note, no analysis.
    collector.getResolvedUnit('A1', a);
    collector.getResolvedUnit('B2', b);
    await assertEventsText(collector, r'''
[future] getResolvedUnit A1
  ResolvedUnitResult #0
[future] getResolvedUnit B2
  ResolvedUnitResult #1
''');

    // Ask for resolved library.
    // Note, no analysis.
    // Note, the units are cached.
    collector.getResolvedLibrary('L1', a);
    await assertEventsText(collector, r'''
[future] getResolvedLibrary L1
  ResolvedLibraryResult #2
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
      ResolvedUnitResult #1
''');
  }

  test_cachedPriorityResults_wholeLibrary_priorityPart_askPart() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [b];

    // Ask the result for `b`, should cache for both `a` and `b`.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');

    // Verify that the results for `a` and `b` are cached.
    // Note, no analysis.
    collector.getResolvedUnit('A1', a);
    collector.getResolvedUnit('B2', b);
    await assertEventsText(collector, r'''
[future] getResolvedUnit A1
  ResolvedUnitResult #0
[future] getResolvedUnit B2
  ResolvedUnitResult #1
''');

    // Ask for resolved library.
    // Note, no analysis.
    // Note, the units are cached.
    collector.getResolvedLibrary('L1', a);
    await assertEventsText(collector, r'''
[future] getResolvedLibrary L1
  ResolvedLibraryResult #2
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
      ResolvedUnitResult #1
''');
  }

  test_changeFile_implicitlyAnalyzed() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
import 'b.dart';
var A = B;
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
var B = 0;
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a];
    driver.addFile2(a);

    configuration.libraryConfiguration.unitConfiguration.nodeSelector =
        (result) {
          return result.findNode.simple('B;');
        };

    // We have a result only for "a".
    // The type of `B` is `int`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedNode: SimpleIdentifier
      token: B
      element: package:test/b.dart::@getter::B
      staticType: int
[status] idle
''');

    // Change "b" and notify.
    modifyFile2(b, r'''
var B = 1.2;
''');
    driver.changeFile2(b);

    // While "b" is not analyzed explicitly, it is analyzed implicitly.
    // The change causes "a" to be reanalyzed.
    // The type of `B` is now `double`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedNode: SimpleIdentifier
      token: B
      element: package:test/b.dart::@getter::B
      staticType: double
[status] idle
''');
  }

  test_changeFile_notAbsolutePath() async {
    var driver = driverFor(testFile);
    expect(() {
      driver.changeFile('not_absolute.dart');
    }, throwsArgumentError);
  }

  test_changeFile_notExisting_toEmpty() async {
    var b = newFile('$testPackageLibPath/b.dart', '''
// ignore:unused_import
import 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(b);

    // `b` is analyzed, has an error.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    errors
      31 +8 URI_DOES_NOT_EXIST
[status] idle
''');

    // Create `a`, empty.
    var a = newFile('$testPackageLibPath/a.dart', '');
    driver.addFile2(a);

    // Both `a` and `b` are analyzed.
    // No errors anymore.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_changeFile_notPriority_errorsFromBytes() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);

    // Initial analysis, no errors.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
''');

    // Update the file, has an error.
    // Note, we analyze the file.
    modifyFile2(a, ';');
    driver.changeFile2(a);
    await assertEventsText(collector, r'''
[status] working
[operation] reuseLinkedBundle
  package:test/a.dart
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    errors
      0 +1 UNEXPECTED_TOKEN
[status] idle
''');

    // Update the file, no errors.
    // Note, we return errors from bytes.
    // We must update latest signatures, not reflected in the text.
    // If we don't, the next assert will fail.
    modifyFile2(a, '');
    driver.changeFile2(a);
    await assertEventsText(collector, r'''
[status] working
[operation] reuseLinkedBundle
  package:test/a.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ErrorsResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
[status] idle
''');

    // Update the file, has an error.
    // Note, we return errors from bytes.
    modifyFile2(a, ';');
    driver.changeFile2(a);
    await assertEventsText(collector, r'''
[status] working
[operation] reuseLinkedBundle
  package:test/a.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ErrorsResult #3
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
    errors
      0 +1 UNEXPECTED_TOKEN
[status] idle
''');
  }

  test_changeFile_notUsed() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var b = newFile('$testPackageLibPath/b.dart', 'class B1 {}');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);

    // Nothing interesting, "a" is analyzed.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
''');

    // Change "b" and notify.
    modifyFile2(b, 'class B2 {}');
    driver.changeFile2(b);

    // Nothing depends on "b", so nothing is analyzed.
    await assertEventsText(collector, r'''
[status] working
[status] idle
''');
  }

  test_changeFile_potentiallyAffected_imported() async {
    newFile('$testPackageLibPath/a.dart', '');

    var b = newFile('$testPackageLibPath/b.dart', '''
import 'a.dart';
''');

    var c = newFile('$testPackageLibPath/c.dart', '''
import 'b.dart';
''');

    var d = newFile('$testPackageLibPath/d.dart', '''
import 'c.dart';
''');

    newFile('$testPackageLibPath/e.dart', '');

    var driver = driverFor(testFile);

    Future<LibraryElement> getLibrary(String shortName) async {
      var uriStr = 'package:test/$shortName';
      var result = await driver.getLibraryByUriValid(uriStr);
      return result.element;
    }

    var a_element = await getLibrary('a.dart');
    var b_element = await getLibrary('b.dart');
    var c_element = await getLibrary('c.dart');
    var d_element = await getLibrary('d.dart');
    var e_element = await getLibrary('e.dart');

    // We have all libraries loaded after analysis.
    driver.assertLoadedLibraryUriSet(
      included: [
        'package:test/a.dart',
        'package:test/b.dart',
        'package:test/c.dart',
        'package:test/d.dart',
        'package:test/e.dart',
      ],
    );

    // All libraries have the current session.
    var session1 = driver.currentSession;
    expect(a_element.session, session1);
    expect(b_element.session, session1);
    expect(c_element.session, session1);
    expect(d_element.session, session1);
    expect(e_element.session, session1);

    // Change `b.dart`, also removes `c.dart` and `d.dart` that import it.
    // But `a.dart` and `d.dart` is not affected.
    driver.changeFile2(b);
    var affectedPathList = await driver.applyPendingFileChanges();
    expect(affectedPathList, unorderedEquals([b.path, c.path, d.path]));

    // We have a new session.
    var session2 = driver.currentSession;
    expect(session2, isNot(session1));

    driver.assertLoadedLibraryUriSet(
      excluded: [
        'package:test/b.dart',
        'package:test/c.dart',
        'package:test/d.dart',
      ],
      included: ['package:test/a.dart', 'package:test/e.dart'],
    );

    // `a.dart` and `e.dart` moved to the new session.
    // Invalidated libraries stuck with the old session.
    expect(a_element.session, session2);
    expect(b_element.session, session1);
    expect(c_element.session, session1);
    expect(d_element.session, session1);
    expect(e_element.session, session2);
  }

  test_changeFile_potentiallyAffected_part() async {
    var a = newFile('$testPackageLibPath/a.dart', '''
part of 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', '''
part 'a.dart';
''');

    var c = newFile('$testPackageLibPath/c.dart', '''
import 'b.dart';
''');

    newFile('$testPackageLibPath/d.dart', '');

    var driver = driverFor(testFile);

    Future<LibraryElement> getLibrary(String shortName) async {
      var uriStr = 'package:test/$shortName';
      var result = await driver.getLibraryByUriValid(uriStr);
      return result.element;
    }

    var b_element = await getLibrary('b.dart');
    var c_element = await getLibrary('c.dart');
    var d_element = await getLibrary('d.dart');

    // We have all libraries loaded after analysis.
    driver.assertLoadedLibraryUriSet(
      included: [
        'package:test/b.dart',
        'package:test/c.dart',
        'package:test/d.dart',
      ],
    );

    // All libraries have the current session.
    var session1 = driver.currentSession;
    expect(b_element.session, session1);
    expect(c_element.session, session1);
    expect(d_element.session, session1);

    // Change `a.dart`, remove `b.dart` that part it.
    // Removes `c.dart` that imports `b.dart`.
    // But `d.dart` is not affected.
    driver.changeFile2(a);
    var affectedPathList = await driver.applyPendingFileChanges();
    expect(affectedPathList, unorderedEquals([a.path, b.path, c.path]));

    // We have a new session.
    var session2 = driver.currentSession;
    expect(session2, isNot(session1));

    driver.assertLoadedLibraryUriSet(
      excluded: ['package:test/b.dart', 'package:test/c.dart'],
      included: ['package:test/d.dart'],
    );

    // `d.dart` moved to the new session.
    // Invalidated libraries stuck with the old session.
    expect(b_element.session, session1);
    expect(c_element.session, session1);
    expect(d_element.session, session2);
  }

  test_changeFile_selfConsistent() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
import 'b.dart';
final A1 = 1;
final A2 = B1;
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
import 'a.dart';
final B1 = A1;
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a, b];
    driver.addFile2(a);
    driver.addFile2(b);

    configuration.libraryConfiguration.unitConfiguration.variableTypesSelector =
        (result) {
          return switch (result.uriStr) {
            'package:test/a.dart' => [
              result.findElement2.topVar('A1'),
              result.findElement2.topVar('A2'),
            ],
            'package:test/b.dart' => [result.findElement2.topVar('B1')],
            _ => [],
          };
        };

    // We have results for both "a" and "b".
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedVariableTypes
      A1: int
      A2: int
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    selectedVariableTypes
      B1: int
[status] idle
''');

    // Update "a".
    modifyFile2(a, r'''
import 'b.dart';
final A1 = 1.2;
final A2 = B1;
''');
    driver.changeFile2(a);

    // We again get results for both "a" and "b".
    // The results are consistent.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedVariableTypes
      A1: double
      A2: double
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #3
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    selectedVariableTypes
      B1: double
[status] idle
''');
  }

  test_changeFile_single() async {
    var a = newFile('$testPackageLibPath/a.dart', 'var V = 1;');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.priorityFiles2 = [a];

    configuration.libraryConfiguration.unitConfiguration.variableTypesSelector =
        (result) {
          switch (result.uriStr) {
            case 'package:test/a.dart':
              return [result.findElement2.topVar('V')];
            default:
              return [];
          }
        };

    // Initial analysis.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedVariableTypes
      V: int
[status] idle
''');

    // Update the file, but don't notify the driver.
    // No new results.
    modifyFile2(a, 'var V = 1.2;');
    await assertEventsText(collector, r'''
''');

    // Notify the driver about the change.
    // We get a new result.
    driver.changeFile2(a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedVariableTypes
      V: double
[status] idle
''');
  }

  test_currentSession() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final v = 0;
''');

    var driver = driverFor(testFile);

    await driver.getResolvedUnit2(a);

    var session1 = driver.currentSession;
    expect(session1, isNotNull);

    modifyFile2(a, r'''
final v = 2;
''');
    driver.changeFile2(a);
    await driver.getResolvedUnit2(a);

    var session2 = driver.currentSession;
    expect(session2, isNotNull);

    // We get a new session.
    expect(session2, isNot(session1));
  }

  test_discoverAvailableFiles_packages() {
    writeTestPackageConfig(
      PackageConfigFileBuilder()
        ..add(name: 'aaa', rootPath: '$packagesRootPath/aaa')
        ..add(name: 'bbb', rootPath: '$packagesRootPath/bbb'),
    );

    var t1 = newFile('$testPackageLibPath/t1.dart', '');
    var a1 = newFile('$packagesRootPath/aaa/lib/a1.dart', '');
    var a2 = newFile('$packagesRootPath/aaa/lib/src/a2.dart', '');
    var a3 = newFile('$packagesRootPath/aaa/lib/a3.txt', '');
    var b1 = newFile('$packagesRootPath/bbb/lib/b1.dart', '');
    var c1 = newFile('$packagesRootPath/ccc/lib/c1.dart', '');

    var driver = driverFor(testFile);
    driver.addFile2(t1);

    // Don't add `a1`, `a2`, or `b1` - they should be discovered.
    // And `c` is not in the package config, so should not be discovered.
    driver.discoverAvailableFiles();

    var knownFiles = driver.knownFiles.resources;
    expect(knownFiles, contains(t1));
    expect(knownFiles, contains(a1));
    expect(knownFiles, contains(a2));
    expect(knownFiles, isNot(contains(a3)));
    expect(knownFiles, contains(b1));
    expect(knownFiles, isNot(contains(c1)));

    // We can wait for discovery more than once.
    driver.discoverAvailableFiles();
  }

  test_discoverAvailableFiles_sdk() {
    var driver = driverFor(testFile);
    driver.discoverAvailableFiles();
    expect(
      driver.knownFiles.resources,
      containsAll([
        sdkRoot.getChildAssumingFile('lib/async/async.dart'),
        sdkRoot.getChildAssumingFile('lib/collection/collection.dart'),
        sdkRoot.getChildAssumingFile('lib/core/core.dart'),
        sdkRoot.getChildAssumingFile('lib/math/math.dart'),
      ]),
    );
  }

  test_getCachedResolvedUnit() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    // Not cached.
    // Note, no analysis.
    collector.getCachedResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[future] getCachedResolvedUnit A1
  null
''');

    driver.priorityFiles2 = [a];
    collector.getResolvedUnit('A2', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A2
  ResolvedUnitResult #0
''');

    // Has cached.
    // Note, no analysis.
    collector.getCachedResolvedUnit('A3', a);
    await assertEventsText(collector, r'''
[future] getCachedResolvedUnit A3
  ResolvedUnitResult #0
''');
  }

  test_getErrors() async {
    var a = newFile('$testPackageLibPath/a.dart', '''
var v = 0
''');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    collector.getErrors('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    errors
      8 +1 EXPECTED_TOKEN
[status] idle
[future] getErrors A1
  ErrorsResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
    errors
      8 +1 EXPECTED_TOKEN
''');

    // The result is produced from bytes.
    collector.getErrors('A2', a);
    await assertEventsText(collector, r'''
[status] working
[operation] getErrorsFromBytes
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
[future] getErrors A2
  ErrorsResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
    errors
      8 +1 EXPECTED_TOKEN
''');
  }

  test_getErrors_library_part() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getErrors('A1', a);
    collector.getErrors('B1', b);

    // Note, both `getErrors()` returned during the library analysis.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getErrors A1
  ErrorsResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
[future] getErrors B1
  ErrorsResult #3
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: isPart
''');
  }

  test_getErrors_missingDartLibrary_async() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/async/async.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getErrors('A1', a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getErrors A1
  MissingSdkLibraryResult #0
    missingUri: dart:async
''');
  }

  test_getErrors_missingDartLibrary_core() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/core/core.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getErrors('A1', a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getErrors A1
  MissingSdkLibraryResult #0
    missingUri: dart:core
''');
  }

  test_getErrors_notAbsolutePath() async {
    var driver = driverFor(testFile);
    var result = await driver.getErrors('not_absolute.dart');
    expect(result, isA<InvalidPathResult>());
  }

  test_getFilesDefiningClassMemberName_class() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {
  void m1() {}
}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
class B {
  void m2() {}
}
''');

    var c = newFile('$testPackageLibPath/c.dart', r'''
class C {
  void m2() {}
}
''');

    var d = newFile('$testPackageLibPath/d.dart', r'''
class D {
  void m3() {}
}
''');

    var driver = driverFor(testFile);
    driver.addFile2(a);
    driver.addFile2(b);
    driver.addFile2(c);
    driver.addFile2(d);

    await driver.assertFilesDefiningClassMemberName('m1', [a]);
    await driver.assertFilesDefiningClassMemberName('m2', [b, c]);
    await driver.assertFilesDefiningClassMemberName('m3', [d]);
  }

  test_getFilesDefiningClassMemberName_mixin() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
mixin A {
  void m1() {}
}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
mixin B {
  void m2() {}
}
''');

    var c = newFile('$testPackageLibPath/c.dart', r'''
mixin C {
  void m2() {}
}
''');

    var d = newFile('$testPackageLibPath/d.dart', r'''
mixin D {
  void m3() {}
}
''');

    var driver = driverFor(testFile);
    driver.addFile2(a);
    driver.addFile2(b);
    driver.addFile2(c);
    driver.addFile2(d);

    await driver.assertFilesDefiningClassMemberName('m1', [a]);
    await driver.assertFilesDefiningClassMemberName('m2', [b, c]);
    await driver.assertFilesDefiningClassMemberName('m3', [d]);
  }

  test_getFilesReferencingName() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
import 'a.dart';
void f(A a) {}
''');

    var c = newFile('$testPackageLibPath/c.dart', r'''
import 'a.dart';
void f(A a) {}
''');

    var d = newFile('$testPackageLibPath/d.dart', r'''
class A {}
void f(A a) {}
''');

    var e = newFile('$testPackageLibPath/e.dart', r'''
import 'a.dart';
void main() {}
''');

    var driver = driverFor(testFile);
    driver.addFile2(a);
    driver.addFile2(b);
    driver.addFile2(c);
    driver.addFile2(d);
    driver.addFile2(e);

    // `b` references an external `A`.
    // `c` references an external `A`.
    // `d` references the local `A`.
    // `e` does not reference `A` at all.
    await driver.assertFilesReferencingName(
      'A',
      includesAll: [b, c],
      excludesAll: [d, e],
    );

    // We get the same results second time.
    await driver.assertFilesReferencingName(
      'A',
      includesAll: [b, c],
      excludesAll: [d, e],
    );
  }

  test_getFilesReferencingName_discover() async {
    writeTestPackageConfig(
      PackageConfigFileBuilder()
        ..add(name: 'aaa', rootPath: '$packagesRootPath/aaa')
        ..add(name: 'bbb', rootPath: '$packagesRootPath/bbb'),
    );

    var t = newFile('$testPackageLibPath/t.dart', '''
int t = 0;
''');

    var a = newFile('$packagesRootPath/aaa/lib/a.dart', '''
int a = 0;
''');

    var b = newFile('$packagesRootPath/bbb/lib/b.dart', '''
int b = 0;
''');

    var c = newFile('$packagesRootPath/ccc/lib/c.dart', '''
int c = 0;
''');

    var driver = driverFor(testFile);
    driver.addFile2(t);

    await driver.assertFilesReferencingName(
      'int',
      includesAll: [t, a, b],
      excludesAll: [c],
    );
  }

  test_getFileSync_changedFile() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var b = newFile('$testPackageLibPath/b.dart', r'''
import 'a.dart';

void f(A a) {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Ensure that `a` library cycle is loaded.
    // So, `a` is in the library context.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // Update the file, changing its API signature.
    // Note that we don't call `changeFile`.
    modifyFile2(a, 'class A {}\n');

    // Get the file.
    // We have not called `changeFile(a)`, so we should not read the file.
    // Moreover, doing this will create a new library cycle [a.dart].
    // Library cycles are compared by their identity, so we would try to
    // reload linked summary for [a.dart], and crash.
    expect(driver.getFileSyncValid(a).lineInfo.lineCount, 1);

    // We have not read `a.dart`, so `A` is still not declared.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    errors
      25 +1 UNDEFINED_CLASS
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');

    // Notify the driver that the file was changed.
    driver.changeFile2(a);

    // ...and apply this change.
    await driver.applyPendingFileChanges();
    await assertEventsText(collector, r'''
[status] working
[status] idle
''');

    // So, `class A {}` is declared now.
    expect(driver.getFileSyncValid(a).lineInfo.lineCount, 2);

    // ...and `b` has no errors.
    collector.getResolvedUnit('B2', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit B2
  ResolvedUnitResult #2
''');
  }

  test_getFileSync_library() async {
    var content = 'class A {}';
    var a = newFile('$testPackageLibPath/a.dart', content);
    var driver = driverFor(testFile);
    var result = driver.getFileSyncValid(a);
    expect(result.path, a.path);
    expect(result.uri.toString(), 'package:test/a.dart');
    expect(result.content, content);
    expect(result.isLibrary, isTrue);
    expect(result.isPart, isFalse);
  }

  test_getFileSync_notAbsolutePath() async {
    var driver = driverFor(testFile);
    var result = driver.getFileSync('not_absolute.dart');
    expect(result, isA<InvalidPathResult>());
  }

  test_getFileSync_part() async {
    var content = 'part of lib;';
    var a = newFile('$testPackageLibPath/a.dart', content);
    var driver = driverFor(testFile);
    var result = driver.getFileSyncValid(a);
    expect(result.path, a.path);
    expect(result.uri.toString(), 'package:test/a.dart');
    expect(result.content, content);
    expect(result.isLibrary, isFalse);
    expect(result.isPart, isTrue);
  }

  test_getIndex() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
void foo() {}

void f() {
  foo();
}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getIndex('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getIndex A1
  strings
    --nullString--
    foo
    package:test/a.dart
''');
  }

  test_getIndex_missingDartLibrary_async() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/async/async.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getIndex('A1', a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getIndex A1
''');
  }

  test_getIndex_missingDartLibrary_core() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/core/core.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getIndex('A1', a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getIndex A1
''');
  }

  test_getIndex_notAbsolutePath() async {
    var driver = driverFor(testFile);
    expect(() async {
      await driver.getIndex('not_absolute.dart');
    }, throwsArgumentError);
  }

  test_getLibraryByUri() async {
    var aUriStr = 'package:test/a.dart';
    var bUriStr = 'package:test/b.dart';

    newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';

class A {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';

class B {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    var result = await driver.getLibraryByUri(aUriStr);
    result as LibraryElementResult;
    expect(result.element.getClass('A'), isNotNull);
    expect(result.element.getClass('B'), isNotNull);

    // It is an error to ask for a library when we know that it is a part.
    expect(
      await driver.getLibraryByUri(bUriStr),
      isA<NotLibraryButPartResult>(),
    );

    // No analysis.
    await assertEventsText(collector, r'''
[status] working
[status] idle
''');
  }

  test_getLibraryByUri_cannotResolveUri() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getLibraryByUri('X', 'foo:bar');

    await assertEventsText(collector, r'''
[future] getLibraryByUri X
  CannotResolveUriResult
''');
  }

  test_getLibraryByUri_missingDartLibrary_async() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/async/async.dart').delete();

    newFile('$testPackageLibPath/a.dart', '');
    collector.getLibraryByUri('A1', 'package:test/a.dart');

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getLibraryByUri A1
  MissingSdkLibraryResult #0
    missingUri: dart:async
''');
  }

  test_getLibraryByUri_missingDartLibrary_core() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/core/core.dart').delete();

    newFile('$testPackageLibPath/a.dart', '');
    collector.getLibraryByUri('A1', 'package:test/a.dart');

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getLibraryByUri A1
  MissingSdkLibraryResult #0
    missingUri: dart:core
''');
  }

  test_getLibraryByUri_notLibrary_part() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part of 'b.dart';
''');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    var uriStr = 'package:test/a.dart';
    collector.getLibraryByUri('X', uriStr);

    await assertEventsText(collector, r'''
[future] getLibraryByUri X
  NotLibraryButPartResult
''');
  }

  test_getLibraryByUri_subsequentCallsDoesNoWork() async {
    var aUriStr = 'package:test/a.dart';
    var bUriStr = 'package:test/b.dart';

    newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';

class A {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';

class B {}
''');

    for (var run = 0; run < 5; run++) {
      var driver = driverFor(testFile);
      var collector = DriverEventCollector(driver);

      var result = await driver.getLibraryByUri(aUriStr);
      result as LibraryElementResult;
      expect(result.element.getClass('A'), isNotNull);
      expect(result.element.getClass('B'), isNotNull);

      // It is an error to ask for a library when we know that it is a part.
      expect(
        await driver.getLibraryByUri(bUriStr),
        isA<NotLibraryButPartResult>(),
      );

      if (run == 0) {
        // First `getLibraryByUri` call does actual work.
        await assertEventsText(collector, r'''
[status] working
[status] idle
''');
      } else {
        // Subsequent `getLibraryByUri` just grabs the result via rootReference
        // and thus does no actual work.
        await assertEventsText(collector, '');
      }
    }
  }

  test_getLibraryByUri_unresolvedUri() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    var result = await driver.getLibraryByUri('package:foo/foo.dart');
    expect(result, isA<CannotResolveUriResult>());

    // No analysis.
    await assertEventsText(collector, '');
  }

  test_getParsedLibrary() async {
    var content = 'class A {}';
    var a = newFile('$testPackageLibPath/a.dart', content);

    var driver = driverFor(testFile);
    var result = driver.getParsedLibrary2(a);
    result as ParsedLibraryResult;
    expect(result.units, hasLength(1));
    expect(result.units[0].path, a.path);
    expect(result.units[0].content, content);
    expect(result.units[0].unit, isNotNull);
    expect(result.units[0].diagnostics, isEmpty);
  }

  test_getParsedLibrary_invalidPath_notAbsolute() async {
    var driver = driverFor(testFile);
    var result = driver.getParsedLibrary('not_absolute.dart');
    expect(result, isA<InvalidPathResult>());
  }

  test_getParsedLibrary_notLibraryButPart() async {
    var driver = driverFor(testFile);
    var a = newFile('$testPackageLibPath/a.dart', 'part of my;');
    var result = driver.getParsedLibrary2(a);
    expect(result, isA<NotLibraryButPartResult>());
  }

  test_getParsedLibraryByUri() async {
    var content = 'class A {}';
    var a = newFile('$testPackageLibPath/a.dart', content);

    var driver = driverFor(testFile);

    var uri = Uri.parse('package:test/a.dart');
    var result = driver.getParsedLibraryByUri(uri);
    result as ParsedLibraryResult;
    expect(result.units, hasLength(1));
    expect(result.units[0].uri, uri);
    expect(result.units[0].path, a.path);
    expect(result.units[0].content, content);
  }

  test_getParsedLibraryByUri_cannotResolveUri() async {
    var driver = driverFor(testFile);
    var uri = Uri.parse('foo:bar');
    expect(driver.getParsedLibraryByUri(uri), isA<CannotResolveUriResult>());
  }

  test_getParsedLibraryByUri_notLibrary_part() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part of 'b.dart';
''');

    var driver = driverFor(a);
    var uri = Uri.parse('package:test/a.dart');
    expect(driver.getParsedLibraryByUri(uri), isA<NotLibraryButPartResult>());
  }

  test_getParsedLibraryByUri_notLibraryButPart() async {
    newFile('$testPackageLibPath/a.dart', 'part of my;');
    var driver = driverFor(testFile);
    var uri = Uri.parse('package:test/a.dart');
    var result = driver.getParsedLibraryByUri(uri);
    expect(result, isA<NotLibraryButPartResult>());
  }

  test_getParsedLibraryByUri_unresolvedUri() async {
    var driver = driverFor(testFile);
    var uri = Uri.parse('package:unknown/a.dart');
    var result = driver.getParsedLibraryByUri(uri);
    expect(result, isA<CannotResolveUriResult>());
  }

  test_getResolvedLibrary() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getResolvedLibrary('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedLibrary A1
  ResolvedLibraryResult #1
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
''');
  }

  test_getResolvedLibrary_cachePriority() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a];

    collector.getResolvedLibrary('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedLibrary A1
  ResolvedLibraryResult #2
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
      ResolvedUnitResult #1
''');

    // Ask again, the same cached instance should be returned.
    // Note, no analysis.
    // Note, the result is cached.
    collector.getResolvedLibrary('A2', a);
    await assertEventsText(collector, r'''
[future] getResolvedLibrary A2
  ResolvedLibraryResult #2
''');

    // Ask `a`, returns cached.
    // Note, no analysis.
    collector.getResolvedUnit('A3', a);
    await assertEventsText(collector, r'''
[future] getResolvedUnit A3
  ResolvedUnitResult #0
''');

    // Ask `b`, returns cached.
    // Note, no analysis.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');
  }

  test_getResolvedLibrary_missingDartLibrary_async() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/async/async.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getResolvedLibrary('A1', a);

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
[future] getResolvedLibrary A1
  MissingSdkLibraryResult #0
    missingUri: dart:async
''');
  }

  test_getResolvedLibrary_missingDartLibrary_core() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/core/core.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getResolvedLibrary('A1', a);

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
[future] getResolvedLibrary A1
  MissingSdkLibraryResult #0
    missingUri: dart:core
''');
  }

  test_getResolvedLibrary_notAbsolutePath() async {
    var driver = driverFor(testFile);
    var result = await driver.getResolvedLibrary('not_absolute.dart');
    expect(result, isA<InvalidPathResult>());
  }

  test_getResolvedLibrary_notLibrary_part() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part of 'b.dart';
''');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    collector.getResolvedLibrary('X', a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getResolvedLibrary X
  NotLibraryButPartResult
''');
  }

  test_getResolvedLibrary_pending_changeFile() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Ask the resolved library.
    // We used to record the request with the `LibraryFileKind`.
    collector.getResolvedLibrary('A1', a);

    // ...the request is pending, notify that the file changed.
    // This forces its reading, and rebuilding its `kind`.
    // So, the old `kind` is not valid anymore.
    // This used to cause infinite processing of the request.
    // https://github.com/dart-lang/sdk/issues/54708
    driver.changeFile2(a);

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedLibrary A1
  ResolvedLibraryResult #1
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
''');
  }

  test_getResolvedLibraryByUri() async {
    newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    var uri = Uri.parse('package:test/a.dart');
    collector.getResolvedLibraryByUri('A1', uri);

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedLibraryByUri A1
  ResolvedLibraryResult #1
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
''');
  }

  test_getResolvedLibraryByUri_cannotResolveUri() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    var uri = Uri.parse('foo:bar');
    collector.getResolvedLibraryByUri('X', uri);

    await assertEventsText(collector, r'''
[future] getResolvedLibraryByUri X
  CannotResolveUriResult
''');
  }

  test_getResolvedLibraryByUri_library_pending_getResolvedUnit() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(a);

    var collector = DriverEventCollector(driver);
    collector.getResolvedUnit('A1', a);
    collector.getResolvedUnit('B1', b);

    var uri = Uri.parse('package:test/a.dart');
    collector.getResolvedLibraryByUri('A2', uri);

    // Note, the library is resolved only once.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
[future] getResolvedUnit B1
  ResolvedUnitResult #1
[future] getResolvedLibraryByUri A2
  ResolvedLibraryResult #2
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
      ResolvedUnitResult #1
''');
  }

  test_getResolvedLibraryByUri_missingDartLibrary_async() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/async/async.dart').delete();

    newFile('$testPackageLibPath/a.dart', '');
    collector.getResolvedLibraryByUri('A1', Uri.parse('package:test/a.dart'));

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
[future] getResolvedLibraryByUri A1
  MissingSdkLibraryResult #0
    missingUri: dart:async
''');
  }

  test_getResolvedLibraryByUri_missingDartLibrary_core() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/core/core.dart').delete();

    newFile('$testPackageLibPath/a.dart', '');
    collector.getResolvedLibraryByUri('A1', Uri.parse('package:test/a.dart'));

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
[future] getResolvedLibraryByUri A1
  MissingSdkLibraryResult #0
    missingUri: dart:core
''');
  }

  test_getResolvedLibraryByUri_notLibrary_part() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part of 'b.dart';
''');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    var uri = Uri.parse('package:test/a.dart');
    collector.getResolvedLibraryByUri('X', uri);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getResolvedLibraryByUri X
  NotLibraryButPartResult
''');
  }

  test_getResolvedLibraryByUri_notLibraryButPart() async {
    newFile('$testPackageLibPath/a.dart', 'part of my;');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    var uri = Uri.parse('package:test/a.dart');
    collector.getResolvedLibraryByUri('A1', uri);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getResolvedLibraryByUri A1
  NotLibraryButPartResult
''');
  }

  test_getResolvedLibraryByUri_unresolvedUri() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    var uri = Uri.parse('package:unknown/a.dart');
    collector.getResolvedLibraryByUri('A1', uri);

    await assertEventsText(collector, r'''
[future] getResolvedLibraryByUri A1
  CannotResolveUriResult
''');
  }

  test_getResolvedUnit() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_added() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    collector.getResolvedUnit('A1', a);

    // Note, no separate `ErrorsResult`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_importLibrary_thenRemoveIt() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
import 'a.dart';
class B extends A {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);

    // No errors in `a` or `b`.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #0
''');

    // Remove `a` and reanalyze.
    deleteFile(a.path);
    driver.removeFile2(a);

    // The unresolved URI error must be reported.
    collector.getResolvedUnit('B2', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[operation] reuseLinkedBundle SDK
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    errors
      7 +8 URI_DOES_NOT_EXIST
      33 +1 EXTENDS_NON_CLASS
[status] idle
[future] getResolvedUnit B2
  ResolvedUnitResult #2
''');

    // Restore `a`.
    newFile(a.path, 'class A {}');
    driver.addFile2(a);

    // No errors in `b` again.
    collector.getResolvedUnit('B2', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[operation] reuseLinkedBundle
  package:test/a.dart
[operation] reuseLinkedBundle
  package:test/b.dart
[stream]
  ResolvedUnitResult #3
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[operation] getErrorsFromBytes
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ErrorsResult #4
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
[status] idle
[future] getResolvedUnit B2
  ResolvedUnitResult #3
''');
  }

  test_getResolvedUnit_library_added_part() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);
    collector.getResolvedUnit('A1', a);

    // Note, the library is resolved only once.
    // Note, no separate `ErrorsResult` for `a` or `b`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_library_part() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getResolvedUnit('A1', a);
    collector.getResolvedUnit('B1', b);

    // Note, the library is resolved only once.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');
  }

  test_getResolvedUnit_library_pending_getErrors_part() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getErrors('B1', b);
    collector.getResolvedUnit('A1', a);

    // Note, the library is resolved only once.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getErrors B1
  ErrorsResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: isPart
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_missingDartLibrary_async() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/async/async.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getResolvedUnit('A1', a);

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
[future] getResolvedUnit A1
  MissingSdkLibraryResult #0
    missingUri: dart:async
''');
  }

  test_getResolvedUnit_missingDartLibrary_core() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/core/core.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getResolvedUnit('A1', a);

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
[future] getResolvedUnit A1
  MissingSdkLibraryResult #0
    missingUri: dart:core
''');
  }

  test_getResolvedUnit_notDartFile() async {
    var a = newFile('$testPackageLibPath/a.txt', r'''
final foo = 0;
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    configuration.libraryConfiguration.unitConfiguration.variableTypesSelector =
        (result) {
          return [result.findElement2.topVar('foo')];
        };

    // The extension of the file does not matter.
    // If asked, we analyze it as Dart.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.txt
  library: /home/test/lib/a.txt
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.txt
    uri: package:test/a.txt
    flags: exists isLibrary
    selectedVariableTypes
      foo: int
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_part_doesNotExist_lints() async {
    newFile('$testPackageRootPath/analysis_options.yaml', r'''
linter:
  rules:
    - omit_local_variable_types
''');

    await assertErrorsInCode(
      r'''
library my.lib;
part 'a.dart';
''',
      [error(CompileTimeErrorCode.uriDoesNotExist, 21, 8)],
    );
  }

  test_getResolvedUnit_part_empty_lints() async {
    newFile('$testPackageRootPath/analysis_options.yaml', r'''
linter:
  rules:
    - omit_local_variable_types
''');

    newFile('$testPackageLibPath/a.dart', '');

    await assertErrorsInCode(
      r'''
library my.lib;
part 'a.dart';
''',
      [error(CompileTimeErrorCode.partOfNonPart, 21, 8)],
    );
  }

  test_getResolvedUnit_part_hasPartOfName_notThisLibrary_lints() async {
    newFile('$testPackageRootPath/analysis_options.yaml', r'''
linter:
  rules:
    - omit_local_variable_types
''');

    newFile('$testPackageLibPath/a.dart', r'''
part of other.lib;
''');

    await assertErrorsInCode(
      r'''
library my.lib;
part 'a.dart';
''',
      [error(CompileTimeErrorCode.partOfDifferentLibrary, 21, 8)],
    );
  }

  test_getResolvedUnit_part_hasPartOfUri_notThisLibrary_lints() async {
    newFile('$testPackageRootPath/analysis_options.yaml', r'''
linter:
  rules:
    - omit_local_variable_types
''');

    newFile('$testPackageLibPath/a.dart', r'''
part of 'not_test.dart';
''');

    await assertErrorsInCode(
      r'''
library my.lib;
part 'a.dart';
''',
      [error(CompileTimeErrorCode.partOfDifferentLibrary, 21, 8)],
    );
  }

  test_getResolvedUnit_part_library() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getResolvedUnit('B1', b);
    collector.getResolvedUnit('A1', a);

    // Note, the library is resolved only once.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #1
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_part_pending_getErrors_library() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getErrors('A1', a);
    collector.getResolvedUnit('B1', b);

    // Note, the library is resolved only once.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getErrors A1
  ErrorsResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');
  }

  test_getResolvedUnit_pending_getErrors() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getResolvedUnit('A1', a);
    collector.getErrors('A2', a);

    // Note, the library is resolved only once.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
[future] getErrors A2
  ErrorsResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
''');
  }

  test_getResolvedUnit_pending_getErrors2() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getErrors('A1', a);
    collector.getResolvedUnit('A2', a);

    // Note, the library is resolved only once.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getErrors A1
  ErrorsResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
[future] getResolvedUnit A2
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_pending_getIndex() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getIndex('A1', a);
    collector.getResolvedUnit('A2', a);

    // Note, no separate `getIndex` result.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getIndex A1
  strings
    --nullString--
[future] getResolvedUnit A2
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_thenRemove() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Schedule resolved unit computation.
    collector.getResolvedUnit('A1', a);

    // ...and remove the file.
    driver.removeFile2(a);

    // The future with the result still completes.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');
  }

  test_getResolvedUnit_twoPendingFutures() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Ask the same file twice.
    collector.getResolvedUnit('A1', a);
    collector.getResolvedUnit('A2', a);

    // Both futures complete.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
[future] getResolvedUnit A2
  ResolvedUnitResult #0
''');
  }

  test_getUnitElement() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
void foo() {}
void bar() {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    configuration.unitElementConfiguration.elementSelector = (unitFragment) {
      return unitFragment.functions
          .map((fragment) => fragment.element)
          .toList();
    };

    collector.getUnitElement('A1', a);
    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getUnitElement A1
  path: /home/test/lib/a.dart
  uri: package:test/a.dart
  flags: isLibrary
  enclosing: <null>
  selectedElements
    package:test/a.dart::@function::foo
    package:test/a.dart::@function::bar
''');
  }

  test_getUnitElement_doesNotExist_afterResynthesized() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
import 'package:test/b.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    collector.getResolvedLibrary('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    errors
      7 +21 URI_DOES_NOT_EXIST
[status] idle
[future] getResolvedLibrary A1
  ResolvedLibraryResult #1
    element: package:test/a.dart
    units
      ResolvedUnitResult #0
''');

    collector.getUnitElement('A2', a);
    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getUnitElement A2
  path: /home/test/lib/a.dart
  uri: package:test/a.dart
  flags: isLibrary
  enclosing: <null>
''');
  }

  test_getUnitElement_invalidPath_notAbsolute() async {
    var driver = driverFor(testFile);
    var result = await driver.getUnitElement('not_absolute.dart');
    expect(result, isA<InvalidPathResult>());
  }

  test_getUnitElement_missingDartLibrary_async() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/async/async.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getUnitElement('A1', a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getUnitElement A1
  MissingSdkLibraryResult #0
    missingUri: dart:async
''');
  }

  test_getUnitElement_missingDartLibrary_core() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    sdkRoot.getChildAssumingFile('lib/core/core.dart').delete();

    var a = newFile('$testPackageLibPath/a.dart', '');
    collector.getUnitElement('A1', a);

    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getUnitElement A1
  MissingSdkLibraryResult #0
    missingUri: dart:core
''');
  }

  test_hermetic_modifyLibraryFile_resolvePart() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
part 'b.dart';
final A = 0;
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
part of 'a.dart';
final B = A;
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    configuration.libraryConfiguration.unitConfiguration.variableTypesSelector =
        (result) {
          switch (result.uriStr) {
            case 'package:test/b.dart':
              return [result.findElement2.topVar('B')];
            default:
              return [];
          }
        };

    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
    selectedVariableTypes
      B: int
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');

    // Modify the library, but don't notify the driver.
    // The driver should use the previous library content and elements.
    modifyFile2(a, r'''
part 'b.dart';
final A = 1.2;
''');

    // Note, still `B: int`, not `B: double` yet.
    collector.getResolvedUnit('B2', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #3
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
    selectedVariableTypes
      B: int
[status] idle
[future] getResolvedUnit B2
  ResolvedUnitResult #3
''');
  }

  test_importOfNonLibrary_part_afterLibrary() async {
    var a = newFile('$testPackageLibPath/a.dart', '''
part 'b.dart';
''');

    newFile('$testPackageLibPath/b.dart', '''
part of 'a.dart';
class B {}
''');

    var c = newFile('$testPackageLibPath/c.dart', '''
import 'b.dart';
''');

    var driver = driverFor(testFile);

    // This ensures that `a` linked library is cached.
    await driver.getResolvedUnit2(a);

    // Should not fail because of considering `b` part as `a` library.
    await driver.getResolvedUnit2(c);
  }

  test_knownFiles() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
// ignore:unused_import
import 'b.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
''');

    var c = newFile('$testPackageLibPath/c.dart', r'''
''');

    var driver = driverFor(testFile);

    driver.addFile2(a);
    driver.addFile2(c);
    await pumpEventQueue(times: 5000);
    expect(driver.knownFiles.resources, contains(a));
    expect(driver.knownFiles.resources, contains(b));
    expect(driver.knownFiles.resources, contains(c));

    // Remove `a` and analyze.
    // Both `a` and `b` are not known now.
    driver.removeFile2(a);
    await pumpEventQueue(times: 5000);
    expect(driver.knownFiles.resources, isNot(contains(a)));
    expect(driver.knownFiles.resources, isNot(contains(b)));
    expect(driver.knownFiles.resources, contains(c));
  }

  test_knownFiles_beforeAnalysis() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var driver = driverFor(testFile);

    // `a` is added, but not processed yet.
    // So, the set of known files is empty yet.
    driver.addFile2(a);
    expect(driver.knownFiles, isEmpty);
  }

  test_linkedBundleProvider_changeFile() async {
    var a = newFile('$testPackageLibPath/a.dart', 'var V = 1;');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.priorityFiles2 = [a];

    configuration.libraryConfiguration.unitConfiguration.variableTypesSelector =
        (result) {
          switch (result.uriStr) {
            case 'package:test/a.dart':
              return [result.findElement2.topVar('V')];
            default:
              return [];
          }
        };

    // Initial analysis.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedVariableTypes
      V: int
[status] idle
''');

    // When no fine-grained dependencies, we don't cache bundles.
    // So, [LinkedBundleProvider] is empty, and not printed.
    assertDriverStateString(testFile, r'''
files
  /home/test/lib/a.dart
    uri: package:test/a.dart
    current
      id: file_0
      kind: library_0
        libraryImports
          library_1 dart:core synthetic
        fileKinds: library_0
        cycle_0
          dependencies: dart:core
          libraries: library_0
          apiSignature_0
      unlinkedKey: k00
libraryCycles
  /home/test/lib/a.dart
    current: cycle_0
      key: k01
    get: []
    put: [k01]
elementFactory
  hasElement
    package:test/a.dart
''');

    // Update the file, but don't notify the driver.
    // No new results.
    modifyFile2(a, 'var V = 1.2;');
    await assertEventsText(collector, r'''
''');

    // Notify the driver about the change.
    // We get a new result.
    driver.changeFile2(a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedVariableTypes
      V: double
[status] idle
''');

    assertDriverStateString(testFile, r'''
files
  /home/test/lib/a.dart
    uri: package:test/a.dart
    current
      id: file_0
      kind: library_6
        libraryImports
          library_1 dart:core synthetic
        fileKinds: library_6
        cycle_2
          dependencies: dart:core
          libraries: library_6
          apiSignature_1
      unlinkedKey: k02
libraryCycles
  /home/test/lib/a.dart
    current: cycle_2
      key: k03
    get: []
    put: [k01, k03]
elementFactory
  hasElement
    package:test/a.dart
''');
  }

  test_parseFileSync_appliesPendingFileChanges() async {
    var initialContent = 'initial content';
    var updatedContent = 'updated content';
    var a = newFile('$testPackageLibPath/a.dart', initialContent);

    // Check initial content.
    var driver = driverFor(testFile);
    var parsed = driver.parseFileSync(a.path) as ParsedUnitResult;
    expect(parsed.content, initialContent);

    // Update the file.
    newFile(a.path, updatedContent);
    driver.changeFile(a.path);

    // Expect parseFileSync to return the updated content.
    parsed = driver.parseFileSync(a.path) as ParsedUnitResult;
    expect(parsed.content, updatedContent);
  }

  test_parseFileSync_changedFile() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// ignore:unused_import
import 'a.dart';
void f(A a) {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Ensure that [a] library cycle is loaded.
    // So, `a` is in the library context.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // Update the file, changing its API signature.
    // Note that we don't call `changeFile`.
    modifyFile2(a, r'''
class A {}
''');

    // Parse the file.
    // We have not called `changeFile(a)`, so we should not read the file.
    // Moreover, doing this will create a new library cycle [a].
    // Library cycles are compared by their identity, so we would try to
    // reload linked summary for [a], and crash.
    {
      var parseResult = driver.parseFileSync2(a) as ParsedUnitResult;
      expect(parseResult.unit.declarations, isEmpty);
    }

    // We have not read `a`, so `A` is still not declared.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    errors
      48 +1 UNDEFINED_CLASS
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');

    // Notify the driver that `a` was changed.
    driver.changeFile2(a);

    // The pending change to `a` declares `A`.
    // So, `b` does not have errors anymore.
    collector.getResolvedUnit('B2', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
[future] getResolvedUnit B2
  ResolvedUnitResult #2
''');

    // We apply pending changes while handling request.
    // So, now `class A {}` is declared.
    {
      var result = driver.parseFileSync2(a) as ParsedUnitResult;
      assertParsedNodeText(result.unit, r'''
CompilationUnit
  declarations
    ClassDeclaration
      classKeyword: class
      name: A
      leftBracket: {
      rightBracket: }
''');
    }
  }

  test_parseFileSync_doesNotReadImportedFiles() async {
    newFile('$testPackageLibPath/a.dart', r'''
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// ignore:unused_import
import 'a.dart';
''');

    var driver = driverFor(testFile);
    expect(driver.knownFiles, isEmpty);

    // Don't read `a` when parse.
    driver.parseFileSync2(b);
    expect(driver.knownFiles.resources, unorderedEquals([b]));

    // Still don't read `a.dart` when parse the second time.
    driver.parseFileSync2(b);
    expect(driver.knownFiles.resources, unorderedEquals([b]));
  }

  test_parseFileSync_notAbsolutePath() async {
    var driver = driverFor(testFile);
    var result = driver.parseFileSync('not_absolute.dart');
    expect(result, isA<InvalidPathResult>());
  }

  test_parseFileSync_notDart() async {
    var a = newFile('$testPackageLibPath/a.txt', r'''
class A {}
''');

    var driver = driverFor(testFile);

    var result = driver.parseFileSync2(a) as ParsedUnitResult;
    assertParsedNodeText(result.unit, r'''
CompilationUnit
  declarations
    ClassDeclaration
      classKeyword: class
      name: A
      leftBracket: {
      rightBracket: }
''');

    expect(driver.knownFiles.resources, unorderedEquals([a]));
  }

  test_partOfName_getErrors_afterLibrary() async {
    // Note, we put the library into a different directory.
    // Otherwise we will discover it.
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Process `a` so that we know that it's a library for `b`.
    collector.getErrors('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/hidden/a.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getErrors A1
  ErrorsResult #2
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: isLibrary
''');

    // We return cached errors.
    // TODO(scheglov): don't switch to analysis?
    collector.getErrors('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] getErrorsFromBytes
  file: /home/test/lib/b.dart
  library: /home/test/lib/hidden/a.dart
[status] idle
[future] getErrors B1
  ErrorsResult #3
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: isPart
''');
  }

  test_partOfName_getErrors_beforeLibrary_addedFiles() async {
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// preEnhancedParts
// @dart = 3.4
part of a;
final a = A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // We discover all added files are maybe libraries.
    driver.addFile2(a);
    driver.addFile2(b);

    // Because `a` is added, we know how to analyze `b`.
    // So, it has no errors.
    collector.getErrors('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getErrors B1
  ErrorsResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: isPart
''');
  }

  test_partOfName_getErrors_beforeLibrary_discovered() async {
    newFile('$testPackageLibPath/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part 'b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // We discover sibling files as libraries.
    // So, we know that `a` is the library of `b`.
    // So, no errors.
    collector.getErrors('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getErrors B1
  ErrorsResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: isPart
''');
  }

  test_partOfName_getErrors_beforeLibrary_notDiscovered() async {
    newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // We don't know that `a` is the library of `b`.
    // So, we treat it as its own library, has errors.
    collector.getErrors('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[operation] reuseLinkedBundle
  package:test/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
    errors
      60 +1 CREATION_WITH_NON_TYPE
[status] idle
[future] getErrors B1
  ErrorsResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: isPart
    errors
      60 +1 CREATION_WITH_NON_TYPE
''');
  }

  test_partOfName_getResolvedUnit_afterLibrary() async {
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Process `a` so that we know that it's a library for `b`.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/hidden/a.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // We know that `b` is analyzed as part of `a`.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #3
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #3
''');
  }

  test_partOfName_getResolvedUnit_beforeLibrary_addedFiles() async {
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // We discover all added files are maybe libraries.
    driver.addFile2(a);
    driver.addFile2(b);

    // Because `a` is added, we know how to analyze `b`.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #1
''');
  }

  test_partOfName_getResolvedUnit_beforeLibrary_notDiscovered() async {
    newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // We don't know that `a` is the library of `b`.
    // So, we treat it as its own library.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
    errors
      60 +1 CREATION_WITH_NON_TYPE
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #0
''');
  }

  test_partOfName_getResolvedUnit_changePart_invalidatesLibraryCycle() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
import 'dart:async';
part 'b.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);

    // Analyze the library without the part.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    errors
      61 +8 URI_DOES_NOT_EXIST
      42 +12 UNUSED_IMPORT
[status] idle
''');

    // Create the part file.
    // This should invalidate library file state (specifically the library
    // cycle), so that we can re-link the library, and get new dependencies.
    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of 'a.dart';
Future<int>? f;
''');
    driver.changeFile2(b);

    // This should not crash.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #2
''');
  }

  test_partOfName_getResolvedUnit_hasLibrary_noPart() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library my.lib;
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of my.lib;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Discover the library.
    driver.getFileSync2(a);

    // There is no library which `b` is a part of, so `A` is unresolved.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
    errors
      65 +1 CREATION_WITH_NON_TYPE
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #0
''');
  }

  test_partOfName_getResolvedUnit_noLibrary() async {
    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of my.lib;
var a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // There is no library which `b` is a part of, so `A` is unresolved.
    collector.getResolvedUnit('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
    errors
      63 +1 CREATION_WITH_NON_TYPE
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #0
''');
  }

  test_partOfName_getUnitElement_afterLibrary() async {
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // Process `a` so that we know that it's a library for `b`.
    collector.getResolvedUnit('A1', a);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/hidden/a.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getResolvedUnit A1
  ResolvedUnitResult #0
''');

    // We know that `a` is the library for `b`.
    collector.getUnitElement('B1', b);
    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getUnitElement B1
  path: /home/test/lib/b.dart
  uri: package:test/b.dart
  flags: isPart
  enclosing: #F0
''');
  }

  test_partOfName_getUnitElement_beforeLibrary_addedFiles() async {
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // We discover all added files are maybe libraries.
    driver.addFile2(a);
    driver.addFile2(b);

    // Because `a` is added, we know how to analyze `b`.
    collector.getUnitElement('B1', b);
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/hidden/a.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
[future] getUnitElement B1
  path: /home/test/lib/b.dart
  uri: package:test/b.dart
  flags: isPart
  enclosing: #F0
''');
  }

  test_partOfName_getUnitElement_noLibrary() async {
    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // We don't know the library for `b`.
    // So, we treat it as its own library.
    collector.getUnitElement('B1', b);
    await assertEventsText(collector, r'''
[status] working
[status] idle
[future] getUnitElement B1
  path: /home/test/lib/b.dart
  uri: package:test/b.dart
  flags: isPart
  enclosing: <null>
''');
  }

  test_partOfName_results_afterLibrary() async {
    // Note, we put the library into a different directory.
    // Otherwise we will discover it.
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // The order does not matter.
    // It used to matter, but not anymore.
    driver.addFile2(a);
    driver.addFile2(b);

    // We discover all added libraries.
    // So, we know that `a` is the library of `b`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/hidden/a.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
''');
  }

  test_partOfName_results_beforeLibrary() async {
    // Note, we put the library into a different directory.
    // Otherwise we will discover it.
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // The order does not matter.
    // It used to matter, but not anymore.
    driver.addFile2(b);
    driver.addFile2(a);

    // We discover all added libraries.
    // So, we know that `a` is the library of `b`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
''');
  }

  test_partOfName_results_beforeLibrary_priority() async {
    // Note, we put the library into a different directory.
    // Otherwise we will discover it.
    var a = newFile('$testPackageLibPath/hidden/a.dart', r'''
// @dart = 3.4
// preEnhancedParts
library a;
part '../b.dart';
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // The order does not matter.
    // It used to matter, but not anymore.
    driver.addFile2(b);
    driver.addFile2(a);
    driver.priorityFiles2 = [b];

    // We discover all added libraries.
    // So, we know that `a` is the library of `b`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/hidden/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/hidden/a.dart
    uri: package:test/hidden/a.dart
    flags: exists isLibrary
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
[status] idle
''');
  }

  test_partOfName_results_noLibrary() async {
    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(b);

    // There is no library for `b`.
    // So, we analyze `b` as its own library.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[operation] reuseLinkedBundle
  package:test/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
    errors
      60 +1 CREATION_WITH_NON_TYPE
[status] idle
''');
  }

  test_partOfName_results_noLibrary_priority() async {
    var b = newFile('$testPackageLibPath/b.dart', r'''
// @dart = 3.4
// preEnhancedParts
part of a;
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(b);
    driver.priorityFiles2 = [b];

    // There is no library for `b`.
    // So, we analyze `b` as its own library.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isPart
    errors
      60 +1 CREATION_WITH_NON_TYPE
[status] idle
''');
  }

  test_priorities_changed_importing_rest() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
import 'c.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
class B {}
''');

    var c = newFile('$testPackageLibPath/c.dart', r'''
import 'b.dart';
''');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);
    driver.addFile2(c);

    // Discard results so far.
    await collector.nextStatusIdle();
    collector.take();

    modifyFile2(b, r'''
class B2 {}
''');
    driver.changeFile2(b);

    // We analyze `b` first, because it was changed.
    // Then we analyze `c`, because it imports `b`.
    // Then we analyze `a`, because it also affected.
    // Note, there is no specific rule that says when `a` is analyzed.
    configuration.withStreamResolvedUnitResults = false;
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[operation] analyzeFile
  file: /home/test/lib/c.dart
  library: /home/test/lib/c.dart
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
''');
  }

  test_priorities_changed_importing_withErrors_rest() async {
    // Note, is affected by `b`, but does not import it.
    var a = newFile('$testPackageLibPath/a.dart', r'''
export 'b.dart';
''');

    // We will change this file.
    var b = newFile('$testPackageLibPath/b.dart', r'''
class B {}
''');

    // Note, does not import `b` directly.
    var c = newFile('$testPackageLibPath/c.dart', r'''
import 'a.dart';
class C extends X {}
''');

    // Note, does import `b`.
    var d = newFile('$testPackageLibPath/d.dart', r'''
import 'b.dart';
''');

    var driver = driverFor(a);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);
    driver.addFile2(c);
    driver.addFile2(d);

    // Discard results so far.
    await collector.nextStatusIdle();
    collector.take();

    modifyFile2(b, r'''
class B2 {}
''');
    driver.changeFile2(b);

    // We analyze `b` first, because it was changed.
    // The we analyze `d` because it import `b`.
    // Then we analyze `c` because it has errors.
    // Then we analyze `a` because it is affected.
    // For `a` because it just exports, there are no special rules.
    configuration.withStreamResolvedUnitResults = false;
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[operation] analyzeFile
  file: /home/test/lib/d.dart
  library: /home/test/lib/d.dart
[operation] analyzeFile
  file: /home/test/lib/c.dart
  library: /home/test/lib/c.dart
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[status] idle
''');
  }

  test_priorities_changedAll() async {
    // Make sure that `test2` is its own analysis context.
    var test1Path = '$workspaceRootPath/test1';
    writePackageConfig(
      test1Path,
      PackageConfigFileBuilder()..add(name: 'test1', rootPath: test1Path),
    );

    // Make sure that `test2` is its own analysis context.
    var test2Path = '$workspaceRootPath/test2';
    writePackageConfig(
      test2Path,
      PackageConfigFileBuilder()..add(name: 'test2', rootPath: test2Path),
    );

    // `b` imports `a`, so `b` is reanalyzed when `a` API changes.
    var a = newFile('$test1Path/lib/a.dart', 'class A {}');
    var b = newFile('$test1Path/lib/b.dart', "import 'a.dart';");

    // `d` imports `c`, so `d` is reanalyzed when `b` API changes.
    var c = newFile('$test2Path/lib/c.dart', 'class C {}');
    var d = newFile('$test2Path/lib/d.dart', "import 'c.dart';");

    var collector = DriverEventCollector.forCollection(
      analysisContextCollection,
    );

    var driver1 = driverFor(a);
    var driver2 = driverFor(c);

    // Ensure that we actually have two separate analysis contexts.
    expect(driver1, isNot(same(driver2)));

    // Subscribe for analysis.
    driver1.addFile2(a);
    driver1.addFile2(b);
    driver2.addFile2(c);
    driver2.addFile2(d);

    // Discard results so far.
    await collector.nextStatusIdle();
    collector.take();

    // Change `a` and `c` in a way that changed their API signatures.
    modifyFile2(a, 'class A2 {}');
    modifyFile2(c, 'class C2 {}');
    driver1.changeFile2(a);
    driver2.changeFile2(c);

    // Note, `a` and `c` analyzed first, because they were changed.
    // Even though they are in different drivers.
    configuration.withStreamResolvedUnitResults = false;
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test1/lib/a.dart
  library: /home/test1/lib/a.dart
[operation] analyzeFile
  file: /home/test2/lib/c.dart
  library: /home/test2/lib/c.dart
[operation] analyzeFile
  file: /home/test1/lib/b.dart
  library: /home/test1/lib/b.dart
[operation] analyzeFile
  file: /home/test2/lib/d.dart
  library: /home/test2/lib/d.dart
[status] idle
''');
  }

  test_priorities_getResolvedUnit_beforePriority() async {
    // Make sure that `test1` is its own analysis context.
    var test1Path = '$workspaceRootPath/test1';
    writePackageConfig(
      test1Path,
      PackageConfigFileBuilder()..add(name: 'test1', rootPath: test1Path),
    );

    // Make sure that `test2` is its own analysis context.
    var test2Path = '$workspaceRootPath/test2';
    writePackageConfig(
      test2Path,
      PackageConfigFileBuilder()..add(name: 'test2', rootPath: test2Path),
    );

    var a = newFile('$test1Path/lib/a.dart', '');
    var b = newFile('$test2Path/lib/b.dart', '');
    var c = newFile('$test2Path/lib/c.dart', '');

    var collector = DriverEventCollector.forCollection(
      analysisContextCollection,
    );

    var driver1 = driverFor(a);
    var driver2 = driverFor(c);

    // Ensure that we actually have two separate analysis contexts.
    expect(driver1, isNot(same(driver2)));

    // Subscribe for analysis.
    driver1.addFile2(a);
    driver2.addFile2(b);
    driver2.addFile2(c);

    driver1.priorityFiles2 = [a];
    driver2.priorityFiles2 = [c];

    collector.driver = driver2;
    collector.getResolvedUnit('B1', b);

    // We asked for `b`, so it is analyzed.
    // Even if it is not a priority file.
    // Even if it is in the `driver2`.
    configuration.withStreamResolvedUnitResults = false;
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test2/lib/b.dart
  library: /home/test2/lib/b.dart
[operation] analyzeFile
  file: /home/test1/lib/a.dart
  library: /home/test1/lib/a.dart
[operation] reuseLinkedBundle SDK
[operation] analyzeFile
  file: /home/test2/lib/c.dart
  library: /home/test2/lib/c.dart
[status] idle
[future] getResolvedUnit B1
  ResolvedUnitResult #0
    path: /home/test2/lib/b.dart
    uri: package:test2/b.dart
    flags: exists isLibrary
''');
  }

  test_priorities_priority_rest() async {
    // Make sure that `test1` is its own analysis context.
    var test1Path = '$workspaceRootPath/test1';
    writePackageConfig(
      test1Path,
      PackageConfigFileBuilder()..add(name: 'test1', rootPath: test1Path),
    );

    // Make sure that `test2` is its own analysis context.
    var test2Path = '$workspaceRootPath/test2';
    writePackageConfig(
      test2Path,
      PackageConfigFileBuilder()..add(name: 'test2', rootPath: test2Path),
    );

    var a = newFile('$test1Path/lib/a.dart', '');
    var b = newFile('$test1Path/lib/b.dart', '');
    var c = newFile('$test2Path/lib/c.dart', '');
    var d = newFile('$test2Path/lib/d.dart', '');

    var collector = DriverEventCollector.forCollection(
      analysisContextCollection,
    );

    var driver1 = driverFor(a);
    var driver2 = driverFor(c);

    // Ensure that we actually have two separate analysis contexts.
    expect(driver1, isNot(same(driver2)));

    driver1.addFile2(a);
    driver1.addFile2(b);
    driver1.priorityFiles2 = [a];

    driver2.addFile2(c);
    driver2.addFile2(d);
    driver2.priorityFiles2 = [c];

    configuration.withStreamResolvedUnitResults = false;
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test1/lib/a.dart
  library: /home/test1/lib/a.dart
[operation] analyzeFile
  file: /home/test2/lib/c.dart
  library: /home/test2/lib/c.dart
[operation] reuseLinkedBundle SDK
[operation] analyzeFile
  file: /home/test1/lib/b.dart
  library: /home/test1/lib/b.dart
[operation] analyzeFile
  file: /home/test2/lib/d.dart
  library: /home/test2/lib/d.dart
[status] idle
''');
  }

  test_removeFile_addFile() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);

    // Initial analysis.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
''');

    driver.removeFile2(a);
    driver.addFile2(a);

    // The cache key for `a` errors is the same, return from bytes.
    // Note, no analysis.
    await assertEventsText(collector, r'''
[status] working
[operation] reuseLinkedBundle SDK
[operation] reuseLinkedBundle
  package:test/a.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ErrorsResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: isLibrary
[status] idle
''');
  }

  test_removeFile_changeFile_implicitlyAnalyzed() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
import 'b.dart';
final A = B;
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
final B = 0;
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.priorityFiles2 = [a, b];
    driver.addFile2(a);
    driver.addFile2(b);

    configuration.libraryConfiguration.unitConfiguration.variableTypesSelector =
        (result) {
          switch (result.uriStr) {
            case 'package:test/a.dart':
              return [result.findElement2.topVar('A')];
            case 'package:test/b.dart':
              return [result.findElement2.topVar('B')];
            default:
              return [];
          }
        };

    // We have results for both `a` and `b`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedVariableTypes
      A: int
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    selectedVariableTypes
      B: int
[status] idle
''');

    // Remove `b` and send the change notification.
    modifyFile2(b, r'''
final B = 1.2;
''');
    driver.removeFile2(b);
    driver.changeFile2(b);

    // While `b` is not analyzed explicitly, it is analyzed implicitly.
    // We don't get a result for `b`.
    // But the change causes `a` to be reanalyzed.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[operation] reuseLinkedBundle SDK
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    selectedVariableTypes
      A: double
[status] idle
''');
  }

  test_removeFile_changeFile_notAnalyzed() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // We don't analyze `a`, so we get nothing.
    await assertEventsText(collector, r'''
''');

    // Remove `a`, and also change it.
    // Still nothing, we still don't analyze `a`.
    driver.removeFile2(a);
    driver.changeFile2(a);
    await assertEventsText(collector, r'''
[status] working
[status] idle
''');
  }

  test_removeFile_invalidate_importers() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
import 'a.dart';
final a = new A();
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);

    // No errors in `b`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
''');

    // Remove `a`, so `b` is reanalyzed and has an error.
    deleteFile(a.path);
    driver.removeFile2(a);
    await assertEventsText(collector, r'''
[status] working
[operation] reuseLinkedBundle SDK
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
    errors
      7 +8 URI_DOES_NOT_EXIST
      31 +1 CREATION_WITH_NON_TYPE
[status] idle
''');
  }

  test_removeFile_notAbsolutePath() async {
    var driver = driverFor(testFile);
    expect(() {
      driver.removeFile('not_absolute.dart');
    }, throwsArgumentError);
  }

  test_results_order() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
// ignore:unused_import
import 'd.dart';
''');

    var b = newFile('$testPackageLibPath/b.dart', '');

    var c = newFile('$testPackageLibPath/c.dart', r'''
// ignore:unused_import
import 'd.dart';
''');

    var d = newFile('$testPackageLibPath/d.dart', r'''
// ignore:unused_import
import 'b.dart';
''');

    var e = newFile('$testPackageLibPath/e.dart', r'''
// ignore:unused_import
export 'b.dart';
''');

    // This file intentionally has an error.
    var f = newFile('$testPackageLibPath/f.dart', r'''
// ignore:unused_import
import 'e.dart';
class F extends X {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);
    driver.addFile2(c);
    driver.addFile2(d);
    driver.addFile2(e);
    driver.addFile2(f);

    // Initial analysis, all files analyzed in order of adding.
    // Note, `f` has an error.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/c.dart
  library: /home/test/lib/c.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/c.dart
    uri: package:test/c.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/d.dart
  library: /home/test/lib/d.dart
[stream]
  ResolvedUnitResult #3
    path: /home/test/lib/d.dart
    uri: package:test/d.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/e.dart
  library: /home/test/lib/e.dart
[stream]
  ResolvedUnitResult #4
    path: /home/test/lib/e.dart
    uri: package:test/e.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/f.dart
  library: /home/test/lib/f.dart
[stream]
  ResolvedUnitResult #5
    path: /home/test/lib/f.dart
    uri: package:test/f.dart
    flags: exists isLibrary
    errors
      57 +1 EXTENDS_NON_CLASS
[status] idle
''');

    // Update `b` with changing its API signature.
    modifyFile2(b, r'''
class B {}
''');
    driver.changeFile2(b);

    // 1. The changed `b` is the first.
    // 2. Then `d` that imports the changed `b`.
    // 3. Then `f` that has an error (even if it is unrelated).
    // 4. Then the rest, in order of adding.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #6
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/d.dart
  library: /home/test/lib/d.dart
[stream]
  ResolvedUnitResult #7
    path: /home/test/lib/d.dart
    uri: package:test/d.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/f.dart
  library: /home/test/lib/f.dart
[stream]
  ResolvedUnitResult #8
    path: /home/test/lib/f.dart
    uri: package:test/f.dart
    flags: exists isLibrary
    errors
      57 +1 EXTENDS_NON_CLASS
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #9
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/c.dart
  library: /home/test/lib/c.dart
[stream]
  ResolvedUnitResult #10
    path: /home/test/lib/c.dart
    uri: package:test/c.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/e.dart
  library: /home/test/lib/e.dart
[stream]
  ResolvedUnitResult #11
    path: /home/test/lib/e.dart
    uri: package:test/e.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_results_order_allChangedFirst_thenImports() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
class B {}
''');

    var c = newFile('$testPackageLibPath/c.dart', r'''
''');

    var d = newFile('$testPackageLibPath/d.dart', r'''
// ignore:unused_import
import 'a.dart';
''');

    var e = newFile('$testPackageLibPath/e.dart', r'''
// ignore:unused_import
import 'b.dart';
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);
    driver.addFile2(c);
    driver.addFile2(d);
    driver.addFile2(e);

    // Initial analysis, all files analyzed in order of adding.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/c.dart
  library: /home/test/lib/c.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/c.dart
    uri: package:test/c.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/d.dart
  library: /home/test/lib/d.dart
[stream]
  ResolvedUnitResult #3
    path: /home/test/lib/d.dart
    uri: package:test/d.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/e.dart
  library: /home/test/lib/e.dart
[stream]
  ResolvedUnitResult #4
    path: /home/test/lib/e.dart
    uri: package:test/e.dart
    flags: exists isLibrary
[status] idle
''');

    // Change b.dart and then a.dart files.
    modifyFile2(a, r'''
class A2 {}
''');
    modifyFile2(b, r'''
class B2 {}
''');
    driver.changeFile2(b);
    driver.changeFile2(a);

    // First `a` and `b`.
    // Then `d` and `e` because they import `a` and `b`.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #5
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #6
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/d.dart
  library: /home/test/lib/d.dart
[stream]
  ResolvedUnitResult #7
    path: /home/test/lib/d.dart
    uri: package:test/d.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/e.dart
  library: /home/test/lib/e.dart
[stream]
  ResolvedUnitResult #8
    path: /home/test/lib/e.dart
    uri: package:test/e.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_results_removeFile_changeFile() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final v = 0;
''');

    var b = getFile('$testPackageLibPath/b.dart');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);

    // Initial analysis.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
''');

    // Update `a` to have an error.
    modifyFile2(a, r'''
final v = 0
''');

    // It does not matter what we do with `b`, it is not analyzed anyway.
    // But we notify that `a` was changed, so it is analyzed.
    driver.removeFile2(b);
    driver.changeFile2(a);
    await assertEventsText(collector, r'''
[status] working
[operation] reuseLinkedBundle SDK
[operation] reuseLinkedBundle
  package:test/a.dart
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
    errors
      10 +1 EXPECTED_TOKEN
[status] idle
''');
  }

  test_results_skipNotAffected() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    var b = newFile('$testPackageLibPath/b.dart', r'''
class B {}
''');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);

    // Initial analysis.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[stream]
  ResolvedUnitResult #1
    path: /home/test/lib/b.dart
    uri: package:test/b.dart
    flags: exists isLibrary
[status] idle
''');

    // Update `a` and notify.
    modifyFile2(a, r'''
class A2 {}
''');
    driver.changeFile2(a);

    // Only `a` is analyzed, `b` is not affected.
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_schedulerStatus_hasAddedFile() async {
    var a = newFile('$testPackageLibPath/a.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);

    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/a.dart
    uri: package:test/a.dart
    flags: exists isLibrary
[status] idle
''');
  }

  test_schedulerStatus_noAddedFile() async {
    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    // No files, so no status changes.
    await assertEventsText(collector, r'''
''');
  }

  test_status_anyWorkTransitionsToAnalyzing() async {
    var a = newFile('$testPackageLibPath/a.dart', '');
    var b = newFile('$testPackageLibPath/b.dart', '');

    var driver = driverFor(testFile);
    var collector = DriverEventCollector(driver);

    driver.addFile2(a);
    driver.addFile2(b);

    // Initial analysis.
    configuration.withStreamResolvedUnitResults = false;
    await assertEventsText(collector, r'''
[status] working
[operation] analyzeFile
  file: /home/test/lib/a.dart
  library: /home/test/lib/a.dart
[operation] analyzeFile
  file: /home/test/lib/b.dart
  library: /home/test/lib/b.dart
[status] idle
''');

    // Any work transitions to analyzing, and back to idle.
    await driver.getFilesReferencingName('X');
    await assertEventsText(collector, r'''
[status] working
[status] idle
''');
  }
}

/// Tracks events reported into the `results` stream, and results of `getXyz`
/// requests. We are interested in relative orders, identity of the objects,
/// absence of duplicate events, etc.
class DriverEventCollector {
  final IdProvider idProvider;
  late AnalysisDriver driver;
  final List<DriverEvent> _events = [];
  final List<_PendingFuture> _pendingFutures = [];
  final List<Completer<void>> _statusIdleCompleters = [];

  DriverEventCollector(this.driver, {IdProvider? idProvider})
    : idProvider = idProvider ?? IdProvider() {
    _listenSchedulerEvents(driver.scheduler);
  }

  DriverEventCollector.forCollection(
    AnalysisContextCollectionImpl collection, {
    IdProvider? idProvider,
  }) : idProvider = idProvider ?? IdProvider() {
    _listenSchedulerEvents(collection.scheduler);
  }

  Future<void> awaitPendingFutures() async {
    var futures = _pendingFutures.toList();
    _pendingFutures.clear();
    for (var future in futures) {
      await future.resolve();
    }
  }

  void getCachedResolvedUnit(String name, File file) {
    var value = driver.getCachedResolvedUnit2(file);
    _events.add(GetCachedResolvedUnitEvent(name: name, result: value));
  }

  void getErrors(String name, File file) {
    _addPending(
      driver.getErrors(file.path),
      (value) => GetErrorsEvent(name: name, result: value),
    );
  }

  void getIndex(String name, File file) {
    _addPending(
      driver.getIndex(file.path),
      (value) => GetIndexEvent(name: name, result: value),
    );
  }

  void getLibraryByUri(String name, String uriStr) {
    _addPending(
      driver.getLibraryByUri(uriStr),
      (value) => GetLibraryByUriEvent(name: name, result: value),
    );
  }

  void getResolvedLibrary(String name, File file) {
    _addPending(
      driver.getResolvedLibrary(file.path),
      (value) => GetResolvedLibraryEvent(name: name, result: value),
    );
  }

  void getResolvedLibraryByUri(String name, Uri uri) {
    _addPending(
      driver.getResolvedLibraryByUri(uri),
      (value) => GetResolvedLibraryByUriEvent(name: name, result: value),
    );
  }

  void getResolvedUnit(
    String name,
    File file, {
    bool sendCachedToStream = false,
  }) {
    _addPending(
      driver.getResolvedUnit(file.path, sendCachedToStream: sendCachedToStream),
      (value) => GetResolvedUnitEvent(name: name, result: value),
    );
  }

  void getUnitElement(String name, File file) {
    _addPending(
      driver.getUnitElement2(file),
      (value) => GetUnitElementEvent(name: name, result: value),
    );
  }

  Future<void> nextStatusIdle() {
    var completer = Completer<void>();
    _statusIdleCompleters.add(completer);
    return completer.future;
  }

  List<DriverEvent> take() {
    if (_pendingFutures.isNotEmpty) {
      throw StateError('Some futures are not complete.');
    }
    var result = _events.toList();
    _events.clear();
    return result;
  }

  void _addPending<T>(
    Future<T> future,
    DriverEvent Function(T value) createEvent,
  ) {
    _pendingFutures.add(
      _PendingFuture<T>(
        future: future,
        createEvent: createEvent,
        eventsList: _events,
      ),
    );
  }

  void _listenSchedulerEvents(AnalysisDriverScheduler scheduler) {
    scheduler.eventsBroadcast.listen((event) {
      switch (event) {
        case AnalysisStatus():
          _events.add(SchedulerStatusEvent(event));
          if (event.isIdle) {
            _statusIdleCompleters.completeAll();
            _statusIdleCompleters.clear();
          }
        case driver_events.AnalyzeFile():
        case driver_events.AnalyzedLibrary():
        case driver_events.GetErrorsFromBytes():
        case driver_events.LinkLibraryCycle():
        case driver_events.CheckLibraryDiagnosticsRequirements():
        case driver_events.CheckLinkedBundleRequirements():
        case driver_events.ReuseLinkedBundle():
        case ErrorsResult():
        case ResolvedUnitResult():
          _events.add(ResultStreamEvent(object: event));
      }
    });
  }
}

@reflectiveTest
class FineAnalysisDriverTest extends PubPackageResolutionTest
    with _EventsMixin {
  @override
  bool get retainDataForTesting => true;

  @override
  void setUp() {
    withFineDependencies = true;
    super.setUp();
    registerLintRules();
    useEmptyByteStore();
  }

  @override
  Future<void> tearDown() async {
    testFineAfterLibraryAnalyzerHook = null;
    return super.tearDown();
  }

  test_dependency_class_add_allSubtypes_final_related() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      state.singleUnit.scopeClassElement('A').allSubtypes;
    });

    await _runChangeScenarioTA(
      initialA: r'''
final class A {}
class B extends A {}
class C {}
void foo() {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M4
        interface: #M5
    declaredFunctions
      foo: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M2
      C: #M4
      foo: #M6
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M8
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M7
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allSubtypes: #M2
[status] idle
''',
      updatedA: r'''
final class A {}
class B extends A {}
class C {}
void foo() {}
class D extends A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M4
        interface: #M5
      D: #M10
        interface: #M11
    declaredFunctions
      foo: #M6
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M2
      C: #M4
      D: #M10
      foo: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    childrenPropertyName: allSubtypes
    expectedIds: #M2
    actualIds: #M2 #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allSubtypes: #M2 #M10
[status] idle
''',
    );
  }

  test_dependency_class_add_allSubtypes_final_unrelated() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      state.singleUnit.scopeClassElement('A').allSubtypes;
    });

    await _runChangeScenarioTA(
      initialA: r'''
final class A {}
class B extends A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allSubtypes: #M2
[status] idle
''',
      updatedA: r'''
final class A {}
class B extends A {}
class C {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M2
      C: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_class_add_directSubtypesOfSealed_related() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeClassElement('A');
      A.directSubtypesOfSealed;
    });

    await _runChangeScenarioTA(
      initialA: r'''
sealed class A {}
class B extends A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            directSubtypesOfSealed: #M2
[status] idle
''',
      updatedA: r'''
sealed class A {}
class B extends A {}
class C extends A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M2
      C: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    childrenPropertyName: directSubtypesOfSealed
    expectedIds: #M2
    actualIds: #M2 #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            directSubtypesOfSealed: #M2 #M7
[status] idle
''',
    );
  }

  test_dependency_class_add_directSubtypesOfSealed_related_switch() async {
    configuration
      ..withGetErrorsEvents = true
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
sealed class A {}
class B extends A {}
''',
      testCode: r'''
import 'a.dart';

void f(A a) {
  switch (a) {
    case B():
      0;
  }
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
          B: #M2
          B=: <null>
        reExportDeprecatedOnly
          A: false
          B: false
        interfaces
          A
            allSubtypes: <null>
            directSubtypesOfSealed: #M2
          B
            allSubtypes: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
sealed class A {}
class B extends A {}
class C extends A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M8
        interface: #M9
    exportMapId: #M10
    exportMap
      A: #M0
      B: #M2
      C: #M8
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    childrenPropertyName: directSubtypesOfSealed
    expectedIds: #M2
    actualIds: #M2 #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M10
        exportMap
          A: #M0
          A=: <null>
          B: #M2
          B=: <null>
        reExportDeprecatedOnly
          A: false
          B: false
        interfaces
          A
            allSubtypes: <null>
            directSubtypesOfSealed: #M2 #M8
          B
            allSubtypes: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      34 +6 NON_EXHAUSTIVE_SWITCH_STATEMENT
''',
    );
  }

  test_dependency_class_add_directSubtypesOfSealed_unrelated() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeClassElement('A');
      A.directSubtypesOfSealed;
    });

    await _runChangeScenarioTA(
      initialA: r'''
sealed class A {}
class B extends A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            directSubtypesOfSealed: #M2
[status] idle
''',
      updatedA: r'''
sealed class A {}
class B extends A {}
class C {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M2
      C: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_class_instanceField_add_hasNonFinalField() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.hasNonFinalField;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            hasNonFinalField: false
[status] idle
''',
      updatedA: r'''
class A {
  var foo = 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M5
        declaredGetters
          foo: #M6
        declaredSetters
          foo=: #M7
        interface: #M8
          map
            foo: #M6
            foo=: #M7
          implemented
            foo: #M6
            foo=: #M7
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceHasNonFinalFieldMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expected: false
    actual: true
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            hasNonFinalField: true
[status] idle
''',
    );
  }

  test_dependency_class_instanceField_add_hasNonFinalField_ofSuper() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
class B extends A {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M3
        interface: #M4
    exportMapId: #M5
    exportMap
      B: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M1
            hasNonFinalField: false
            requestedConstructors
              new: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M7
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: []
        interfaces
          A
            allDeclaredConstructors: #M6
            allInheritedConstructors: []
            requestedConstructors
              new: #M6
[status] idle
''',
      updatedA: r'''
class A {
  var foo = 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M8
        declaredGetters
          foo: #M9
        declaredSetters
          foo=: #M10
        interface: #M11
          map
            foo: #M9
            foo=: #M10
          implemented
            foo: #M9
            foo=: #M10
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expectedId: #M1
    actualId: #M11
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M3
        interface: #M12
          map
            foo: #M9
            foo=: #M10
          implemented
            foo: #M9
            foo=: #M10
          superImplemented
            [0]
              foo: #M9
              foo=: #M10
          inherited
            foo: #M9
            foo=: #M10
    exportMapId: #M5
    exportMap
      B: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M11
            hasNonFinalField: true
            requestedConstructors
              new: #M6
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: getters
    expectedIds: []
    actualIds: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M7
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
            allDeclaredGetters: #M9
            allDeclaredSetters: #M10
            allDeclaredMethods: []
        interfaces
          A
            allDeclaredConstructors: #M6
            allInheritedConstructors: []
            requestedConstructors
              new: #M6
[status] idle
''',
    );
  }

  test_dependency_class_instanceField_add_hasNonFinalField_ofSuper_private() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
class B extends A {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M3
        interface: #M4
    exportMapId: #M5
    exportMap
      B: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M1
            hasNonFinalField: false
            requestedConstructors
              new: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M7
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: []
        interfaces
          A
            allDeclaredConstructors: #M6
            allInheritedConstructors: []
            requestedConstructors
              new: #M6
[status] idle
''',
      updatedA: r'''
class A {
  var _foo = 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          _foo: #M8
        declaredGetters
          _foo: #M9
        declaredSetters
          _foo=: #M10
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceHasNonFinalFieldMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expected: false
    actual: true
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M3
        interface: #M4
    exportMapId: #M5
    exportMap
      B: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M1
            hasNonFinalField: true
            requestedConstructors
              new: #M6
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: getters
    expectedIds: []
    actualIds: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M7
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: #M9
            allDeclaredSetters: #M10
            allDeclaredMethods: []
        interfaces
          A
            allDeclaredConstructors: #M6
            allInheritedConstructors: []
            requestedConstructors
              new: #M6
[status] idle
''',
    );
  }

  test_dependency_class_instanceField_change_getField() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getField('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  final int foo = 0;
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  final double foo = 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M7
        declaredGetters
          foo: #M8
        interface: #M9
          map
            foo: #M8
          implemented
            foo: #M8
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M7
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceField_change_getField_other() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getField('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  final int foo = 0;
  final int bar = 0;
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredGetters
          bar: #M3
          foo: #M4
        interface: #M5
          map
            bar: #M3
            foo: #M4
          implemented
            bar: #M3
            foo: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  final int foo = 0;
  final double bar = 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M9
          foo: #M2
        declaredGetters
          bar: #M10
          foo: #M4
        interface: #M11
          map
            bar: #M10
            foo: #M4
          implemented
            bar: #M10
            foo: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceGetter_add_extended() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
class B extends A {}
''',
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M5
        interface: #M6
          map
            foo: #M2
          implemented
            foo: #M2
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M7
    exportMap
      B: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M3
            hasNonFinalField: false
            requestedConstructors
              new: #M8
[status] idle
[future] getLibraryByUri T1
  library
    classes
      class B
        supertype: A
        constructors
          synthetic new
''',
      updatedA: r'''
class A {
  int get foo => 0;
  int get bar => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M9
          foo: #M1
        declaredGetters
          bar: #M10
          foo: #M2
        interface: #M11
          map
            bar: #M10
            foo: #M2
          implemented
            bar: #M10
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expectedId: #M3
    actualId: #M11
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M5
        interface: #M12
          map
            bar: #M10
            foo: #M2
          implemented
            bar: #M10
            foo: #M2
          superImplemented
            [0]
              bar: #M10
              foo: #M2
          inherited
            bar: #M10
            foo: #M2
    exportMapId: #M7
    exportMap
      B: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M11
            hasNonFinalField: false
            requestedConstructors
              new: #M8
[status] idle
[future] getLibraryByUri T2
  library
    classes
      class B
        supertype: A
        constructors
          synthetic new
''',
    );
  }

  test_dependency_class_instanceGetter_add_getGetters() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getters;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  int get foo => 0;
  int get bar => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M7
          foo: #M1
        declaredGetters
          bar: #M8
          foo: #M2
        interface: #M9
          map
            bar: #M8
            foo: #M2
          implemented
            bar: #M8
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: getters
    expectedIds: #M2
    actualIds: #M2 #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: #M2 #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceGetter_change_classExtends_hasSetter() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo {}
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';

class B extends A {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        interface: #M4
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M6
        interface: #M7
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
          superImplemented
            [0]
              foo: #M2
              foo=: #M3
          inherited
            foo: #M2
            foo=: #M3
    exportMapId: #M8
    exportMap
      B: #M6
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M4
            hasNonFinalField: false
            requestedConstructors
              new: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M10
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            allDeclaredGetters: #M2
            allDeclaredSetters: #M3
            allDeclaredMethods: []
        interfaces
          A
            allDeclaredConstructors: #M9
            allInheritedConstructors: []
            requestedConstructors
              new: #M9
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  double get foo {}
  set foo(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M11
        declaredGetters
          foo: #M12
        declaredSetters
          foo=: #M3
        interface: #M13
          map
            foo: #M12
            foo=: #M3
          implemented
            foo: #M12
            foo=: #M3
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expectedId: #M4
    actualId: #M13
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M6
        interface: #M14
          map
            foo: #M12
            foo=: #M3
          implemented
            foo: #M12
            foo=: #M3
          superImplemented
            [0]
              foo: #M12
              foo=: #M3
          inherited
            foo: #M12
            foo=: #M3
    exportMapId: #M8
    exportMap
      B: #M6
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M13
            hasNonFinalField: false
            requestedConstructors
              new: #M9
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M11
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M10
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M11
            allDeclaredGetters: #M12
            allDeclaredSetters: #M3
            allDeclaredMethods: []
        interfaces
          A
            allDeclaredConstructors: #M9
            allInheritedConstructors: []
            requestedConstructors
              new: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceGetter_change_getField_getter() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getField('foo')?.getter;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  double get foo => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M7
        declaredGetters
          foo: #M8
        interface: #M9
          map
            foo: #M8
          implemented
            foo: #M8
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M7
            requestedDeclaredGetters
              foo: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceGetter_change_getGetter() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getGetter('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  double get foo => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M7
        declaredGetters
          foo: #M8
        interface: #M9
          map
            foo: #M8
          implemented
            foo: #M8
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M2
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceGetter_change_getGetter_other() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getGetter('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo => 0;
  int get bar => 0;
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredGetters
          bar: #M3
          foo: #M4
        interface: #M5
          map
            bar: #M3
            foo: #M4
          implemented
            bar: #M3
            foo: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  int get foo => 0;
  double get bar => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M9
          foo: #M2
        declaredGetters
          bar: #M10
          foo: #M4
        interface: #M11
          map
            bar: #M10
            foo: #M4
          implemented
            bar: #M10
            foo: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  double get foo => 1.2;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M8
        declaredGetters
          foo: #M9
        interface: #M10
          map
            foo: #M9
          implemented
            foo: #M9
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
        interfaces
          A
            methods
              foo: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceGetter_change_invoked_hasNamedConstructor_hasInstanceSetter() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.foo(int _);
  int get foo {}
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  var a = A.foo(0);
  a.foo;
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        declaredConstructors
          foo: #M4
        interface: #M5
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            requestedConstructors
              foo: #M4
            methods
              foo: #M2
              foo=: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.foo(int _);
  double get foo {}
  set foo(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M10
        declaredGetters
          foo: #M11
        declaredSetters
          foo=: #M3
        declaredConstructors
          foo: #M4
        interface: #M12
          map
            foo: #M11
            foo=: #M3
          implemented
            foo: #M11
            foo=: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M10
        interfaces
          A
            requestedConstructors
              foo: #M4
            methods
              foo: #M11
              foo=: #M3
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo => 0;
  int get bar => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredGetters
          bar: #M3
          foo: #M4
        interface: #M5
          map
            bar: #M3
            foo: #M4
          implemented
            bar: #M3
            foo: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
        interfaces
          A
            methods
              foo: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  int get foo => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M2
        declaredGetters
          foo: #M4
        interface: #M10
          map
            foo: #M4
          implemented
            foo: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceGetter_inherited_fromGeneric_extends_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  T get foo {}
}

class B extends A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M4
        interface: #M5
          map
            foo: #M2
          implemented
            foo: #M2
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  T get foo {}
}

class B extends A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M10
        interface: #M11
          map
            foo: #M2
          implemented
            foo: #M2
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M13
    exportMapId: #M14
    exportMap
      f: #M13
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo: #M2
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceGetter_inherited_fromGeneric_implements_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  T get foo {}
}

class B implements A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M4
        interface: #M5
          map
            foo: #M2
          inherited
            foo: #M2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  T get foo {}
}

class B implements A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M10
        interface: #M11
          map
            foo: #M2
          inherited
            foo: #M2
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M13
    exportMapId: #M14
    exportMap
      f: #M13
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo: #M2
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceGetter_inherited_fromGeneric_with_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  T get foo {}
}

class B with A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M4
        interface: #M5
          map
            foo: #M2
          implemented
            foo: #M2
          superImplemented
            [1]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  T get foo {}
}

class B with A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M10
        interface: #M11
          map
            foo: #M2
          implemented
            foo: #M2
          superImplemented
            [1]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M13
    exportMapId: #M14
    exportMap
      f: #M13
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo: #M2
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceGetter_inherited_private() async {
    // Test that there is a dependency between `f()` and `A._foo`.
    // So, that we re-analyze `f()` body when `A._foo` changes.
    // Currently this dependency is implicit: we analyze the whole library
    // when any of its files changes.
    configuration.withStreamResolvedUnitResults = false;

    newFile(testFile.path, r'''
class A {
  int get _foo => 0;
}

class B extends A {}

void f (B b) {
  b._foo.isEven;
}
''');

    await _runChangeScenario(
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          _foo: #M1
        declaredGetters
          _foo: #M2
        interface: #M3
      B: #M4
        interface: #M5
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M4
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updateFiles: () {
        modifyFile2(testFile, r'''
class A {
  String get _foo => '';
}

class B extends A {}

void f (B b) {
  b._foo.isEven;
}
''');
        return [testFile];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          _foo: #M8
        declaredGetters
          _foo: #M9
        interface: #M3
      B: #M4
        interface: #M5
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M4
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      84 +6 UNDEFINED_GETTER
''',
    );
  }

  test_dependency_class_instanceMethod_add_classExtends() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
class B extends A {}
''',
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M4
        interface: #M5
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M6
    exportMap
      B: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M2
            hasNonFinalField: false
            requestedConstructors
              new: #M7
[status] idle
[future] getLibraryByUri T1
  library
    classes
      class B
        supertype: A
        constructors
          synthetic new
''',
      updatedA: r'''
class A {
  void foo() {}
  void bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M8
          foo: #M1
        interface: #M9
          map
            bar: #M8
            foo: #M1
          implemented
            bar: #M8
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expectedId: #M2
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M4
        interface: #M10
          map
            bar: #M8
            foo: #M1
          implemented
            bar: #M8
            foo: #M1
          superImplemented
            [0]
              bar: #M8
              foo: #M1
          inherited
            bar: #M8
            foo: #M1
    exportMapId: #M6
    exportMap
      B: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M9
            hasNonFinalField: false
            requestedConstructors
              new: #M7
[status] idle
[future] getLibraryByUri T2
  library
    classes
      class B
        supertype: A
        constructors
          synthetic new
''',
    );
  }

  test_dependency_class_instanceMethod_add_getMethods() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.methods;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredMethods: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  void foo() {}
  void bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M6
          foo: #M1
        interface: #M7
          map
            bar: #M6
            foo: #M1
          implemented
            bar: #M6
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: methods
    expectedIds: #M1
    actualIds: #M1 #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredMethods: #M1 #M6
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceMethod_add_implementedInterface_warmInterfaceType() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
abstract class A {}
''',
      testCode: r'''
import 'a.dart';
class B implements A {}
''',
      operation: _FineOperationTestFileGetErrors(),
      beforeInitialOperation: (driver) async {
        var a = await driver.getLibraryByUri('package:test/a.dart');
        var A = (a as LibraryElementResult).element.getClass('A')!;
        // InterfaceTypeImpl caches getters, setters, methods.
        // And `thisType` is reused for all instantiations of `A`.
        // But we still want to have requirements recorded.
        A.thisType.getters;
        A.thisType.setters;
        A.thisType.methods;
      },
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M3
        interface: #M4
    exportMapId: #M5
    exportMap
      B: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M1
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M6
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: []
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
abstract class A {
  int get foo;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M7
        declaredGetters
          foo: #M8
        interface: #M9
          map
            foo: #M8
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expectedId: #M1
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M3
        interface: #M10
          map
            foo: #M8
          inherited
            foo: #M8
    exportMapId: #M5
    exportMap
      B: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M9
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: getters
    expectedIds: []
    actualIds: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M6
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M7
            allDeclaredGetters: #M8
            allDeclaredSetters: []
            allDeclaredMethods: []
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      23 +1 NON_ABSTRACT_CLASS_INHERITS_ABSTRACT_MEMBER
''',
    );
  }

  test_dependency_class_instanceMethod_add_invoked() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        interfaces
          A
            methods
              foo: <null>
              foo=: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      35 +3 UNDEFINED_METHOD
''',
      updatedA: r'''
class A {
  int foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M6
        interface: #M7
          map
            foo: #M6
          implemented
            foo: #M6
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: <null>
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M6
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_change_classExtends() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  void foo() {}
  void bar() {}
}
''',
      testCode: r'''
import 'a.dart';

class B extends A {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M1
          foo: #M2
        interface: #M3
          map
            bar: #M1
            foo: #M2
          implemented
            bar: #M1
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M5
        interface: #M6
          map
            bar: #M1
            foo: #M2
          implemented
            bar: #M1
            foo: #M2
          superImplemented
            [0]
              bar: #M1
              foo: #M2
          inherited
            bar: #M1
            foo: #M2
    exportMapId: #M7
    exportMap
      B: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M3
            hasNonFinalField: false
            requestedConstructors
              new: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M9
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: #M2 #M1
        interfaces
          A
            allDeclaredConstructors: #M8
            allInheritedConstructors: []
            requestedConstructors
              new: #M8
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  int foo() {}
  void bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M1
          foo: #M10
        interface: #M11
          map
            bar: #M1
            foo: #M10
          implemented
            bar: #M1
            foo: #M10
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expectedId: #M3
    actualId: #M11
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M5
        interface: #M12
          map
            bar: #M1
            foo: #M10
          implemented
            bar: #M1
            foo: #M10
          superImplemented
            [0]
              bar: #M1
              foo: #M10
          inherited
            bar: #M1
            foo: #M10
    exportMapId: #M7
    exportMap
      B: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M11
            hasNonFinalField: false
            requestedConstructors
              new: #M8
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: methods
    expectedIds: #M2 #M1
    actualIds: #M10 #M1
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M9
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: #M10 #M1
        interfaces
          A
            allDeclaredConstructors: #M8
            allInheritedConstructors: []
            requestedConstructors
              new: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_change_classExtends_topMerged() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  void foo() {}
}

class B {
  dynamic foo() {}
}

class C extends A implements B {}
''',
      testCode: r'''
import 'a.dart';

class X extends C {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        declaredMethods
          foo: #M4
        interface: #M5
          map
            foo: #M4
          implemented
            foo: #M4
      C: #M6
        interface: #M7
          map
            foo: #M8
          combinedIds
            [#M1, #M4]: #M8
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M3
      C: #M6
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      X: #M10
        interface: #M11
          map
            foo: #M8
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M8
    exportMapId: #M12
    exportMap
      X: #M10
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M9
        exportMap
          C: #M6
          C=: <null>
        reExportDeprecatedOnly
          C: false
        interfaces
          C
            interfaceId: #M7
            hasNonFinalField: false
            requestedConstructors
              new: #M13
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M14
        exportMapId: #M9
        exportMap
          C: #M6
          C=: <null>
        reExportDeprecatedOnly
          C: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: #M1
          B
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: #M4
          C
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: []
        interfaces
          C
            allDeclaredConstructors: #M13
            allInheritedConstructors: []
            requestedConstructors
              new: #M13
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  int foo() {}
}

class B {
  dynamic foo() {}
}

class C extends A implements B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M15
        interface: #M16
          map
            foo: #M15
          implemented
            foo: #M15
      B: #M3
        declaredMethods
          foo: #M4
        interface: #M5
          map
            foo: #M4
          implemented
            foo: #M4
      C: #M6
        interface: #M17
          map
            foo: #M18
          combinedIds
            [#M15, #M4]: #M18
          implemented
            foo: #M15
          superImplemented
            [0]
              foo: #M15
          inherited
            foo: #M15
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M3
      C: #M6
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: C
    expectedId: #M7
    actualId: #M17
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      X: #M10
        interface: #M19
          map
            foo: #M15
          implemented
            foo: #M15
          superImplemented
            [0]
              foo: #M15
          inherited
            foo: #M15
    exportMapId: #M12
    exportMap
      X: #M10
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M9
        exportMap
          C: #M6
          C=: <null>
        reExportDeprecatedOnly
          C: false
        interfaces
          C
            interfaceId: #M17
            hasNonFinalField: false
            requestedConstructors
              new: #M13
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: methods
    expectedIds: #M1
    actualIds: #M15
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M14
        exportMapId: #M9
        exportMap
          C: #M6
          C=: <null>
        reExportDeprecatedOnly
          C: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: #M15
          B
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: #M4
          C
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: []
        interfaces
          C
            allDeclaredConstructors: #M13
            allInheritedConstructors: []
            requestedConstructors
              new: #M13
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_change_getInterfaceMember() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var B = state.singleUnit.scopeClassElement('B');
      B.getInterfaceMember(Name(null, 'foo'));
    });

    await _runChangeScenarioTA(
      initialA: r'''
abstract class A {
  int foo();
}

abstract class B extends A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          inherited
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
''',
      updatedA: r'''
abstract class A {
  double foo();
}

abstract class B extends A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M8
        interface: #M9
          map
            foo: #M8
      B: #M3
        interface: #M10
          map
            foo: #M8
          inherited
            foo: #M8
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: B
    methodName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M8
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_getMethod() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getMethod('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredMethods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M6
        interface: #M7
          map
            foo: #M6
          implemented
            foo: #M6
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredMethods
              foo: #M6
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceMethod_change_getMethod_other() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getMethod('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int foo() {}
  int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M1
          foo: #M2
        interface: #M3
          map
            bar: #M1
            foo: #M2
          implemented
            bar: #M1
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredMethods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  int foo() {}
  double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M7
          foo: #M2
        interface: #M8
          map
            bar: #M7
            foo: #M2
          implemented
            bar: #M7
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceMethod_change_inheritanceManager_getInheritedMap_shadowed() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var B = state.singleUnit.scopeClassElement('B');
      B.inheritanceManager.getInheritedMap(B);
    });

    await _runChangeScenarioTA(
      initialA: r'''
abstract class A {
  int foo();
}

abstract class B extends A {
  void foo();
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
      B: #M3
        declaredMethods
          foo: #M4
        interface: #M5
          map
            foo: #M4
          inherited
            foo: #M1
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            interfaceId: #M5
[status] idle
''',
      updatedA: r'''
abstract class A {
  double foo();
}

abstract class B extends A {
  void foo();
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M9
        interface: #M10
          map
            foo: #M9
      B: #M3
        declaredMethods
          foo: #M4
        interface: #M11
          map
            foo: #M4
          inherited
            foo: #M9
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: B
    expectedId: #M5
    actualId: #M11
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            interfaceId: #M11
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_inheritanceManager_getInheritedMap_shadowed2() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var C = state.singleUnit.scopeClassElement('C');
      C.inheritanceManager.getInheritedMap(C);
    });

    await _runChangeScenarioTA(
      initialA: r'''
abstract class A {
  int foo();
}

abstract class B extends A {
  void foo();
}

abstract class C extends B {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
      B: #M3
        declaredMethods
          foo: #M4
        interface: #M5
          map
            foo: #M4
          inherited
            foo: #M1
      C: #M6
        interface: #M7
          map
            foo: #M4
          inherited
            foo: #M4
    exportMapId: #M8
    exportMap
      A: #M0
      B: #M3
      C: #M6
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M9
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M8
        exportMap
          C: #M6
          C=: <null>
        reExportDeprecatedOnly
          C: false
        interfaces
          C
            interfaceId: #M7
[status] idle
''',
      updatedA: r'''
abstract class A {
  double foo();
}

abstract class B extends A {
  void foo();
}

abstract class C extends B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M11
        interface: #M12
          map
            foo: #M11
      B: #M3
        declaredMethods
          foo: #M4
        interface: #M13
          map
            foo: #M4
          inherited
            foo: #M11
      C: #M6
        interface: #M7
          map
            foo: #M4
          inherited
            foo: #M4
    exportMapId: #M8
    exportMap
      A: #M0
      B: #M3
      C: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M7
        interface: #M8
          map
            foo: #M7
          implemented
            foo: #M7
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M7
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_change_invoked_hasNamedConstructor() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.foo(int _);
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo(0).foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        declaredConstructors
          foo: #M2
        interface: #M3
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              foo: #M2
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.foo(int _);
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M8
        declaredConstructors
          foo: #M2
        interface: #M9
          map
            foo: #M8
          implemented
            foo: #M8
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              foo: #M2
            methods
              foo: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_change_invoked_private() async {
    configuration.withStreamResolvedUnitResults = false;

    newFile(testFile.path, r'''
class A {
  int _foo() => 0;
}

class B extends A {}

void f(B b) {
  b._foo();
}
''');

    // Note:
    // 1. No `_foo` in `B`, even though it is in the same library.
    // 2. No dependency of `test.dart` on `_foo` through `B`.
    // However: we reanalyze `test.dart` when we change it, because we
    // always analyze the whole library when one of its files changes.
    // So, we don't need a separate dependency on `_foo`.
    await _runChangeScenario(
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          _foo: #M1
        interface: #M2
      B: #M3
        interface: #M4
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M3
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updateFiles: () {
        modifyFile2(testFile, r'''
class A {
  double _foo() => 0;
}

class B extends A {}

void f(B b) {
  b._foo();
}
''');
        return [testFile];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredMethods
          _foo: #M7
        interface: #M2
      B: #M3
        interface: #M4
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M3
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_change_invoked_private2() async {
    configuration.withStreamResolvedUnitResults = false;

    newFile('$testPackageLibPath/a.dart', r'''
import 'test.dart';

class B extends A {}
''');

    newFile(testFile.path, r'''
import 'a.dart';

class A {
  void _foo() {}
}

void f(B b) {
  b._foo();
}
''');

    // Note:
    // 1. No `_foo` in `B`.
    // 2. No dependency of `test.dart` on `_foo` through `B`.
    // However: we reanalyze `test.dart` when we change it, because we
    // always analyze the whole library when one of its files changes.
    // So, we don't need a separate dependency on `_foo`.
    await _runChangeScenario(
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      B: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      B: #M0
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M3
        declaredMethods
          _foo: #M4
        interface: #M5
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      A: #M3
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M2
        exportMap
          B: #M0
          B=: <null>
        reExportDeprecatedOnly
          B: false
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updateFiles: () {
        modifyFile2(testFile, r'''
import 'a.dart';

class A {
  void _bar() {}
}

void f(B b) {
  b._foo();
}
''');
        return [testFile];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      B: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      B: #M0
  package:test/test.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M3
        declaredMethods
          _bar: #M9
        interface: #M5
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      A: #M3
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M2
        exportMap
          B: #M0
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          B
            requestedDeclaredGetters
              _foo: <null>
            requestedDeclaredMethods
              _foo: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      66 +4 UNDEFINED_METHOD
      35 +4 UNUSED_ELEMENT
''',
    );
  }

  test_dependency_class_instanceMethod_change_lookUpMethod() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeClassElement('A');
      A.thisType.lookUpMethod('foo', A.library);
    });

    await _runChangeScenarioTA(
      initialA: r'''
abstract class A {
  int foo();
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M1
[status] idle
''',
      updatedA: r'''
abstract class A {
  double foo();
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M6
        interface: #M7
          map
            foo: #M6
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M6
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_lookUpMethod_concrete() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeClassElement('A');
      A.thisType.lookUpMethod('foo', A.library, concrete: true);
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            implementedMethods
              foo: #M1
[status] idle
''',
      updatedA: r'''
class A {
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M6
        interface: #M7
          map
            foo: #M6
          implemented
            foo: #M6
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  implementedMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            implementedMethods
              foo: #M6
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_lookUpMethod_concrete_makeAbstract() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeClassElement('A');
      A.thisType.lookUpMethod('foo', A.library, concrete: true);
    });

    await _runChangeScenarioTA(
      initialA: r'''
abstract class A {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            implementedMethods
              foo: #M1
[status] idle
''',
      updatedA: r'''
abstract class A {
  void foo();
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M6
        interface: #M7
          map
            foo: #M6
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  implementedMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            implementedMethods
              foo: <null>
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_lookUpMethod_concrete_makeConcrete() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeClassElement('A');
      A.thisType.lookUpMethod('foo', A.library, concrete: true);
    });

    await _runChangeScenarioTA(
      initialA: r'''
abstract class A {
  void foo();
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            implementedMethods
              foo: <null>
[status] idle
''',
      updatedA: r'''
abstract class A {
  void foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M6
        interface: #M7
          map
            foo: #M6
          implemented
            foo: #M6
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  implementedMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: <null>
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            implementedMethods
              foo: #M6
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_lookUpMethod_concrete_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeClassElement('A');
      A.thisType.lookUpMethod('foo', A.library, concrete: true);
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            implementedMethods
              foo: #M1
[status] idle
''',
      updatedA: r'''
class A {
  void foo() {}
  void bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M6
          foo: #M1
        interface: #M7
          map
            bar: #M6
            foo: #M1
          implemented
            bar: #M6
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_lookUpMethod_inherited() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var B = state.singleUnit.scopeClassElement('B');
      B.thisType.lookUpMethod(
        'foo',
        B.library,
        concrete: true,
        inherited: true,
      );
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int foo() {}
}
class B extends A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            superMethods
              [0]
                foo: #M1
[status] idle
''',
      updatedA: r'''
class A {
  double foo() {}
}
class B extends A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M8
        interface: #M9
          map
            foo: #M8
          implemented
            foo: #M8
      B: #M3
        interface: #M10
          map
            foo: #M8
          implemented
            foo: #M8
          superImplemented
            [0]
              foo: #M8
          inherited
            foo: #M8
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  superImplementedMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: B
    superIndex: 0
    methodName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            superMethods
              [0]
                foo: #M8
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_lookUpMethod_inherited_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var B = state.singleUnit.scopeClassElement('B');
      B.thisType.lookUpMethod(
        'foo',
        B.library,
        concrete: true,
        inherited: true,
      );
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  void foo() {}
}
class B extends A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            superMethods
              [0]
                foo: #M1
[status] idle
''',
      updatedA: r'''
class A {
  void foo() {}
  void bar() {}
}
class B extends A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M8
          foo: #M1
        interface: #M9
          map
            bar: #M8
            foo: #M1
          implemented
            bar: #M8
            foo: #M1
      B: #M3
        interface: #M10
          map
            bar: #M8
            foo: #M1
          implemented
            bar: #M8
            foo: #M1
          superImplemented
            [0]
              bar: #M8
              foo: #M1
          inherited
            bar: #M8
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_lookUpMethod_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeClassElement('A');
      A.thisType.lookUpMethod('foo', A.library);
    });

    await _runChangeScenarioTA(
      initialA: r'''
abstract class A {
  void foo();
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M1
[status] idle
''',
      updatedA: r'''
abstract class A {
  void foo();
  void bar();
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M6
          foo: #M1
        interface: #M7
          map
            bar: #M6
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_class_instanceMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int foo() {}
  int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M1
          foo: #M2
        interface: #M3
          map
            bar: #M1
            foo: #M2
          implemented
            bar: #M1
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  int foo() {}
  double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M8
          foo: #M2
        interface: #M9
          map
            bar: #M8
            foo: #M2
          implemented
            bar: #M8
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_inherited_fromGeneric_extends2_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  T foo() {}
}

class B extends A<int> {}

class C extends B {}
''',
      testCode: r'''
import 'a.dart';
void f(C c) {
  c.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
      C: #M5
        interface: #M6
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M3
      C: #M5
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M8
    exportMapId: #M9
    exportMap
      f: #M8
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M7
        exportMap
          C: #M5
          C=: <null>
        reExportDeprecatedOnly
          C: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M7
        exportMap
          C: #M5
          C=: <null>
        reExportDeprecatedOnly
          C: false
        interfaces
          C
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  T foo() {}
}

class B extends A<double> {}

class C extends B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M11
        interface: #M12
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
      C: #M13
        interface: #M14
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M15
    exportMap
      A: #M0
      B: #M11
      C: #M13
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: C
    expectedId: #M5
    actualId: #M13
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M16
    exportMapId: #M17
    exportMap
      f: #M16
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M15
        exportMap
          C: #M13
          C=: <null>
        reExportDeprecatedOnly
          C: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: C
    expectedId: #M5
    actualId: #M13
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M15
        exportMap
          C: #M13
          C=: <null>
        reExportDeprecatedOnly
          C: false
        interfaces
          C
            methods
              foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_inherited_fromGeneric_extends_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  T foo() {}
}

class B extends A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  T foo() {}
}

class B extends A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M9
        interface: #M10
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M11
    exportMap
      A: #M0
      B: #M9
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_inherited_fromGeneric_implements_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  T foo() {}
}

class B implements A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          inherited
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  T foo() {}
}

class B implements A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M9
        interface: #M10
          map
            foo: #M1
          inherited
            foo: #M1
    exportMapId: #M11
    exportMap
      A: #M0
      B: #M9
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_inherited_fromGeneric_with_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  T foo() {}
}

class B with A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [1]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  T foo() {}
}

class B with A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M9
        interface: #M10
          map
            foo: #M1
          implemented
            foo: #M1
          superImplemented
            [1]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M11
    exportMap
      A: #M0
      B: #M9
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceMethod_inherited_private() async {
    // Test that there is a dependency between `f()` and `A._foo`.
    // So, that we re-analyze `f()` body when `A._foo` changes.
    // Currently this dependency is implicit: we analyze the whole library
    // when any of its files changes.
    configuration.withStreamResolvedUnitResults = false;

    newFile(testFile.path, r'''
class A {
  int _foo() => 0;
}

class B extends A {}

void f (B b) {
  b._foo().isEven;
}
''');

    await _runChangeScenario(
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          _foo: #M1
        interface: #M2
      B: #M3
        interface: #M4
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M3
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updateFiles: () {
        modifyFile2(testFile, r'''
class A {
  String _foo() => '';
}

class B extends A {}

void f (B b) {
  b._foo().isEven;
}
''');
        return [testFile];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredMethods
          _foo: #M7
        interface: #M2
      B: #M3
        interface: #M4
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M3
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      84 +6 UNDEFINED_GETTER
''',
    );
  }

  test_dependency_class_instanceMethod_remove_invoked() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M7
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        interfaces
          A
            methods
              foo: <null>
              foo=: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      35 +3 UNDEFINED_METHOD
''',
    );
  }

  test_dependency_class_instanceSetter_add_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      35 +3 UNDEFINED_SETTER
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredSetters
              foo=: <null>
            requestedDeclaredMethods
              foo: <null>
        interfaces
          A
            methods
              foo: <null>
              foo=: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      35 +3 UNDEFINED_SETTER
''',
      updatedA: r'''
class A {
  set foo(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M6
        declaredSetters
          foo=: #M7
        interface: #M8
          map
            foo=: #M7
          implemented
            foo=: #M7
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo=
    expectedId: <null>
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M6
        interfaces
          A
            methods
              foo=: #M7
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceSetter_change_getField_setter() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getField('foo')?.setter;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  int get foo => 0;
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        interface: #M4
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredSetters
              foo=: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  int get foo => 0;
  set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M8
        interface: #M9
          map
            foo: #M2
            foo=: #M8
          implemented
            foo: #M2
            foo=: #M8
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo=
    expectedId: #M3
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredSetters
              foo=: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceSetter_change_getSetter() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getSetter('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredSetters
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M7
        declaredSetters
          foo=: #M8
        interface: #M9
          map
            foo=: #M8
          implemented
            foo=: #M8
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo=
    expectedId: #M2
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredSetters
              foo=: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceSetter_change_getSetter_other() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.getSetter('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  set foo(int _) {}
  set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredSetters
          bar=: #M3
          foo=: #M4
        interface: #M5
          map
            bar=: #M3
            foo=: #M4
          implemented
            bar=: #M3
            foo=: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredSetters
              foo=: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  set foo(int _) {}
  set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M9
          foo: #M2
        declaredSetters
          bar=: #M10
          foo=: #M4
        interface: #M11
          map
            bar=: #M10
            foo=: #M4
          implemented
            bar=: #M10
            foo=: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_instanceSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M8
        declaredSetters
          foo=: #M9
        interface: #M10
          map
            foo=: #M9
          implemented
            foo=: #M9
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
        interfaces
          A
            methods
              foo=: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceSetter_change_invoked_hasNamedConstructor_hasInstanceGetter() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.foo(int _);
  int get foo {}
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  var a = A.foo(0);
  a.foo;
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        declaredConstructors
          foo: #M4
        interface: #M5
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            requestedConstructors
              foo: #M4
            methods
              foo: #M2
              foo=: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.foo(int _);
  int get foo {}
  set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M10
        declaredConstructors
          foo: #M4
        interface: #M11
          map
            foo: #M2
            foo=: #M10
          implemented
            foo: #M2
            foo=: #M10
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo=
    expectedId: #M3
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            requestedConstructors
              foo: #M4
            methods
              foo: #M2
              foo=: #M10
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  set foo(int _) {}
  set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredSetters
          bar=: #M3
          foo=: #M4
        interface: #M5
          map
            bar=: #M3
            foo=: #M4
          implemented
            bar=: #M3
            foo=: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
        interfaces
          A
            methods
              foo=: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  set foo(int _) {}
  set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M10
          foo: #M2
        declaredSetters
          bar=: #M11
          foo=: #M4
        interface: #M12
          map
            bar=: #M11
            foo=: #M4
          implemented
            bar=: #M11
            foo=: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceSetter_inherited_fromGeneric_extends_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  set foo(T _) {}
}

class B extends A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M4
        interface: #M5
          map
            foo=: #M2
          implemented
            foo=: #M2
          superImplemented
            [0]
              foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  set foo(T _) {}
}

class B extends A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M10
        interface: #M11
          map
            foo=: #M2
          implemented
            foo=: #M2
          superImplemented
            [0]
              foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M13
    exportMapId: #M14
    exportMap
      f: #M13
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo=: #M2
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceSetter_inherited_fromGeneric_implements_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  set foo(T _) {}
}

class B implements A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M4
        interface: #M5
          map
            foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  set foo(T _) {}
}

class B implements A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M10
        interface: #M11
          map
            foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M13
    exportMapId: #M14
    exportMap
      f: #M13
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo=: #M2
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceSetter_inherited_fromGeneric_with_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A<T> {
  set foo(T _) {}
}

class B with A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M4
        interface: #M5
          map
            foo=: #M2
          implemented
            foo=: #M2
          superImplemented
            [1]
              foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A<T> {
  set foo(T _) {}
}

class B with A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M10
        interface: #M11
          map
            foo=: #M2
          implemented
            foo=: #M2
          superImplemented
            [1]
              foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M13
    exportMapId: #M14
    exportMap
      f: #M13
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo=: #M2
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_instanceSetter_inherited_private() async {
    // Test that there is a dependency between `f()` and `A._foo`.
    // So, that we re-analyze `f()` body when `A._foo` changes.
    // Currently this dependency is implicit: we analyze the whole library
    // when any of its files changes.
    configuration.withStreamResolvedUnitResults = false;

    newFile(testFile.path, r'''
class A {
  set _foo(int _) {}
}

class B extends A {}

void f (B b) {
  b._foo = 0;
}
''');

    await _runChangeScenario(
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          _foo: #M1
        declaredSetters
          _foo=: #M2
        interface: #M3
      B: #M4
        interface: #M5
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M4
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updateFiles: () {
        modifyFile2(testFile, r'''
class A {
  set _foo(String _) {}
}

class B extends A {}

void f (B b) {
  b._foo = 0;
}
''');
        return [testFile];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          _foo: #M8
        declaredSetters
          _foo=: #M9
        interface: #M3
      B: #M4
        interface: #M5
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M4
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      85 +1 INVALID_ASSIGNMENT
''',
    );
  }

  test_dependency_class_instanceSetter_remove_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M8
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      35 +3 UNDEFINED_SETTER
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredSetters
              foo=: <null>
            requestedDeclaredMethods
              foo: <null>
        interfaces
          A
            methods
              foo: <null>
              foo=: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      35 +3 UNDEFINED_SETTER
''',
    );
  }

  test_dependency_class_instanceSetters_add_getSetters() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.setters;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredSetters: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  set foo(int _) {}
  set bar(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M7
          foo: #M1
        declaredSetters
          bar=: #M8
          foo=: #M2
        interface: #M9
          map
            bar=: #M8
            foo=: #M2
          implemented
            bar=: #M8
            foo=: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: setters
    expectedIds: #M2
    actualIds: #M2 #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredSetters: #M2 #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_it_add() async {
    await _runChangeScenarioTA(
      initialA: '',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    exportMapId: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M1
    exportMapId: #M2
    exportMap
      foo: #M1
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M0
        exportMap
          A: <null>
          A=: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      17 +1 UNDEFINED_CLASS
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M0
        exportMap
          A: <null>
          A=: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      17 +1 UNDEFINED_CLASS
''',
      updatedA: r'''
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M4
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: <null>
    actualId: #M4
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M7
    exportMapId: #M8
    exportMap
      foo: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M4
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: <null>
    actualId: #M4
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M6
        exportMap
          A: #M4
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
    );
  }

  test_dependency_class_it_add_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M3
    exportMapId: #M4
    exportMap
      foo: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
      updatedA: r'''
class A {}
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M6
        interface: #M7
    exportMapId: #M8
    exportMap
      A: #M0
      B: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
    );
  }

  test_dependency_class_it_change() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class B {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M5
    exportMapId: #M6
    exportMap
      foo: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
      updatedA: r'''
class A extends B {}
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M8
        interface: #M9
      B: #M2
        interface: #M3
    exportMapId: #M10
    exportMap
      A: #M8
      B: #M2
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M8
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M11
    exportMapId: #M12
    exportMap
      foo: #M11
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M10
        exportMap
          A: #M8
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M10
        exportMap
          A: #M8
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
    );
  }

  test_dependency_class_it_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class B {}
class C {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M2
      C: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M7
    exportMapId: #M8
    exportMap
      foo: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
      updatedA: r'''
class A {}
class B extends C {}
class C {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M10
        interface: #M11
      C: #M4
        interface: #M5
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
      C: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
    );
  }

  test_dependency_class_it_conflict_fix() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class A {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredConflicts
      A: #M0
      A=: #M0
    exportMapId: #M1
    exportMap
      A: #M0
      A=: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M2
    exportMapId: #M3
    exportMap
      foo: #M2
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M1
        exportMap
          A: #M0
          A=: #M0
        reExportDeprecatedOnly
          A: false
          A=: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M1
        exportMap
          A: #M0
          A=: #M0
        reExportDeprecatedOnly
          A: false
          A=: false
[status] idle
''',
      updatedA: r'''
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      A: #M5
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M5
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M8
    exportMapId: #M9
    exportMap
      foo: #M8
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M7
        exportMap
          A: #M5
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M5
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M7
        exportMap
          A: #M5
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
''',
    );
  }

  test_dependency_class_it_conflict_introduce() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M3
    exportMapId: #M4
    exportMap
      foo: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
''',
      updatedA: r'''
class A {}
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredConflicts
      A: #M6
      A=: #M6
    exportMapId: #M7
    exportMap
      A: #M6
      A=: #M6
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M6
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M8
    exportMapId: #M9
    exportMap
      foo: #M8
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M7
        exportMap
          A: #M6
          A=: #M6
        reExportDeprecatedOnly
          A: false
          A=: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M7
        exportMap
          A: #M6
          A=: #M6
        reExportDeprecatedOnly
          A: false
          A=: false
[status] idle
''',
    );
  }

  test_dependency_class_it_conflict_keep() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class A {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredConflicts
      A: #M0
      A=: #M0
    exportMapId: #M1
    exportMap
      A: #M0
      A=: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M2
    exportMapId: #M3
    exportMap
      foo: #M2
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M1
        exportMap
          A: #M0
          A=: #M0
        reExportDeprecatedOnly
          A: false
          A=: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M1
        exportMap
          A: #M0
          A=: #M0
        reExportDeprecatedOnly
          A: false
          A=: false
[status] idle
''',
      updatedA: r'''
class A {}
class A {}
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredConflicts
      A: #M5
      A=: #M5
    declaredClasses
      B: #M6
        interface: #M7
    exportMapId: #M8
    exportMap
      A: #M5
      A=: #M5
      B: #M6
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M5
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M9
    exportMapId: #M10
    exportMap
      foo: #M9
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M8
        exportMap
          A: #M5
          A=: #M5
        reExportDeprecatedOnly
          A: false
          A=: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M5
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M8
        exportMap
          A: #M5
          A=: #M5
        reExportDeprecatedOnly
          A: false
          A=: false
[status] idle
''',
    );
  }

  test_dependency_class_it_extension_fix() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
extension E on int {
  void foo() {}
}
extension E on int {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredConflicts
      E: #M0
      E=: #M0
    exportMapId: #M1
    exportMap
      E: #M0
      E=: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M2
    exportMapId: #M3
    exportMap
      f: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M1
        exportedExtensions: []
[status] idle
''',
      updatedA: r'''
extension E on int {
  void foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M5
        declaredMethods
          foo: #M6
    exportMapId: #M7
    exportMap
      E: #M5
    exportedExtensions: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  exportedExtensionsMismatch
    libraryUri: package:test/a.dart
    expectedIds: []
    actualIds: #M5
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M7
        instances
          E
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M6
        exportedExtensions: #M5
[status] idle
''',
    );
  }

  test_dependency_class_it_extension_introduce() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
extension E on int {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M2
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        instances
          E
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        exportedExtensions: #M0
[status] idle
''',
      updatedA: r'''
extension E on int {
  void foo() {}
}
extension E on int {
  void foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredConflicts
      E: #M6
      E=: #M6
    exportMapId: #M7
    exportMap
      E: #M6
      E=: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelNotInstance
    libraryUri: package:test/a.dart
    name: E
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M7
        exportedExtensions: []
[status] idle
''',
    );
  }

  test_dependency_class_it_remove() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
A foo() => throw 0;
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M3
    exportMapId: #M4
    exportMap
      foo: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: '',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    exportMapId: #M6
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M7
    exportMapId: #M8
    exportMap
      foo: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: <null>
          A=: <null>
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      17 +1 UNDEFINED_CLASS
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M6
        exportMap
          A: <null>
          A=: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      17 +1 UNDEFINED_CLASS
''',
    );
  }

  test_dependency_class_it_remove_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class B {}
''',
      testCode: r'''
import 'a.dart';
A foo() => throw 0;
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M5
    exportMapId: #M6
    exportMap
      foo: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_namedConstructor_add_invoked() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.c1();
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.c2();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          c1: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              c2: <null>
            requestedDeclaredMethods
              c2: <null>
        interfaces
          A
            requestedConstructors
              c2: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      32 +2 UNDEFINED_METHOD
''',
      updatedA: r'''
class A {
  A.c1();
  A.c2();
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          c1: #M1
          c2: #M7
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: c2
    expectedId: <null>
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              c2: #M7
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_namedConstructor_add_isValidMixin() async {
    configuration.withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      state.singleUnit.scopeClassElement('A').isValidMixin;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M5
            allInheritedConstructors: []
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  A.named();
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M6
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    childrenPropertyName: declaredConstructors
    expectedIds: #M5
    actualIds: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M6
            allInheritedConstructors: []
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_namedConstructor_change_getConstructors() async {
    configuration.includeDefaultConstructors();

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.constructors;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A(int _);
  A.named(int _);
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
          new: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M2 #M1
            allInheritedConstructors: []
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  A(int _);
  A.named(double _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M7
          new: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    childrenPropertyName: declaredConstructors
    expectedIds: #M2 #M1
    actualIds: #M2 #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M2 #M7
            allInheritedConstructors: []
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_namedConstructor_change_getNamedConstructor() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.getNamedConstructor('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.foo(int _);
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          foo: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  A.foo(double _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          foo: #M6
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: foo
    expectedId: #M1
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              foo: #M6
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_namedConstructor_change_getNamedConstructor_other() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.getNamedConstructor('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.foo(int _);
  A.bar(int _);
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          bar: #M1
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  A.foo(int _);
  A.bar(double _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          bar: #M7
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_namedConstructor_change_getUnnamedConstructor() async {
    configuration.includeDefaultConstructors();

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.unnamedConstructor;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A();
  A.named(int _);
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
          new: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              new: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  A();
  A.named(double _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M7
          new: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_namedConstructor_change_invoked() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.named(int _);
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.named(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              named: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.named(double _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M7
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: named
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              named: #M7
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_namedConstructor_change_invoked_hasInstanceGetterSetter() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.foo(int _);
  int get foo {}
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';

void f() {
  var a = A.foo(0);
  a.foo;
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        declaredConstructors
          foo: #M4
        interface: #M5
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            requestedConstructors
              foo: #M4
            methods
              foo: #M2
              foo=: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.foo(double _);
  int get foo {}
  set foo(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        declaredConstructors
          foo: #M10
        interface: #M5
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: foo
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            requestedConstructors
              foo: #M10
            methods
              foo: #M2
              foo=: #M3
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_namedConstructor_change_invoked_hasInstanceMethod() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.foo(int _);
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo(0).foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        declaredConstructors
          foo: #M2
        interface: #M3
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              foo: #M2
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.foo(double _);
  int foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        declaredConstructors
          foo: #M8
        interface: #M3
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: foo
    expectedId: #M2
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              foo: #M8
            methods
              foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_namedConstructor_change_invoked_superInvocation() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.named(int _);
}
''',
      testCode: r'''
import 'a.dart';
class B extends A {
  B.foo() : super.named(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M4
        declaredConstructors
          foo: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      B: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
          named: <null>
          named=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M2
            hasNonFinalField: false
            requestedConstructors
              named: #M1
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M8
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
          named: <null>
          named=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: []
        interfaces
          A
            allDeclaredConstructors: #M1
            allInheritedConstructors: []
            requestedConstructors
              named: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.named(double _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M9
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: named
    expectedId: #M1
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M4
        declaredConstructors
          foo: #M10
        interface: #M6
    exportMapId: #M7
    exportMap
      B: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
          named: <null>
          named=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M2
            hasNonFinalField: false
            requestedConstructors
              named: #M9
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    childrenPropertyName: declaredConstructors
    expectedIds: #M1
    actualIds: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M8
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
          named: <null>
          named=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: []
        interfaces
          A
            allDeclaredConstructors: #M9
            allInheritedConstructors: []
            requestedConstructors
              named: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_namedConstructor_change_notUsed() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.c1();
  A.c2(int _);
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.c1();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          c1: #M1
          c2: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              c1: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.c1();
  A.c2(double _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          c1: #M1
          c2: #M8
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_namedConstructor_remove_invoked() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.c1();
  A.c2();
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.c2();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          c1: #M1
          c2: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              c2: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.c1();
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          c1: #M1
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: c2
    expectedId: #M2
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              c2: <null>
            requestedDeclaredMethods
              c2: <null>
        interfaces
          A
            requestedConstructors
              c2: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      32 +2 UNDEFINED_METHOD
''',
    );
  }

  test_dependency_class_staticField_add_getFields() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInstanceElement('A');
      A.fields;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  static final int foo = 0;
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredFields: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  static final int foo = 0;
  static final int bar = 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M7
          foo: #M1
        declaredGetters
          bar: #M8
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    instanceName: A
    childrenPropertyName: fields
    expectedIds: #M1
    actualIds: #M1 #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredFields: #M1 #M7
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_staticGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  static int get foo {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M2
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  static double get foo {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M8
        declaredGetters
          foo: #M9
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
            requestedDeclaredGetters
              foo: #M9
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_staticGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  static int get foo {}
  static int get bar {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredGetters
          bar: #M3
          foo: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredGetters
              foo: #M4
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  static int get foo {}
  static double get bar {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M10
          foo: #M2
        declaredGetters
          bar: #M11
          foo: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_staticMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  static int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  static double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M7
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M7
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_staticMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  static int foo() {}
  static int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M1
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M2
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  static int foo() {}
  static double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredMethods
          bar: #M8
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_staticSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  static set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredSetters
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  static set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          foo: #M8
        declaredSetters
          foo=: #M9
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
            requestedDeclaredSetters
              foo=: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_staticSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  static set foo(int _) {}
  static set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredSetters
          bar=: #M3
          foo=: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredSetters
              foo=: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  static set foo(int _) {}
  static set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredFields
          bar: #M10
          foo: #M2
        declaredSetters
          bar=: #M11
          foo=: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_unnamedConstructor_change_getConstructors() async {
    configuration.includeDefaultConstructors();

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.constructors;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A(int _);
  A.named(int _);
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
          new: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M2 #M1
            allInheritedConstructors: []
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  A(double _);
  A.named(int _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
          new: #M7
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    childrenPropertyName: declaredConstructors
    expectedIds: #M2 #M1
    actualIds: #M7 #M1
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M7 #M1
            allInheritedConstructors: []
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_unnamedConstructor_change_getUnnamedConstructor() async {
    configuration.includeDefaultConstructors();

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.unnamedConstructor;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A(int _);
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          new: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              new: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {
  A(int _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] reuseLinkedBundle
  package:test/a.dart
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_class_unnamedConstructor_change_invoked() async {
    configuration
      ..includeDefaultConstructors()
      ..withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A(int _);
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          new: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              new: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A(double _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          new: #M7
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: new
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              new: #M7
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_class_unnamedConstructor_change_notUsed() async {
    configuration
      ..includeDefaultConstructors()
      ..withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A(int _);
  A.named(int _);
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.named(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
          new: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              named: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A(double _);
  A.named(int _);
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
          new: #M8
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_classTypaAlias_namedConstructor_change_invoked() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.named(int _);
}
mixin M {}
class B = A with M;
''',
      testCode: r'''
import 'a.dart';
void f() {
  B.named(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
        interface: #M2
      B: #M3
        inheritedConstructors
          named: #M1
        interface: #M4
    declaredMixins
      M: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M3
      M: #M5
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M8
    exportMapId: #M9
    exportMap
      f: #M8
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M7
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            requestedConstructors
              named: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.named(double _);
}
mixin M {}
class B = A with M;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M11
        interface: #M2
      B: #M3
        inheritedConstructors
          named: #M11
        interface: #M4
    declaredMixins
      M: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M3
      M: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: B
    constructorName: named
    expectedId: #M1
    actualId: #M11
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M7
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            requestedConstructors
              named: #M11
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_classTypeAlias_namedConstructor_add_getConstructors() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var B = state.singleUnit.scopeInterfaceElement('B');
      B.constructors;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.foo();
}
class B = A with M;
mixin M {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          foo: #M1
        interface: #M2
      B: #M3
        inheritedConstructors
          foo: #M1
        interface: #M4
    declaredMixins
      M: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M3
      M: #M5
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M8
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M7
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            allDeclaredConstructors: []
            allInheritedConstructors: #M1
[status] idle
''',
      updatedA: r'''
class A {
  A.foo();
  A.bar();
}
class B = A with M;
mixin M {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          bar: #M10
          foo: #M1
        interface: #M2
      B: #M3
        inheritedConstructors
          bar: #M10
          foo: #M1
        interface: #M4
    declaredMixins
      M: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M3
      M: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: B
    childrenPropertyName: inheritedConstructors
    expectedIds: #M1
    actualIds: #M1 #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M7
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            allDeclaredConstructors: []
            allInheritedConstructors: #M1 #M10
[status] idle
''',
    );
  }

  test_dependency_enum_constant_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  foo(0);
  const A(int _)
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          foo: #M1
          values: #M2
        declaredGetters
          foo: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M7
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M8
    exportMapId: #M9
    exportMap
      f: #M8
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M7
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M3
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  foo(1);
  const A(int _)
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          foo: #M11
          values: #M12
        declaredGetters
          foo: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M7
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M11
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M7
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M11
            requestedDeclaredGetters
              foo: #M3
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_instanceGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          foo: #M1
          v: #M2
          values: #M3
        declaredGetters
          foo: #M4
          v: #M5
          values: #M6
        interface: #M7
          map
            foo: #M4
            index: #M8
          implemented
            foo: #M4
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M10
    exportMapId: #M11
    exportMap
      f: #M10
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  double get foo => 1.2;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          foo: #M13
          v: #M2
          values: #M3
        declaredGetters
          foo: #M14
          v: #M5
          values: #M6
        interface: #M15
          map
            foo: #M14
            index: #M8
          implemented
            foo: #M14
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M13
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M13
        interfaces
          A
            methods
              foo: #M14
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_instanceGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  int get foo => 0;
  int get bar => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
          v: #M3
          values: #M4
        declaredGetters
          bar: #M5
          foo: #M6
          v: #M7
          values: #M8
        interface: #M9
          map
            bar: #M5
            foo: #M6
            index: #M10
          implemented
            bar: #M5
            foo: #M6
            index: #M10
          superImplemented
            [0]
              index: #M10
          inherited
            index: #M10
    exportMapId: #M11
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M11
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M14
        exportMapId: #M11
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
        interfaces
          A
            methods
              foo: #M6
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  int get foo => 0;
  double get bar => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          bar: #M15
          foo: #M2
          v: #M3
          values: #M4
        declaredGetters
          bar: #M16
          foo: #M6
          v: #M7
          values: #M8
        interface: #M17
          map
            bar: #M16
            foo: #M6
            index: #M10
          implemented
            bar: #M16
            foo: #M6
            index: #M10
          superImplemented
            [0]
              index: #M10
          inherited
            index: #M10
    exportMapId: #M11
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_instanceMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        declaredMethods
          foo: #M5
        interface: #M6
          map
            foo: #M5
            index: #M7
          implemented
            foo: #M5
            index: #M7
          superImplemented
            [0]
              index: #M7
          inherited
            index: #M7
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M9
    exportMapId: #M10
    exportMap
      f: #M9
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M11
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M5
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        declaredMethods
          foo: #M12
        interface: #M13
          map
            foo: #M12
            index: #M7
          implemented
            foo: #M12
            index: #M7
          superImplemented
            [0]
              index: #M7
          inherited
            index: #M7
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M5
    actualId: #M12
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M11
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M12
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_instanceMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  int foo() {}
  int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        declaredMethods
          bar: #M5
          foo: #M6
        interface: #M7
          map
            bar: #M5
            foo: #M6
            index: #M8
          implemented
            bar: #M5
            foo: #M6
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M10
    exportMapId: #M11
    exportMap
      f: #M10
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M6
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  int foo() {}
  double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        declaredMethods
          bar: #M13
          foo: #M6
        interface: #M14
          map
            bar: #M13
            foo: #M6
            index: #M8
          implemented
            bar: #M13
            foo: #M6
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_instanceSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          foo: #M1
          v: #M2
          values: #M3
        declaredGetters
          v: #M4
          values: #M5
        declaredSetters
          foo=: #M6
        interface: #M7
          map
            foo=: #M6
            index: #M8
          implemented
            foo=: #M6
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M10
    exportMapId: #M11
    exportMap
      f: #M10
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo=: #M6
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          foo: #M13
          v: #M2
          values: #M3
        declaredGetters
          v: #M4
          values: #M5
        declaredSetters
          foo=: #M14
        interface: #M15
          map
            foo=: #M14
            index: #M8
          implemented
            foo=: #M14
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M13
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M13
        interfaces
          A
            methods
              foo=: #M14
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_instanceSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  set foo(int _) {}
  set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
          v: #M3
          values: #M4
        declaredGetters
          v: #M5
          values: #M6
        declaredSetters
          bar=: #M7
          foo=: #M8
        interface: #M9
          map
            bar=: #M7
            foo=: #M8
            index: #M10
          implemented
            bar=: #M7
            foo=: #M8
            index: #M10
          superImplemented
            [0]
              index: #M10
          inherited
            index: #M10
    exportMapId: #M11
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M11
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M14
        exportMapId: #M11
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
        interfaces
          A
            methods
              foo=: #M8
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  set foo(int _) {}
  set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          bar: #M15
          foo: #M2
          v: #M3
          values: #M4
        declaredGetters
          v: #M5
          values: #M6
        declaredSetters
          bar=: #M16
          foo=: #M8
        interface: #M17
          map
            bar=: #M16
            foo=: #M8
            index: #M10
          implemented
            bar=: #M16
            foo=: #M8
            index: #M10
          superImplemented
            [0]
              index: #M10
          inherited
            index: #M10
    exportMapId: #M11
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_staticGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  static int get foo {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          foo: #M1
          v: #M2
          values: #M3
        declaredGetters
          foo: #M4
          v: #M5
          values: #M6
        interface: #M7
          map
            index: #M8
          implemented
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M10
    exportMapId: #M11
    exportMap
      f: #M10
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M4
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  static double get foo {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          foo: #M13
          v: #M2
          values: #M3
        declaredGetters
          foo: #M14
          v: #M5
          values: #M6
        interface: #M7
          map
            index: #M8
          implemented
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M13
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M13
            requestedDeclaredGetters
              foo: #M14
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_staticGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  static int get foo {}
  static int get bar {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
          v: #M3
          values: #M4
        declaredGetters
          bar: #M5
          foo: #M6
          v: #M7
          values: #M8
        interface: #M9
          map
            index: #M10
          implemented
            index: #M10
          superImplemented
            [0]
              index: #M10
          inherited
            index: #M10
    exportMapId: #M11
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M14
        exportMapId: #M11
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredGetters
              foo: #M6
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  static int get foo {}
  static double get bar {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          bar: #M15
          foo: #M2
          v: #M3
          values: #M4
        declaredGetters
          bar: #M16
          foo: #M6
          v: #M7
          values: #M8
        interface: #M9
          map
            index: #M10
          implemented
            index: #M10
          superImplemented
            [0]
              index: #M10
          inherited
            index: #M10
    exportMapId: #M11
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_staticMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  static int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        declaredMethods
          foo: #M5
        interface: #M6
          map
            index: #M7
          implemented
            index: #M7
          superImplemented
            [0]
              index: #M7
          inherited
            index: #M7
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M9
    exportMapId: #M10
    exportMap
      f: #M9
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M11
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M5
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  static double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        declaredMethods
          foo: #M12
        interface: #M6
          map
            index: #M7
          implemented
            index: #M7
          superImplemented
            [0]
              index: #M7
          inherited
            index: #M7
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M5
    actualId: #M12
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M11
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M12
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_staticMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  static int foo() {}
  static int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        declaredMethods
          bar: #M5
          foo: #M6
        interface: #M7
          map
            index: #M8
          implemented
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M10
    exportMapId: #M11
    exportMap
      f: #M10
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M6
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  static int foo() {}
  static double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        declaredMethods
          bar: #M13
          foo: #M6
        interface: #M7
          map
            index: #M8
          implemented
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_staticSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  static set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          foo: #M1
          v: #M2
          values: #M3
        declaredGetters
          v: #M4
          values: #M5
        declaredSetters
          foo=: #M6
        interface: #M7
          map
            index: #M8
          implemented
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M10
    exportMapId: #M11
    exportMap
      f: #M10
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredSetters
              foo=: #M6
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  static set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          foo: #M13
          v: #M2
          values: #M3
        declaredGetters
          v: #M4
          values: #M5
        declaredSetters
          foo=: #M14
        interface: #M7
          map
            index: #M8
          implemented
            index: #M8
          superImplemented
            [0]
              index: #M8
          inherited
            index: #M8
    exportMapId: #M9
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M13
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M12
        exportMapId: #M9
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M13
            requestedDeclaredSetters
              foo=: #M14
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_enum_staticSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
enum A {
  v;
  static set foo(int _) {}
  static set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
          v: #M3
          values: #M4
        declaredGetters
          v: #M5
          values: #M6
        declaredSetters
          bar=: #M7
          foo=: #M8
        interface: #M9
          map
            index: #M10
          implemented
            index: #M10
          superImplemented
            [0]
              index: #M10
          inherited
            index: #M10
    exportMapId: #M11
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M14
        exportMapId: #M11
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredSetters
              foo=: #M8
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
enum A {
  v;
  static set foo(int _) {}
  static set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          bar: #M15
          foo: #M2
          v: #M3
          values: #M4
        declaredGetters
          v: #M5
          values: #M6
        declaredSetters
          bar=: #M16
          foo=: #M8
        interface: #M9
          map
            index: #M10
          implemented
            index: #M10
          superImplemented
            [0]
              index: #M10
          inherited
            index: #M10
    exportMapId: #M11
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_export_class_excludePrivate() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
class _B {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      _B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      _B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
    exportMap
      A: #M0
    reExportMap
      A: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            A: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A
    exportNamespace
      A: package:test/a.dart::@class::A
''',
      updateFiles: () {
        modifyFile2(a, r'''
class A {}
class _B2 {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      _B2: #M6
        interface: #M7
    exportMapId: #M8
    exportMap
      A: #M0
      _B2: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A
    exportNamespace
      A: package:test/a.dart::@class::A
''',
    );
  }

  test_dependency_export_class_localHidesExport_addHidden() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
class B {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
class B {}
class C {}
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M5
        interface: #M6
      C: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M5
      C: #M7
    reExportMap
      A: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        declaredTopNames: B C
        exports
          package:test/a.dart
            A: #M0
[status] idle
[future] getLibraryByUri T1
  library
    classes
      class B
        constructors
          synthetic new
      class C
        constructors
          synthetic new
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A
      declared <testLibrary>::@class::B
      declared <testLibrary>::@class::C
    exportNamespace
      A: package:test/a.dart::@class::A
      B: <testLibrary>::@class::B
      C: <testLibrary>::@class::C
''',
      updateFiles: () {
        modifyFile2(a, r'''
class A {}
class B {}
class C {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M10
        interface: #M11
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M2
      C: #M10
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    classes
      class B
        constructors
          synthetic new
      class C
        constructors
          synthetic new
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A
      declared <testLibrary>::@class::B
      declared <testLibrary>::@class::C
    exportNamespace
      A: package:test/a.dart::@class::A
      B: <testLibrary>::@class::B
      C: <testLibrary>::@class::C
''',
    );
  }

  test_dependency_export_class_localHidesExport_addNotHidden() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
class B {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
class B {}
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M5
    reExportMap
      A: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        declaredTopNames: B
        exports
          package:test/a.dart
            A: #M0
[status] idle
[future] getLibraryByUri T1
  library
    classes
      class B
        constructors
          synthetic new
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A
      declared <testLibrary>::@class::B
    exportNamespace
      A: package:test/a.dart::@class::A
      B: <testLibrary>::@class::B
''',
      updateFiles: () {
        modifyFile2(a, r'''
class A {}
class B {}
class C {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M8
        interface: #M9
    exportMapId: #M10
    exportMap
      A: #M0
      B: #M2
      C: #M8
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: C
    expectedId: <null>
    actualId: #M8
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M5
        interface: #M6
    exportMapId: #M11
    exportMap
      A: #M0
      B: #M5
      C: #M8
    reExportMap
      A: #M0
      C: #M8
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        declaredTopNames: B
        exports
          package:test/a.dart
            A: #M0
            C: #M8
[status] idle
[future] getLibraryByUri T2
  library
    classes
      class B
        constructors
          synthetic new
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A
      exported[(0, 0)] package:test/a.dart::@class::C
      declared <testLibrary>::@class::B
    exportNamespace
      A: package:test/a.dart::@class::A
      B: <testLibrary>::@class::B
      C: package:test/a.dart::@class::C
''',
    );
  }

  test_dependency_export_class_reExport_chain_external() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A1 {}
class A2 {}
class A3 {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
export 'a.dart';
class B {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'b.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A1: #M0
        interface: #M1
      A2: #M2
        interface: #M3
      A3: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A1: #M0
      A2: #M2
      A3: #M4
  requirements
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A1: #M0
      A2: #M2
      A3: #M4
      B: #M7
    reExportMap
      A1: #M0
      A2: #M2
      A3: #M4
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/b.dart
        declaredTopNames: B
        exports
          package:test/a.dart
            A1: #M0
            A2: #M2
            A3: #M4
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    exportMapId: #M10
    exportMap
      A1: #M0
      A2: #M2
      A3: #M4
      B: #M7
    reExportMap
      A1: #M0
      A2: #M2
      A3: #M4
      B: #M7
    exportedLibraryUris: package:test/b.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/b.dart
            A1: #M0
            A2: #M2
            A3: #M4
            B: #M7
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A1
      exported[(0, 0)] package:test/a.dart::@class::A2
      exported[(0, 0)] package:test/a.dart::@class::A3
      exported[(0, 0)] package:test/b.dart::@class::B
    exportNamespace
      A1: package:test/a.dart::@class::A1
      A2: package:test/a.dart::@class::A2
      A3: package:test/a.dart::@class::A3
      B: package:test/b.dart::@class::B
''',
      updateFiles: () {
        modifyFile2(a, r'''
abstract class A2 {}
class A3 {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    declaredClasses
      A2: #M11
        interface: #M12
      A3: #M4
        interface: #M5
    exportMapId: #M13
    exportMap
      A2: #M11
      A3: #M4
  requirements
[operation] checkLinkedBundleRequirements
  package:test/b.dart
  exportIdMismatch
    fragmentUri: package:test/b.dart
    exportedUri: package:test/a.dart
    name: A2
    expectedId: #M2
    actualId: #M11
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H4
    declaredClasses
      B: #M7
        interface: #M8
    exportMapId: #M14
    exportMap
      A2: #M11
      A3: #M4
      B: #M7
    reExportMap
      A2: #M11
      A3: #M4
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/b.dart
        declaredTopNames: B
        exports
          package:test/a.dart
            A2: #M11
            A3: #M4
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/b.dart
    name: A2
    expectedId: #M2
    actualId: #M11
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H5
    exportMapId: #M15
    exportMap
      A2: #M11
      A3: #M4
      B: #M7
    reExportMap
      A2: #M11
      A3: #M4
      B: #M7
    exportedLibraryUris: package:test/b.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/b.dart
            A2: #M11
            A3: #M4
            B: #M7
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A2
      exported[(0, 0)] package:test/a.dart::@class::A3
      exported[(0, 0)] package:test/b.dart::@class::B
    exportNamespace
      A2: package:test/a.dart::@class::A2
      A3: package:test/a.dart::@class::A3
      B: package:test/b.dart::@class::B
''',
    );
  }

  test_dependency_export_class_reExport_chain_thisCycle() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
import 'test.dart';
class A1 {}
class A2 {}
class A3 {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
export 'a.dart';
class B {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'b.dart';
''');

    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A1: #M0
        interface: #M1
      A2: #M2
        interface: #M3
      A3: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A1: #M0
      A2: #M2
      A3: #M4
  package:test/b.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A1: #M0
      A2: #M2
      A3: #M4
      B: #M7
    reExportMap
      A1: #M0
      A2: #M2
      A3: #M4
    exportedLibraryUris: package:test/a.dart
  package:test/test.dart
    hashForRequirements: #H2
    exportMapId: #M10
    exportMap
      A1: #M0
      A2: #M2
      A3: #M4
      B: #M7
    reExportMap
      A1: #M0
      A2: #M2
      A3: #M4
      B: #M7
    exportedLibraryUris: package:test/b.dart
  requirements
[status] idle
[future] getLibraryByUri T1
  library
''',
      updateFiles: () {
        modifyFile2(a, r'''
import 'test.dart';
abstract class A2 {}
class A3 {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    declaredClasses
      A2: #M11
        interface: #M12
      A3: #M4
        interface: #M5
    exportMapId: #M13
    exportMap
      A2: #M11
      A3: #M4
  package:test/b.dart
    hashForRequirements: #H4
    declaredClasses
      B: #M7
        interface: #M8
    exportMapId: #M14
    exportMap
      A2: #M11
      A3: #M4
      B: #M7
    reExportMap
      A2: #M11
      A3: #M4
    exportedLibraryUris: package:test/a.dart
  package:test/test.dart
    hashForRequirements: #H5
    exportMapId: #M15
    exportMap
      A2: #M11
      A3: #M4
      B: #M7
    reExportMap
      A2: #M11
      A3: #M4
      B: #M7
    exportedLibraryUris: package:test/b.dart
  requirements
[status] idle
[future] getLibraryByUri T2
  library
''',
    );
  }

  test_dependency_export_class_reExport_combinatorShow() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
class A {}
''');

    newFile('$testPackageLibPath/b.dart', r'''
export 'a.dart' show A;
class B {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'b.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M3
        interface: #M4
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
    reExportMap
      A: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/b.dart
        declaredTopNames: B
        exports
          package:test/a.dart
            combinators
              show A
            A: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M3
    reExportMap
      A: #M0
      B: #M3
    exportedLibraryUris: package:test/b.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/b.dart
            A: #M0
            B: #M3
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A
      exported[(0, 0)] package:test/b.dart::@class::B
    exportNamespace
      A: package:test/a.dart::@class::A
      B: package:test/b.dart::@class::B
''',
      updateFiles: () {
        modifyFile2(a, r'''
class A {}
class A2 {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    declaredClasses
      A: #M0
        interface: #M1
      A2: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      A2: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/b.dart
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@class::A
      exported[(0, 0)] package:test/b.dart::@class::B
    exportNamespace
      A: package:test/a.dart::@class::A
      B: package:test/b.dart::@class::B
''',
    );
  }

  test_dependency_export_dynamic() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
export 'dart:core' show int, dynamic, Never;
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      int: #M4
    reExportMap
      int: #M4
    exportedLibraryUris: dart:core
  requirements
    exportRequirements
      package:test/test.dart
        exports
          dart:core
            combinators
              show Never, dynamic, int
            int: #M4
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          Never: <null>
          Never=: <null>
          dynamic: <null>
          dynamic=: <null>
          int: <null>
          int=: <null>
[status] idle
''',
      updatedA: r'''
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      B: #M6
        interface: #M7
    exportMapId: #M8
    exportMap
      B: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_export_enum() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
enum E {v}
enum _E {v}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      E: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
      _E: #M7
        declaredFields
          v: #M8
          values: #M9
        declaredGetters
          v: #M10
          values: #M11
        interface: #M12
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M13
    exportMap
      E: #M0
      _E: #M7
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M14
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@enum::E
    exportNamespace
      E: package:test/a.dart::@enum::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
enum E {v}
enum _E2 {v}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      E: #M0
        declaredFields
          v: #M1
          values: #M2
        declaredGetters
          v: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
      _E2: #M15
        declaredFields
          v: #M16
          values: #M17
        declaredGetters
          v: #M18
          values: #M19
        interface: #M20
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M21
    exportMap
      E: #M0
      _E2: #M15
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@enum::E
    exportNamespace
      E: package:test/a.dart::@enum::E
''',
    );
  }

  test_dependency_export_extension() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {}
extension _E on int {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
      _E: #M1
    exportMapId: #M2
    exportMap
      E: #M0
      _E: #M1
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {}
extension _E2 on int {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
      _E2: #M4
    exportMapId: #M5
    exportMap
      E: #M0
      _E2: #M4
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_getter_add() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  int get foo => 0;
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          foo: #M1
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredGetters
          foo: #M2
            flags: isSimplyBounded
            returnType: int @ dart:core
    exportMapId: #M3
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  int get foo => 0;
  int get bar => 0;
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          bar: #M5
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
          foo: #M1
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredGetters
          bar: #M6
            flags: isSimplyBounded
            returnType: int @ dart:core
          foo: #M2
            flags: isSimplyBounded
            returnType: int @ dart:core
    exportMapId: #M3
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_getter_change() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  int get foo => 0;
  int get bar => 0;
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          bar: #M1
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
          foo: #M2
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredGetters
          bar: #M3
            flags: isSimplyBounded
            returnType: int @ dart:core
          foo: #M4
            flags: isSimplyBounded
            returnType: int @ dart:core
    exportMapId: #M5
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  int get foo => 0;
  double get bar => 0;
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          bar: #M7
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: double @ dart:core
          foo: #M2
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredGetters
          bar: #M8
            flags: isSimplyBounded
            returnType: double @ dart:core
          foo: #M4
            flags: isSimplyBounded
            returnType: int @ dart:core
    exportMapId: #M5
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_getter_remove() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  int get foo => 0;
  int get bar => 0;
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          bar: #M1
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
          foo: #M2
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredGetters
          bar: #M3
            flags: isSimplyBounded
            returnType: int @ dart:core
          foo: #M4
            flags: isSimplyBounded
            returnType: int @ dart:core
    exportMapId: #M5
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  int get foo => 0;
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          foo: #M2
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredGetters
          foo: #M4
            flags: isSimplyBounded
            returnType: int @ dart:core
    exportMapId: #M5
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_it_add() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E1 on int {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        extendedType: int @ dart:core
    exportMapId: #M1
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
    exportMap
      E1: #M0
    reExportMap
      E1: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E1: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E1
    exportNamespace
      E1: package:test/a.dart::@extension::E1
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E1 on int {}
extension E2 on int {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        extendedType: int @ dart:core
      E2: #M3
        extendedType: int @ dart:core
    exportMapId: #M4
    exportMap
      E1: #M0
      E2: #M3
    exportedExtensions: #M0 #M3
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: E2
    expectedId: <null>
    actualId: #M3
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    exportMapId: #M5
    exportMap
      E1: #M0
      E2: #M3
    reExportMap
      E1: #M0
      E2: #M3
    exportedExtensions: #M0 #M3
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E1: #M0
            E2: #M3
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E1
      exported[(0, 0)] package:test/a.dart::@extension::E2
    exportNamespace
      E1: package:test/a.dart::@extension::E1
      E2: package:test/a.dart::@extension::E2
''',
    );
  }

  test_dependency_export_extension_it_remove() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E1 on int {}
extension E2 on int {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        extendedType: int @ dart:core
      E2: #M1
        extendedType: int @ dart:core
    exportMapId: #M2
    exportMap
      E1: #M0
      E2: #M1
    exportedExtensions: #M0 #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      E1: #M0
      E2: #M1
    reExportMap
      E1: #M0
      E2: #M1
    exportedExtensions: #M0 #M1
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E1: #M0
            E2: #M1
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E1
      exported[(0, 0)] package:test/a.dart::@extension::E2
    exportNamespace
      E1: package:test/a.dart::@extension::E1
      E2: package:test/a.dart::@extension::E2
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E1 on int {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        extendedType: int @ dart:core
    exportMapId: #M4
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportCountMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    expected: 2
    actual: 1
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    exportMapId: #M5
    exportMap
      E1: #M0
    reExportMap
      E1: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E1: #M0
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E1
    exportNamespace
      E1: package:test/a.dart::@extension::E1
''',
    );
  }

  test_dependency_export_extension_method_add() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  void foo() {}
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredMethods
          foo: #M1
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
    exportMapId: #M2
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  void foo() {}
  void bar() {}
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredMethods
          bar: #M4
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
          foo: #M1
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
    exportMapId: #M2
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_method_change() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  void foo() {}
  void bar() {}
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredMethods
          bar: #M1
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
          foo: #M2
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
    exportMapId: #M3
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  void foo() {}
  void bar(int _) {}
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredMethods
          bar: #M5
            flags: isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
          foo: #M2
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
    exportMapId: #M3
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_method_remove() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  void foo() {}
  void bar() {}
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredMethods
          bar: #M1
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
          foo: #M2
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
    exportMapId: #M3
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  void foo() {}
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredMethods
          foo: #M2
            flags: isSimplyBounded
            functionType: FunctionType
              returnType: void
    exportMapId: #M3
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_setter_add() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  set foo(int _) {}
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          foo: #M1
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredSetters
          foo=: #M2
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
    exportMapId: #M3
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  set foo(int _) {}
  set bar(int _) {}
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          bar: #M5
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
          foo: #M1
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredSetters
          bar=: #M6
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
          foo=: #M2
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
    exportMapId: #M3
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_setter_change() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  set foo(int _) {}
  set bar(int _) {}
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          bar: #M1
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
          foo: #M2
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredSetters
          bar=: #M3
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
          foo=: #M4
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
    exportMapId: #M5
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  set foo(int _) {}
  set bar(double _) {}
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          bar: #M7
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: double @ dart:core
          foo: #M2
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredSetters
          bar=: #M8
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required double @ dart:core
              returnType: void
          foo=: #M4
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
    exportMapId: #M5
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extension_setter_remove() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension E on int {
  set foo(int _) {}
  set bar(int _) {}
}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration
      ..withElementManifests = true
      ..elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          bar: #M1
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
          foo: #M2
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredSetters
          bar=: #M3
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
          foo=: #M4
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
    exportMapId: #M5
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedExtensions: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension E on int {
  set foo(int _) {}
}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        extendedType: int @ dart:core
        declaredFields
          foo: #M2
            flags: isSynthetic shouldUseTypeForInitializerInference
            type: int @ dart:core
        declaredSetters
          foo=: #M4
            flags: hasImplicitReturnType isSimplyBounded
            functionType: FunctionType
              positional
                required int @ dart:core
              returnType: void
    exportMapId: #M5
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extension::E
    exportNamespace
      E: package:test/a.dart::@extension::E
''',
    );
  }

  test_dependency_export_extensionType() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
extension type E(int it) {}
extension type _E(int it) {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      E: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        interface: #M3
          map
            it: #M2
          implemented
            it: #M2
      _E: #M4
        declaredFields
          it: #M5
        declaredGetters
          it: #M6
        interface: #M7
          map
            it: #M6
          implemented
            it: #M6
    exportMapId: #M8
    exportMap
      E: #M0
      _E: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M9
    exportMap
      E: #M0
    reExportMap
      E: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            E: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extensionType::E
    exportNamespace
      E: package:test/a.dart::@extensionType::E
''',
      updateFiles: () {
        modifyFile2(a, r'''
extension type E(int it) {}
extension type _E2(int it) {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      E: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        interface: #M3
          map
            it: #M2
          implemented
            it: #M2
      _E2: #M10
        declaredFields
          it: #M11
        declaredGetters
          it: #M12
        interface: #M13
          map
            it: #M12
          implemented
            it: #M12
    exportMapId: #M14
    exportMap
      E: #M0
      _E2: #M10
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@extensionType::E
    exportNamespace
      E: package:test/a.dart::@extensionType::E
''',
    );
  }

  test_dependency_export_mixin() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
mixin M {}
mixin _M {}
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      M: #M0
        interface: #M1
      _M: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      M: #M0
      _M: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
    exportMap
      M: #M0
    reExportMap
      M: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            M: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@mixin::M
    exportNamespace
      M: package:test/a.dart::@mixin::M
''',
      updateFiles: () {
        modifyFile2(a, r'''
mixin M {}
mixin _M2 {}
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      M: #M0
        interface: #M1
      _M2: #M6
        interface: #M7
    exportMapId: #M8
    exportMap
      M: #M0
      _M2: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@mixin::M
    exportNamespace
      M: package:test/a.dart::@mixin::M
''',
    );
  }

  test_dependency_export_noLibrary() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
export ':';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 1;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
    );
  }

  test_dependency_export_topLevelVariable_add() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
final b = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
      b: #M4
    declaredVariables
      a: #M1
      b: #M5
    exportMapId: #M6
    exportMap
      a: #M0
      b: #M4
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: b
    expectedId: <null>
    actualId: #M4
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    exportMapId: #M7
    exportMap
      a: #M0
      b: #M4
    reExportMap
      a: #M0
      b: #M4
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            a: #M0
            b: #M4
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
      exported[(0, 0)] package:test/a.dart::@getter::b
    exportNamespace
      a: package:test/a.dart::@getter::a
      b: package:test/a.dart::@getter::b
''',
    );
  }

  test_dependency_export_topLevelVariable_add_combinators_hide_false() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart' hide b;
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              hide b
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
final b = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
      b: #M4
    declaredVariables
      a: #M1
      b: #M5
    exportMapId: #M6
    exportMap
      a: #M0
      b: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
    );
  }

  test_dependency_export_topLevelVariable_add_combinators_hide_true() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart' hide c;
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              hide c
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
final b = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
      b: #M4
    declaredVariables
      a: #M1
      b: #M5
    exportMapId: #M6
    exportMap
      a: #M0
      b: #M4
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: b
    expectedId: <null>
    actualId: #M4
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    exportMapId: #M7
    exportMap
      a: #M0
      b: #M4
    reExportMap
      a: #M0
      b: #M4
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              hide c
            a: #M0
            b: #M4
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
      exported[(0, 0)] package:test/a.dart::@getter::b
    exportNamespace
      a: package:test/a.dart::@getter::a
      b: package:test/a.dart::@getter::b
''',
    );
  }

  test_dependency_export_topLevelVariable_add_combinators_show_false() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart' show a;
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              show a
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
final b = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
      b: #M4
    declaredVariables
      a: #M1
      b: #M5
    exportMapId: #M6
    exportMap
      a: #M0
      b: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
    );
  }

  test_dependency_export_topLevelVariable_add_combinators_show_true() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart' show a, b;
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              show a, b
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
final b = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
      b: #M4
    declaredVariables
      a: #M1
      b: #M5
    exportMapId: #M6
    exportMap
      a: #M0
      b: #M4
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: b
    expectedId: <null>
    actualId: #M4
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    exportMapId: #M7
    exportMap
      a: #M0
      b: #M4
    reExportMap
      a: #M0
      b: #M4
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              show a, b
            a: #M0
            b: #M4
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
      exported[(0, 0)] package:test/a.dart::@getter::b
    exportNamespace
      a: package:test/a.dart::@getter::a
      b: package:test/a.dart::@getter::b
''',
    );
  }

  test_dependency_export_topLevelVariable_add_combinators_showHide_true() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart' show a, b hide c;
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              show a, b
              hide c
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
final b = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
      b: #M4
    declaredVariables
      a: #M1
      b: #M5
    exportMapId: #M6
    exportMap
      a: #M0
      b: #M4
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: b
    expectedId: <null>
    actualId: #M4
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    exportMapId: #M7
    exportMap
      a: #M0
      b: #M4
    reExportMap
      a: #M0
      b: #M4
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              show a, b
              hide c
            a: #M0
            b: #M4
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
      exported[(0, 0)] package:test/a.dart::@getter::b
    exportNamespace
      a: package:test/a.dart::@getter::a
      b: package:test/a.dart::@getter::b
''',
    );
  }

  test_dependency_export_topLevelVariable_add_private() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
final _b = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      _b: #M4
      a: #M0
    declaredVariables
      _b: #M5
      a: #M1
    exportMapId: #M6
    exportMap
      _b: #M4
      a: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
    );
  }

  test_dependency_export_topLevelVariable_remove() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
final b = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
      b: #M1
    declaredVariables
      a: #M2
      b: #M3
    exportMapId: #M4
    exportMap
      a: #M0
      b: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
    exportMap
      a: #M0
      b: #M1
    reExportMap
      a: #M0
      b: #M1
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            a: #M0
            b: #M1
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
      exported[(0, 0)] package:test/a.dart::@getter::b
    exportNamespace
      a: package:test/a.dart::@getter::a
      b: package:test/a.dart::@getter::b
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
    declaredVariables
      a: #M2
    exportMapId: #M6
    exportMap
      a: #M0
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportCountMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    expected: 2
    actual: 1
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    exportMapId: #M7
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            a: #M0
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
    );
  }

  test_dependency_export_topLevelVariable_remove_show_false() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
final b = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart' show a;
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
      b: #M1
    declaredVariables
      a: #M2
      b: #M3
    exportMapId: #M4
    exportMap
      a: #M0
      b: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            combinators
              show a
            a: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
    declaredVariables
      a: #M2
    exportMapId: #M6
    exportMap
      a: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
    exportNamespace
      a: package:test/a.dart::@getter::a
''',
    );
  }

  test_dependency_export_topLevelVariable_replace() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
final b = 0;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
      b: #M1
    declaredVariables
      a: #M2
      b: #M3
    exportMapId: #M4
    exportMap
      a: #M0
      b: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
    exportMap
      a: #M0
      b: #M1
    reExportMap
      a: #M0
      b: #M1
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            a: #M0
            b: #M1
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
      exported[(0, 0)] package:test/a.dart::@getter::b
    exportNamespace
      a: package:test/a.dart::@getter::a
      b: package:test/a.dart::@getter::b
''',
      updateFiles: () {
        modifyFile2(a, r'''
final a = 0;
final c = 0;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
      c: #M6
    declaredVariables
      a: #M2
      c: #M7
    exportMapId: #M8
    exportMap
      a: #M0
      c: #M6
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: c
    expectedId: <null>
    actualId: #M6
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    exportMapId: #M9
    exportMap
      a: #M0
      c: #M6
    reExportMap
      a: #M0
      c: #M6
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            a: #M0
            c: #M6
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@getter::a
      exported[(0, 0)] package:test/a.dart::@getter::c
    exportNamespace
      a: package:test/a.dart::@getter::a
      c: package:test/a.dart::@getter::c
''',
    );
  }

  test_dependency_export_topLevelVariable_type() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/b.dart', r'''
export 'a.dart';
''');

    // Uses exported `a`.
    newFile('$testPackageLibPath/test.dart', r'''
import 'b.dart';
final x = a;
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/b.dart
        exports
          package:test/a.dart
            a: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    declaredGetters
      x: #M4
    declaredVariables
      x: #M5
    exportMapId: #M6
    exportMap
      x: #M4
  requirements
    libraries
      package:test/b.dart
        exportMapId: #M3
        exportMap
          a: #M0
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T1
  library
    topLevelVariables
      final hasInitializer x
        type: int
    exportedReferences
      declared <testLibrary>::@getter::x
    exportNamespace
      x: <testLibrary>::@getter::x
''',
      // Change the initializer, now `double`.
      updateFiles: () {
        modifyFile2(a, r'''
final a = 1.2;
''');
        return [a];
      },
      // Linked, `x` has type `double`.
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    declaredGetters
      a: #M7
    declaredVariables
      a: #M8
    exportMapId: #M9
    exportMap
      a: #M7
  requirements
[operation] checkLinkedBundleRequirements
  package:test/b.dart
  exportIdMismatch
    fragmentUri: package:test/b.dart
    exportedUri: package:test/a.dart
    name: a
    expectedId: #M0
    actualId: #M7
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H4
    exportMapId: #M10
    exportMap
      a: #M7
    reExportMap
      a: #M7
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/b.dart
        exports
          package:test/a.dart
            a: #M7
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/b.dart
    name: a
    expectedId: #M0
    actualId: #M7
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H5
    declaredGetters
      x: #M11
    declaredVariables
      x: #M12
    exportMapId: #M13
    exportMap
      x: #M11
  requirements
    libraries
      package:test/b.dart
        exportMapId: #M10
        exportMap
          a: #M7
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T2
  library
    topLevelVariables
      final hasInitializer x
        type: double
    exportedReferences
      declared <testLibrary>::@getter::x
    exportNamespace
      x: <testLibrary>::@getter::x
''',
    );
  }

  test_dependency_export_typeAlias() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
typedef A = int;
typedef _A = int;
''');

    newFile('$testPackageLibPath/test.dart', r'''
export 'a.dart';
''');

    configuration.elementTextConfiguration.withExportScope = true;
    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredTypeAliases
      A: #M0
      _A: #M1
    exportMapId: #M2
    exportMap
      A: #M0
      _A: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      A: #M0
    reExportMap
      A: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            A: #M0
[status] idle
[future] getLibraryByUri T1
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@typeAlias::A
    exportNamespace
      A: package:test/a.dart::@typeAlias::A
''',
      updateFiles: () {
        modifyFile2(a, r'''
typedef A = int;
typedef _A2 = int;
''');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredTypeAliases
      A: #M0
      _A2: #M4
    exportMapId: #M5
    exportMap
      A: #M0
      _A2: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    exportedReferences
      exported[(0, 0)] package:test/a.dart::@typeAlias::A
    exportNamespace
      A: package:test/a.dart::@typeAlias::A
''',
    );
  }

  test_dependency_exportedExtensions_add_accessibleExtensions() async {
    _ManualRequirements.install((state) {
      state.singleUnit.libraryFragment.accessibleExtensions;
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
    exportMapId: #M1
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
extension E1 on Object {}
extension E2 on Object {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
      E2: #M4
    exportMapId: #M5
    exportMap
      E1: #M0
      E2: #M4
    exportedExtensions: #M0 #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  exportedExtensionsMismatch
    libraryUri: package:test/a.dart
    expectedIds: #M0
    actualIds: #M0 #M4
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M5
        exportedExtensions: #M0 #M4
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_exportedExtensions_add_accessibleExtensions_private() async {
    _ManualRequirements.install((state) {
      state.singleUnit.libraryFragment.accessibleExtensions;
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension E on Object {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
    exportMapId: #M1
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
extension E on Object {}
extension _E on Object {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
      _E: #M4
    exportMapId: #M5
    exportMap
      E: #M0
      _E: #M4
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_exportedExtensions_remove_accessibleExtensions() async {
    _ManualRequirements.install((state) {
      state.singleUnit.libraryFragment.accessibleExtensions;
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {}
extension E2 on Object {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
      E2: #M1
    exportMapId: #M2
    exportMap
      E1: #M0
      E2: #M1
    exportedExtensions: #M0 #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        exportedExtensions: #M0 #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
extension E1 on Object {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
    exportMapId: #M5
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  exportedExtensionsMismatch
    libraryUri: package:test/a.dart
    expectedIds: #M0 #M1
    actualIds: #M0
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M5
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_extension_instanceGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E on Object {
  int get foo => 0;
  set foo(int _) {}
  int get bar => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredGetters
          bar: #M3
          foo: #M4
        declaredSetters
          foo=: #M5
    exportMapId: #M6
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        instances
          E
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredGetters
              foo: #M4
            requestedDeclaredSetters
              foo=: #M5
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E on Object {
  int get foo => 0;
  set foo(int _) {}
  double get bar => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        declaredFields
          bar: #M10
          foo: #M2
        declaredGetters
          bar: #M11
          foo: #M4
        declaredSetters
          foo=: #M5
    exportMapId: #M6
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_instanceGetter_change_usedGetter() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E on Object {
  int get foo => 0;
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
    exportMapId: #M4
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M2
            requestedDeclaredSetters
              foo=: #M3
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E on Object {
  double get foo => 0;
  set foo(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        declaredFields
          foo: #M8
        declaredGetters
          foo: #M9
        declaredSetters
          foo=: #M3
    exportMapId: #M4
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E
            requestedDeclaredFields
              foo: #M8
            requestedDeclaredGetters
              foo: #M9
            requestedDeclaredSetters
              foo=: #M3
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_instanceGetter_change_usedGetter_private() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension A on Object {
  int get _foo => 0;
}
''',
      testCode: r'''
import 'a.dart';

extension B on Object {
  int get _foo => 0;
}

void f() {
  0._foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
        declaredFields
          _foo: #M1
        declaredGetters
          _foo: #M2
    exportMapId: #M3
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredExtensions
      B: #M4
        declaredFields
          _foo: #M5
        declaredGetters
          _foo: #M6
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      B: #M4
      f: #M7
    exportedExtensions: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          Object: <null>
          Object=: <null>
          int: <null>
          int=: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M3
        exportMap
          Object: <null>
          Object=: <null>
          int: <null>
          int=: <null>
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
extension A on Object {
  double get _foo => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
        declaredFields
          _foo: #M10
        declaredGetters
          _foo: #M11
    exportMapId: #M3
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_extension_instanceMethod_change_notUsedMemberName() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  int foo() {}
}
extension E2 on Object {
  int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
      E2: #M2
        declaredMethods
          bar: #M3
    exportMapId: #M4
    exportMap
      E1: #M0
      E2: #M2
    exportedExtensions: #M0 #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
          E2
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0 #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  int foo() {}
}
extension E2 on Object {
  double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
      E2: #M2
        declaredMethods
          bar: #M8
    exportMapId: #M4
    exportMap
      E1: #M0
      E2: #M2
    exportedExtensions: #M0 #M2
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_instanceMethod_change_usedMemberName() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  int foo() {}
}
extension E2 on Object {
  int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
      E2: #M2
        declaredMethods
          bar: #M3
    exportMapId: #M4
    exportMap
      E1: #M0
      E2: #M2
    exportedExtensions: #M0 #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
          E2
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0 #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  double foo() {}
}
extension E2 on Object {
  int bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M8
      E2: #M2
        declaredMethods
          bar: #M3
    exportMapId: #M4
    exportMap
      E1: #M0
      E2: #M2
    exportedExtensions: #M0 #M2
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E1
    methodName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M8
          E2
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0 #M2
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_instanceMethod_change_usedMemberName_multiImport() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E on Object {
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
import 'a.dart' as prefix;
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M2
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        instances
          E
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E on Object {
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        declaredMethods
          foo: #M6
    exportMapId: #M2
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E
    methodName: foo
    expectedId: #M1
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        instances
          E
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M6
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_instanceSetter_change_usedGetter() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E on Object {
  int get foo => 0;
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
    exportMapId: #M4
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M2
            requestedDeclaredSetters
              foo=: #M3
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E on Object {
  int get foo => 0;
  set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M8
    exportMapId: #M4
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E
    methodName: foo=
    expectedId: #M3
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M2
            requestedDeclaredSetters
              foo=: #M8
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_it_add_notUsedMemberName() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M2
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  void foo() {}
}
extension E2 on int {
  void bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
      E2: #M6
        declaredMethods
          bar: #M7
    exportMapId: #M8
    exportMap
      E1: #M0
      E2: #M6
    exportedExtensions: #M0 #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  exportedExtensionsMismatch
    libraryUri: package:test/a.dart
    expectedIds: #M0
    actualIds: #M0 #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M8
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
          E2
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0 #M6
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_it_add_private() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M2
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  void foo() {}
  void _bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          _bar: #M6
          foo: #M1
    exportMapId: #M2
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_it_add_usedMemberName() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M2
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  void foo() {}
}
extension E2 on int {
  void foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
      E2: #M6
        declaredMethods
          foo: #M7
    exportMapId: #M8
    exportMap
      E1: #M0
      E2: #M6
    exportedExtensions: #M0 #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  exportedExtensionsMismatch
    libraryUri: package:test/a.dart
    expectedIds: #M0
    actualIds: #M0 #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M8
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
          E2
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M7
        exportedExtensions: #M0 #M6
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_it_add_usedMemberName2() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E on Object {}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E: #M0
    exportMapId: #M1
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M2
    exportMapId: #M3
    exportMap
      f: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      32 +3 UNDEFINED_METHOD
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M1
        instances
          E
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      32 +3 UNDEFINED_METHOD
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
extension E on Object {
  void foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E: #M0
        declaredMethods
          foo: #M5
    exportMapId: #M1
    exportMap
      E: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E
    methodName: foo
    expectedId: <null>
    actualId: #M5
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M1
        instances
          E
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M5
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_it_remove_notUsedMemberName() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  void foo() {}
}
extension E2 on int {
  void bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
      E2: #M2
        declaredMethods
          bar: #M3
    exportMapId: #M4
    exportMap
      E1: #M0
      E2: #M2
    exportedExtensions: #M0 #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
          E2
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        exportedExtensions: #M0 #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  void foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M8
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelNotInstance
    libraryUri: package:test/a.dart
    name: E2
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M8
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_it_remove_usedMemberName() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  void foo() {}
}
extension E2 on int {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
      E2: #M2
        declaredMethods
          foo: #M3
    exportMapId: #M4
    exportMap
      E1: #M0
      E2: #M2
    exportedExtensions: #M0 #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
          E2
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M3
        exportedExtensions: #M0 #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  void foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M8
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelNotInstance
    libraryUri: package:test/a.dart
    name: E2
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M8
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_it_remove_usedMemberName_importWithPrefix() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  void foo() {}
}
extension E2 on int {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart' as prefix;
void f() {
  0.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
      E2: #M2
        declaredMethods
          foo: #M3
    exportMapId: #M4
    exportMap
      E1: #M0
      E2: #M2
    exportedExtensions: #M0 #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
          E2
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M3
        exportedExtensions: #M0 #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  void foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M8
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelNotInstance
    libraryUri: package:test/a.dart
    name: E2
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M8
        instances
          E1
            requestedDeclaredFields
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_operatorIndex_change_usedIndex() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  int operator[](int index) {}
  operator[]=(int index, int value) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0[0];
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          []: #M1
          []=: #M2
    exportMapId: #M3
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        instances
          E1
            requestedDeclaredMethods
              []: #M1
              []=: #M2
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  double operator[](int index) {}
  operator[]=(int index, int value) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          []: #M7
          []=: #M2
    exportMapId: #M3
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E1
    methodName: []
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        instances
          E1
            requestedDeclaredMethods
              []: #M7
              []=: #M2
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_operatorIndex_change_usedIndexEq() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  int operator[](int index) {}
  operator[]=(int index, int value) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0[0] = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          []: #M1
          []=: #M2
    exportMapId: #M3
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        instances
          E1
            requestedDeclaredMethods
              []: #M1
              []=: #M2
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  double operator[](int index) {}
  operator[]=(int index, int value) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          []: #M7
          []=: #M2
    exportMapId: #M3
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E1
    methodName: []
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        instances
          E1
            requestedDeclaredMethods
              []: #M7
              []=: #M2
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_operatorIndexEq_change_usedIndex() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  int operator[](int index) {}
  operator[]=(int index, int value) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0[0];
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          []: #M1
          []=: #M2
    exportMapId: #M3
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        instances
          E1
            requestedDeclaredMethods
              []: #M1
              []=: #M2
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  int operator[](int index) {}
  operator[]=(int index, double value) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          []: #M1
          []=: #M7
    exportMapId: #M3
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E1
    methodName: []=
    expectedId: #M2
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        instances
          E1
            requestedDeclaredMethods
              []: #M1
              []=: #M7
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_operatorIndexEq_change_usedIndexEq() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension E1 on Object {
  int operator[](int index) {}
  operator[]=(int index, int value) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  0[0] = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      E1: #M0
        declaredMethods
          []: #M1
          []=: #M2
    exportMapId: #M3
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        instances
          E1
            requestedDeclaredMethods
              []: #M1
              []=: #M2
        exportedExtensions: #M0
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension E1 on Object {
  int operator[](int index) {}
  operator[]=(int index, double value) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      E1: #M0
        declaredMethods
          []: #M1
          []=: #M7
    exportMapId: #M3
    exportMap
      E1: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: E1
    methodName: []=
    expectedId: #M2
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        instances
          E1
            requestedDeclaredMethods
              []: #M1
              []=: #M7
        exportedExtensions: #M0
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_staticGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {
  static int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
    exportMapId: #M3
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension A on int {
  static double get foo => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
        declaredFields
          foo: #M7
        declaredGetters
          foo: #M8
    exportMapId: #M3
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M7
            requestedDeclaredGetters
              foo: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_staticGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {
  static int get foo => 0;
  static int get bar => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredGetters
          bar: #M3
          foo: #M4
    exportMapId: #M5
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredGetters
              foo: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension A on int {
  static int get foo => 0;
  static double get bar => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
        declaredFields
          bar: #M9
          foo: #M2
        declaredGetters
          bar: #M10
          foo: #M4
    exportMapId: #M5
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_staticMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {
  static int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
        declaredMethods
          foo: #M1
    exportMapId: #M2
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension A on int {
  static double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
        declaredMethods
          foo: #M6
    exportMapId: #M2
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M6
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_staticMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {
  static int foo() {}
  static int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
        declaredMethods
          bar: #M1
          foo: #M2
    exportMapId: #M3
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension A on int {
  static int foo() {}
  static double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
        declaredMethods
          bar: #M7
          foo: #M2
    exportMapId: #M3
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_staticSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {
  static set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
    exportMapId: #M3
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredSetters
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension A on int {
  static set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
        declaredFields
          foo: #M7
        declaredSetters
          foo=: #M8
    exportMapId: #M3
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M7
            requestedDeclaredSetters
              foo=: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extension_staticSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {
  static set foo(int _) {}
  static set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredSetters
          bar=: #M3
          foo=: #M4
    exportMapId: #M5
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredSetters
              foo=: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension A on int {
  static set foo(int _) {}
  static set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
        declaredFields
          bar: #M9
          foo: #M2
        declaredSetters
          bar=: #M10
          foo=: #M4
    exportMapId: #M5
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_instanceGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          foo: #M1
          it: #M2
        declaredGetters
          foo: #M3
          it: #M4
        interface: #M5
          map
            foo: #M3
            it: #M4
          implemented
            foo: #M3
            it: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  double get foo => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          foo: #M10
          it: #M2
        declaredGetters
          foo: #M11
          it: #M4
        interface: #M12
          map
            foo: #M11
            it: #M4
          implemented
            foo: #M11
            it: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M10
        interfaces
          A
            methods
              foo: #M11
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_instanceGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  int get foo => 0;
  int get bar => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
          it: #M3
        declaredGetters
          bar: #M4
          foo: #M5
          it: #M6
        interface: #M7
          map
            bar: #M4
            foo: #M5
            it: #M6
          implemented
            bar: #M4
            foo: #M5
            it: #M6
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M9
    exportMapId: #M10
    exportMap
      f: #M9
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M11
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
        interfaces
          A
            methods
              foo: #M5
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  int get foo => 0;
  double get bar => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          bar: #M12
          foo: #M2
          it: #M3
        declaredGetters
          bar: #M13
          foo: #M5
          it: #M6
        interface: #M14
          map
            bar: #M13
            foo: #M5
            it: #M6
          implemented
            bar: #M13
            foo: #M5
            it: #M6
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_instanceMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          foo: #M3
        interface: #M4
          map
            foo: #M3
            it: #M2
          implemented
            foo: #M3
            it: #M2
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          foo: #M9
        interface: #M10
          map
            foo: #M9
            it: #M2
          implemented
            foo: #M9
            it: #M2
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M3
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_instanceMethod_change_lookUpMethod_inherited() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.thisType.lookUpMethod(
        'foo',
        A.library,
        concrete: true,
        inherited: true,
      );
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          foo: #M3
        interface: #M4
          map
            foo: #M3
            it: #M2
          implemented
            foo: #M3
            it: #M2
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
''',
      updatedA: r'''
extension type A(int it) {
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          foo: #M8
        interface: #M9
          map
            foo: #M8
            it: #M2
          implemented
            foo: #M8
            it: #M2
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_extensionType_instanceMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  int foo() {}
  int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          bar: #M3
          foo: #M4
        interface: #M5
          map
            bar: #M3
            foo: #M4
            it: #M2
          implemented
            bar: #M3
            foo: #M4
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  int foo() {}
  double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          bar: #M10
          foo: #M4
        interface: #M11
          map
            bar: #M10
            foo: #M4
            it: #M2
          implemented
            bar: #M10
            foo: #M4
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_instanceSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          foo: #M1
          it: #M2
        declaredGetters
          it: #M3
        declaredSetters
          foo=: #M4
        interface: #M5
          map
            foo=: #M4
            it: #M3
          implemented
            foo=: #M4
            it: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo=: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          foo: #M10
          it: #M2
        declaredGetters
          it: #M3
        declaredSetters
          foo=: #M11
        interface: #M12
          map
            foo=: #M11
            it: #M3
          implemented
            foo=: #M11
            it: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M10
        interfaces
          A
            methods
              foo=: #M11
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_instanceSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  set foo(int _) {}
  set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
          it: #M3
        declaredGetters
          it: #M4
        declaredSetters
          bar=: #M5
          foo=: #M6
        interface: #M7
          map
            bar=: #M5
            foo=: #M6
            it: #M4
          implemented
            bar=: #M5
            foo=: #M6
            it: #M4
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M9
    exportMapId: #M10
    exportMap
      f: #M9
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M11
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
        interfaces
          A
            methods
              foo=: #M6
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  set foo(int _) {}
  set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          bar: #M12
          foo: #M2
          it: #M3
        declaredGetters
          it: #M4
        declaredSetters
          bar=: #M13
          foo=: #M6
        interface: #M14
          map
            bar=: #M13
            foo=: #M6
            it: #M4
          implemented
            bar=: #M13
            foo=: #M6
            it: #M4
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_namedConstructor_change_getConstructors() async {
    configuration.includeDefaultConstructors();

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.constructors;
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  factory A.named(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          named: #M3
          new: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M4 #M3
            allInheritedConstructors: []
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
extension type A(int it) {
  factory A.named(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          named: #M9
          new: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceChildrenIdsMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    childrenPropertyName: declaredConstructors
    expectedIds: #M4 #M3
    actualIds: #M4 #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M4 #M9
            allInheritedConstructors: []
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_extensionType_namedConstructor_change_invoked() async {
    configuration.includeDefaultConstructors();
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  factory A.named(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.named(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          named: #M3
          new: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              named: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  factory A.named(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          named: #M10
          new: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: named
    expectedId: #M3
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              named: #M10
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_namedConstructor_change_notUsed() async {
    configuration.includeDefaultConstructors();
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  factory A.named(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          named: #M3
          new: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              new: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  factory A.named(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          named: #M10
          new: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_staticGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  static int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          foo: #M1
          it: #M2
        declaredGetters
          foo: #M3
          it: #M4
        interface: #M5
          map
            it: #M4
          implemented
            it: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M3
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  static double get foo => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          foo: #M10
          it: #M2
        declaredGetters
          foo: #M11
          it: #M4
        interface: #M5
          map
            it: #M4
          implemented
            it: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M10
            requestedDeclaredGetters
              foo: #M11
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_staticGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  static int get foo => 0;
  static int get bar => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
          it: #M3
        declaredGetters
          bar: #M4
          foo: #M5
          it: #M6
        interface: #M7
          map
            it: #M6
          implemented
            it: #M6
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M9
    exportMapId: #M10
    exportMap
      f: #M9
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M11
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredGetters
              foo: #M5
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  static int get foo => 0;
  static double get bar => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          bar: #M12
          foo: #M2
          it: #M3
        declaredGetters
          bar: #M13
          foo: #M5
          it: #M6
        interface: #M7
          map
            it: #M6
          implemented
            it: #M6
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_staticMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  static int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          foo: #M3
        interface: #M4
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M3
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  static double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          foo: #M9
        interface: #M4
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M3
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M9
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_staticMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  static int foo() {}
  static int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          bar: #M3
          foo: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M4
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  static int foo() {}
  static double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredMethods
          bar: #M10
          foo: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_staticSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  static set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          foo: #M1
          it: #M2
        declaredGetters
          it: #M3
        declaredSetters
          foo=: #M4
        interface: #M5
          map
            it: #M3
          implemented
            it: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredSetters
              foo=: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  static set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          foo: #M10
          it: #M2
        declaredGetters
          it: #M3
        declaredSetters
          foo=: #M11
        interface: #M5
          map
            it: #M3
          implemented
            it: #M3
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M10
            requestedDeclaredSetters
              foo=: #M11
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_staticSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  static set foo(int _) {}
  static set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
          it: #M3
        declaredGetters
          it: #M4
        declaredSetters
          bar=: #M5
          foo=: #M6
        interface: #M7
          map
            it: #M4
          implemented
            it: #M4
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M9
    exportMapId: #M10
    exportMap
      f: #M9
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M11
        exportMapId: #M8
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredSetters
              foo=: #M6
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(int it) {
  static set foo(int _) {}
  static set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          bar: #M12
          foo: #M2
          it: #M3
        declaredGetters
          it: #M4
        declaredSetters
          bar=: #M13
          foo=: #M6
        interface: #M7
          map
            it: #M4
          implemented
            it: #M4
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_unnamedConstructor_change_getConstructors() async {
    configuration.includeDefaultConstructors();

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.constructors;
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  factory A.named(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          named: #M3
          new: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M4 #M3
            allInheritedConstructors: []
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
extension type A(double it) {
  factory A.named(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M9
        declaredFields
          it: #M10
        declaredGetters
          it: #M11
        declaredConstructors
          named: #M12
          new: #M13
        interface: #M14
          map
            it: #M11
          implemented
            it: #M11
    exportMapId: #M15
    exportMap
      A: #M9
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M15
        exportMap
          A: #M9
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            allDeclaredConstructors: #M13 #M12
            allInheritedConstructors: []
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_extensionType_unnamedConstructor_change_invoked() async {
    configuration.includeDefaultConstructors();
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          new: #M3
        interface: #M4
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M5
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              new: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(double it) {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M9
        declaredFields
          it: #M10
        declaredGetters
          it: #M11
        declaredConstructors
          new: #M12
        interface: #M13
          map
            it: #M11
          implemented
            it: #M11
    exportMapId: #M14
    exportMap
      A: #M9
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M14
        exportMap
          A: #M9
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              new: #M12
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_extensionType_unnamedConstructor_change_notUsed() async {
    configuration.includeDefaultConstructors();
    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {
  factory A.named(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.named(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        declaredConstructors
          named: #M3
          new: #M4
        interface: #M5
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              named: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
extension type A(double it) {
  factory A.named(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M10
        declaredFields
          it: #M11
        declaredGetters
          it: #M12
        declaredConstructors
          named: #M13
          new: #M14
        interface: #M15
          map
            it: #M12
          implemented
            it: #M12
    exportMapId: #M16
    exportMap
      A: #M10
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M16
        exportMap
          A: #M10
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            requestedConstructors
              named: #M13
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_library_exportedDeprecated_class_falseToTrue() async {
    configuration
      ..withStreamResolvedUnitResults = false
      ..withElementManifests = true;

    newFile('$testPackageLibPath/x.dart', r'''
class A {}
''');

    await _runChangeScenarioTA(
      initialA: r'''
library;
export 'x.dart';
''',
      testCode: r'''
import 'a.dart';
void f() {
  A;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/x.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        supertype: Object @ dart:core
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      A: #M0
    reExportMap
      A: #M0
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            A: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    declaredFunctions
      f: #M4
        flags: hasEnclosingTypeParameterReference isSimplyBounded isStatic
        functionType: FunctionType
          returnType: void
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
library;
@deprecated
export 'x.dart';
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    exportMapId: #M3
    exportMap
      A: #M0
    reExportMap
      A: #M0
    reExportDeprecatedOnly: A
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            A: #M0
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  reExportDeprecatedOnlyMismatch
    libraryUri: package:test/a.dart
    name: A
    expected: false
    actual: true
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: true
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      30 +1 DEPRECATED_EXPORT_USE
''',
    );
  }

  test_dependency_library_exportedDeprecated_class_trueToFalse() async {
    configuration
      ..withStreamResolvedUnitResults = false
      ..withElementManifests = true;

    newFile('$testPackageLibPath/x.dart', r'''
class A {}
''');

    await _runChangeScenarioTA(
      initialA: r'''
library;
@deprecated
export 'x.dart';
''',
      testCode: r'''
import 'a.dart';
void f() {
  A;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/x.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        supertype: Object @ dart:core
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      A: #M0
    reExportMap
      A: #M0
    reExportDeprecatedOnly: A
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            A: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    declaredFunctions
      f: #M4
        flags: hasEnclosingTypeParameterReference isSimplyBounded isStatic
        functionType: FunctionType
          returnType: void
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: true
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      30 +1 DEPRECATED_EXPORT_USE
''',
      updatedA: r'''
library;
@deprecated
export 'x.dart';
export 'x.dart';
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    exportMapId: #M3
    exportMap
      A: #M0
    reExportMap
      A: #M0
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            A: #M0
          package:test/x.dart
            A: #M0
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  reExportDeprecatedOnlyMismatch
    libraryUri: package:test/a.dart
    name: A
    expected: true
    actual: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_library_exportedDeprecated_setter_falseToTrue() async {
    configuration
      ..withStreamResolvedUnitResults = false
      ..withElementManifests = true;

    newFile('$testPackageLibPath/x.dart', r'''
set foo(int _) {}
''');

    await _runChangeScenarioTA(
      initialA: r'''
library;
export 'x.dart';
''',
      testCode: r'''
import 'a.dart';
void f() {
  foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/x.dart
    hashForRequirements: #H0
    declaredSetters
      foo=: #M0
        flags: hasImplicitReturnType isSimplyBounded isStatic
        functionType: FunctionType
          positional
            required int @ dart:core
          returnType: void
    declaredVariables
      foo: #M1
        flags: isStatic isSynthetic shouldUseTypeForInitializerInference
        type: int @ dart:core
    exportMapId: #M2
    exportMap
      foo=: #M0
  requirements
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      foo=: #M0
    reExportMap
      foo=: #M0
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            foo=: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    declaredFunctions
      f: #M4
        flags: hasEnclosingTypeParameterReference isSimplyBounded isStatic
        functionType: FunctionType
          returnType: void
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          foo: <null>
          foo=: #M0
        reExportDeprecatedOnly
          foo=: false
      package:test/x.dart
        libraryMetadataId: #M7
        exportMapId: #M2
        requestedDeclaredVariables
          foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
library;
@deprecated
export 'x.dart';
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    exportMapId: #M3
    exportMap
      foo=: #M0
    reExportMap
      foo=: #M0
    reExportDeprecatedOnly: foo=
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            foo=: #M0
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  reExportDeprecatedOnlyMismatch
    libraryUri: package:test/a.dart
    name: foo=
    expected: false
    actual: true
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          foo: <null>
          foo=: #M0
        reExportDeprecatedOnly
          foo=: true
      package:test/x.dart
        libraryMetadataId: #M7
        exportMapId: #M2
        requestedDeclaredVariables
          foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      30 +3 DEPRECATED_EXPORT_USE
''',
    );
  }

  test_dependency_library_exportedDeprecated_setter_trueToFalse() async {
    configuration
      ..withStreamResolvedUnitResults = false
      ..withElementManifests = true;

    newFile('$testPackageLibPath/x.dart', r'''
set foo(int _) {}
''');

    await _runChangeScenarioTA(
      initialA: r'''
library;
@deprecated
export 'x.dart';
''',
      testCode: r'''
import 'a.dart';
void f() {
  foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/x.dart
    hashForRequirements: #H0
    declaredSetters
      foo=: #M0
        flags: hasImplicitReturnType isSimplyBounded isStatic
        functionType: FunctionType
          positional
            required int @ dart:core
          returnType: void
    declaredVariables
      foo: #M1
        flags: isStatic isSynthetic shouldUseTypeForInitializerInference
        type: int @ dart:core
    exportMapId: #M2
    exportMap
      foo=: #M0
  requirements
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      foo=: #M0
    reExportMap
      foo=: #M0
    reExportDeprecatedOnly: foo=
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            foo=: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    declaredFunctions
      f: #M4
        flags: hasEnclosingTypeParameterReference isSimplyBounded isStatic
        functionType: FunctionType
          returnType: void
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          foo: <null>
          foo=: #M0
        reExportDeprecatedOnly
          foo=: true
      package:test/x.dart
        libraryMetadataId: #M7
        exportMapId: #M2
        requestedDeclaredVariables
          foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      30 +3 DEPRECATED_EXPORT_USE
''',
      updatedA: r'''
library;
@deprecated
export 'x.dart';
export 'x.dart';
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    exportMapId: #M3
    exportMap
      foo=: #M0
    reExportMap
      foo=: #M0
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            foo=: #M0
          package:test/x.dart
            foo=: #M0
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  reExportDeprecatedOnlyMismatch
    libraryUri: package:test/a.dart
    name: foo=
    expected: true
    actual: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          foo: <null>
          foo=: #M0
        reExportDeprecatedOnly
          foo=: false
      package:test/x.dart
        libraryMetadataId: #M7
        exportMapId: #M2
        requestedDeclaredVariables
          foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_library_exportedTopLevels_class_remove() async {
    configuration
      ..withStreamResolvedUnitResults = false
      ..withElementManifests = true;

    newFile('$testPackageLibPath/x.dart', r'''
class A {}
''');

    await _runChangeScenarioTA(
      initialA: r'''
library;
export 'x.dart';
''',
      testCode: r'''
import 'a.dart';
void f() {
  A;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/x.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        supertype: Object @ dart:core
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      A: #M0
    reExportMap
      A: #M0
    exportedLibraryUris: package:test/x.dart
  requirements
    exportRequirements
      package:test/a.dart
        exports
          package:test/x.dart
            A: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    declaredFunctions
      f: #M4
        flags: hasEnclosingTypeParameterReference isSimplyBounded isStatic
        functionType: FunctionType
          returnType: void
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
library;
// export 'x.dart';
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    exportMapId: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M7
        exportMap
          A: <null>
          A=: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      30 +1 UNDEFINED_IDENTIFIER
''',
    );
  }

  test_dependency_libraryElement_allClasses_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.classes;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class B {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        allDeclaredClasses: #M0 #M2
[status] idle
''',
      updatedA: r'''
class A {}
class C {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      C: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      C: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: classes
    expectedIds: #M0 #M2
    actualIds: #M0 #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M9
        allDeclaredClasses: #M0 #M7
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allEnums_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.enums;
    });

    await _runChangeScenarioTA(
      initialA: r'''
enum A { a }
enum B { b }
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          a: #M1
          values: #M2
        declaredGetters
          a: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
      B: #M7
        declaredFields
          b: #M8
          values: #M9
        declaredGetters
          b: #M10
          values: #M11
        interface: #M12
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M13
    exportMap
      A: #M0
      B: #M7
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M14
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M15
        exportMapId: #M13
        allDeclaredEnums: #M0 #M7
[status] idle
''',
      updatedA: r'''
enum A { a }
enum C { c }
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          a: #M1
          values: #M2
        declaredGetters
          a: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
      C: #M16
        declaredFields
          c: #M17
          values: #M18
        declaredGetters
          c: #M19
          values: #M20
        interface: #M21
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M22
    exportMap
      A: #M0
      C: #M16
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: enums
    expectedIds: #M0 #M7
    actualIds: #M0 #M16
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M15
        exportMapId: #M22
        allDeclaredEnums: #M0 #M16
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allExtensions_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.extensions;
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {}
extension B on int {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
      B: #M1
    exportMapId: #M2
    exportMap
      A: #M0
      B: #M1
    exportedExtensions: #M0 #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        allDeclaredExtensions: #M0 #M1
[status] idle
''',
      updatedA: r'''
extension A on int {}
extension C on int {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
      C: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      C: #M5
    exportedExtensions: #M0 #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: extensions
    expectedIds: #M0 #M1
    actualIds: #M0 #M5
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M6
        allDeclaredExtensions: #M0 #M5
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allExtensionTypes_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.extensionTypes;
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {}
extension type B(int it) {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        interface: #M3
          map
            it: #M2
          implemented
            it: #M2
      B: #M4
        declaredFields
          it: #M5
        declaredGetters
          it: #M6
        interface: #M7
          map
            it: #M6
          implemented
            it: #M6
    exportMapId: #M8
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M9
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M8
        allDeclaredExtensionTypes: #M0 #M4
[status] idle
''',
      updatedA: r'''
extension type A(int it) {}
extension type C(int it) {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        interface: #M3
          map
            it: #M2
          implemented
            it: #M2
      C: #M11
        declaredFields
          it: #M12
        declaredGetters
          it: #M13
        interface: #M14
          map
            it: #M13
          implemented
            it: #M13
    exportMapId: #M15
    exportMap
      A: #M0
      C: #M11
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: extensionTypes
    expectedIds: #M0 #M4
    actualIds: #M0 #M11
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M15
        allDeclaredExtensionTypes: #M0 #M11
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allGetters_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getters;
    });

    await _runChangeScenarioTA(
      initialA: r'''
int get foo => 0;
int get bar => 1;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      bar: #M0
      foo: #M1
    declaredVariables
      bar: #M2
      foo: #M3
    exportMapId: #M4
    exportMap
      bar: #M0
      foo: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        allDeclaredGetters: #M1 #M0
[status] idle
''',
      updatedA: r'''
int get foo => 0;
int get baz => 1;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      baz: #M7
      foo: #M1
    declaredVariables
      baz: #M8
      foo: #M3
    exportMapId: #M9
    exportMap
      baz: #M7
      foo: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: getters
    expectedIds: #M1 #M0
    actualIds: #M1 #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M9
        allDeclaredGetters: #M1 #M7
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allMixins_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.mixins;
    });

    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
mixin B {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        allDeclaredMixins: #M0 #M2
[status] idle
''',
      updatedA: r'''
mixin A {}
mixin C {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        interface: #M1
      C: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      C: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: mixins
    expectedIds: #M0 #M2
    actualIds: #M0 #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M9
        allDeclaredMixins: #M0 #M7
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allSetters_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.setters;
    });

    await _runChangeScenarioTA(
      initialA: r'''
set foo(int _) {}
set bar(int _) {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredSetters
      bar=: #M0
      foo=: #M1
    declaredVariables
      bar: #M2
      foo: #M3
    exportMapId: #M4
    exportMap
      bar=: #M0
      foo=: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        allDeclaredSetters: #M1 #M0
[status] idle
''',
      updatedA: r'''
set foo(int _) {}
set baz(int _) {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredSetters
      baz=: #M7
      foo=: #M1
    declaredVariables
      baz: #M8
      foo: #M3
    exportMapId: #M9
    exportMap
      baz=: #M7
      foo=: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: setters
    expectedIds: #M1 #M0
    actualIds: #M1 #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M9
        allDeclaredSetters: #M1 #M7
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allTopLevelFunctions_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.topLevelFunctions;
    });

    await _runChangeScenarioTA(
      initialA: r'''
void foo() {}
void bar() {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredFunctions
      bar: #M0
      foo: #M1
    exportMapId: #M2
    exportMap
      bar: #M0
      foo: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        allDeclaredFunctions: #M1 #M0
[status] idle
''',
      updatedA: r'''
void foo() {}
void baz() {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredFunctions
      baz: #M5
      foo: #M1
    exportMapId: #M6
    exportMap
      baz: #M5
      foo: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: topLevelFunctions
    expectedIds: #M1 #M0
    actualIds: #M1 #M5
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M6
        allDeclaredFunctions: #M1 #M5
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allTopLevelVariables_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.topLevelVariables;
    });

    await _runChangeScenarioTA(
      initialA: r'''
var foo = 0;
var bar = 1;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      bar: #M0
      foo: #M1
    declaredSetters
      bar=: #M2
      foo=: #M3
    declaredVariables
      bar: #M4
      foo: #M5
    exportMapId: #M6
    exportMap
      bar: #M0
      bar=: #M2
      foo: #M1
      foo=: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        allDeclaredVariables: #M5 #M4
[status] idle
''',
      updatedA: r'''
var foo = 0;
var baz = 1;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      baz: #M9
      foo: #M1
    declaredSetters
      baz=: #M10
      foo=: #M3
    declaredVariables
      baz: #M11
      foo: #M5
    exportMapId: #M12
    exportMap
      baz: #M9
      baz=: #M10
      foo: #M1
      foo=: #M3
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: topLevelVariables
    expectedIds: #M5 #M4
    actualIds: #M5 #M11
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M12
        allDeclaredVariables: #M5 #M11
[status] idle
''',
    );
  }

  test_dependency_libraryElement_allTypeAliases_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.typeAliases;
    });

    await _runChangeScenarioTA(
      initialA: r'''
typedef A = int;
typedef B = int;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredTypeAliases
      A: #M0
      B: #M1
    exportMapId: #M2
    exportMap
      A: #M0
      B: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        allDeclaredTypeAliases: #M0 #M1
[status] idle
''',
      updatedA: r'''
typedef A = int;
typedef C = int;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredTypeAliases
      A: #M0
      C: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      C: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryChildrenIdsMismatch
    libraryUri: package:test/a.dart
    childrenPropertyName: typeAliases
    expectedIds: #M0 #M1
    actualIds: #M0 #M5
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M6
        allDeclaredTypeAliases: #M0 #M5
[status] idle
''',
    );
  }

  test_dependency_libraryElement_entryPoint() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.entryPoint;
    });

    await _runChangeScenarioTA(
      initialA: r'''
void main() {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredFunctions
      main: #M0
    exportMapId: #M1
    exportMap
      main: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M1
        exportMap
          main: #M0
[status] idle
''',
      updatedA: r'''
@deprecated
void main() {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredFunctions
      main: #M4
    exportMapId: #M5
    exportMap
      main: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: main
    expectedId: #M0
    actualId: #M4
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M5
        exportMap
          main: #M4
[status] idle
''',
    );
  }

  test_dependency_libraryElement_exportedLibraries() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.exportedLibraries;
    });

    await _runChangeScenarioTA(
      initialA: r'''
export 'dart:math' show min;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    exportMapId: #M0
    exportMap
      min: #M1
    reExportMap
      min: #M1
    exportedLibraryUris: dart:math
  requirements
    exportRequirements
      package:test/a.dart
        exports
          dart:math
            combinators
              show min
            min: #M1
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M0
        exportedLibraryUris: dart:math
[status] idle
''',
      updatedA: r'''
export 'dart:io' show exit;
export 'dart:math' show min;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    exportMapId: #M4
    exportMap
      exit: #M5
      min: #M1
    reExportMap
      exit: #M5
      min: #M1
    exportedLibraryUris: dart:io dart:math
  requirements
    exportRequirements
      package:test/a.dart
        exports
          dart:io
            combinators
              show exit
            exit: #M5
          dart:math
            combinators
              show min
            min: #M1
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryExportedUrisMismatch
    libraryUri: package:test/a.dart
    expected: dart:math
    actual: dart:io dart:math
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M4
        exportedLibraryUris: dart:io dart:math
[status] idle
''',
    );
  }

  test_dependency_libraryElement_exportNamespace_ambiguousExport_add() async {
    configuration.withStreamResolvedUnitResults = false;

    newFile('$testPackageLibPath/b.dart', r'''
class B {}
''');

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
export 'a.dart';
export 'b.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M3
        interface: #M4
    exportMapId: #M5
    exportMap
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M3
    reExportMap
      A: #M0
      B: #M3
    exportedLibraryUris: package:test/a.dart package:test/b.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            A: #M0
          package:test/b.dart
            B: #M3
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M2
        exportMap
          A: #M0
      package:test/b.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    declaredClasses
      B: #M9
        interface: #M10
    exportMapId: #M11
    exportMap
      B: #M9
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: B
    expectedId: <null>
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H4
    exportMapId: #M12
    exportMap
      B: #M9
    reExportMap
      B: #M9
    exportedLibraryUris: package:test/a.dart package:test/b.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            B: #M9
          package:test/b.dart
            B: #M3
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M11
        exportMap
          B: #M9
      package:test/b.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      24 +8 AMBIGUOUS_EXPORT
''',
    );
  }

  test_dependency_libraryElement_exportNamespace_ambiguousExport_remove() async {
    configuration.withStreamResolvedUnitResults = false;

    newFile('$testPackageLibPath/b.dart', r'''
class B {}
''');

    await _runChangeScenarioTA(
      initialA: r'''
class B {}
''',
      testCode: r'''
export 'a.dart';
export 'b.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      B: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      B: #M0
  requirements
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M3
        interface: #M4
    exportMapId: #M5
    exportMap
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    exportMapId: #M6
    exportMap
      B: #M0
    reExportMap
      B: #M0
    exportedLibraryUris: package:test/a.dart package:test/b.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            B: #M0
          package:test/b.dart
            B: #M3
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M2
        exportMap
          B: #M0
      package:test/b.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      24 +8 AMBIGUOUS_EXPORT
''',
      updatedA: r'''
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    declaredClasses
      A: #M9
        interface: #M10
    exportMapId: #M11
    exportMap
      A: #M9
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  exportIdMismatch
    fragmentUri: package:test/test.dart
    exportedUri: package:test/a.dart
    name: A
    expectedId: <null>
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H4
    exportMapId: #M12
    exportMap
      A: #M9
      B: #M3
    reExportMap
      A: #M9
      B: #M3
    exportedLibraryUris: package:test/a.dart package:test/b.dart
  requirements
    exportRequirements
      package:test/test.dart
        exports
          package:test/a.dart
            A: #M9
          package:test/b.dart
            B: #M3
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M11
        exportMap
          A: #M9
      package:test/b.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_libraryElement_exportNamespace_definedNames() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.exportNamespace.definedNames2;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class B {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          B: #M2
[status] idle
''',
      updatedA: r'''
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M7
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M2
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M7
        exportMap
          A: #M0
[status] idle
''',
    );
  }

  test_dependency_libraryElement_exportNamespace_get_class() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.exportNamespace.get2('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class B {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
''',
      updatedA: r'''
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M7
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_exportNamespace_get_setter() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.exportNamespace.get2('foo=');
    });

    await _runChangeScenarioTA(
      initialA: r'''
void set foo(int _) {}
void set bar(int _) {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredSetters
      bar=: #M0
      foo=: #M1
    declaredVariables
      bar: #M2
      foo: #M3
    exportMapId: #M4
    exportMap
      bar=: #M0
      foo=: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        exportMap
          foo: <null>
          foo=: #M1
        reExportDeprecatedOnly
          foo=: false
[status] idle
''',
      updatedA: r'''
void set foo(int _) {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredSetters
      foo=: #M1
    declaredVariables
      foo: #M3
    exportMapId: #M7
    exportMap
      foo=: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_featureSet() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.featureSet;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M4
        exportMapId: #M2
[status] idle
''',
      updatedA: r'''
// @dart = 2.19
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryFeatureSetMismatch
    libraryUri: package:test/a.dart
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M4
        exportMapId: #M2
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getClass_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getClass('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        requestedDeclaredClasses
          A: #M0
[status] idle
''',
      updatedA: r'''
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      B: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      B: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M7
        requestedDeclaredClasses
          A: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getClass_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getClass('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
class B {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        requestedDeclaredClasses
          A: #M0
[status] idle
''',
      updatedA: r'''
class A {}
class C {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      C: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      C: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getEnum_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getEnum('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
enum A { a }
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          a: #M1
          values: #M2
        declaredGetters
          a: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M7
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M8
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M7
        requestedDeclaredEnums
          A: #M0
[status] idle
''',
      updatedA: r'''
enum B { b }
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      B: #M10
        declaredFields
          b: #M11
          values: #M12
        declaredGetters
          b: #M13
          values: #M14
        interface: #M15
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M16
    exportMap
      B: #M10
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M16
        requestedDeclaredEnums
          A: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getEnum_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getEnum('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
enum A { a }
enum B { b }
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredEnums
      A: #M0
        declaredFields
          a: #M1
          values: #M2
        declaredGetters
          a: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
      B: #M7
        declaredFields
          b: #M8
          values: #M9
        declaredGetters
          b: #M10
          values: #M11
        interface: #M12
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M13
    exportMap
      A: #M0
      B: #M7
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M14
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M15
        exportMapId: #M13
        requestedDeclaredEnums
          A: #M0
[status] idle
''',
      updatedA: r'''
enum A { a }
enum C { c }
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredEnums
      A: #M0
        declaredFields
          a: #M1
          values: #M2
        declaredGetters
          a: #M3
          values: #M4
        interface: #M5
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
      C: #M16
        declaredFields
          c: #M17
          values: #M18
        declaredGetters
          c: #M19
          values: #M20
        interface: #M21
          map
            index: #M6
          implemented
            index: #M6
          superImplemented
            [0]
              index: #M6
          inherited
            index: #M6
    exportMapId: #M22
    exportMap
      A: #M0
      C: #M16
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getExtension_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getExtension('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
    exportMapId: #M1
    exportMap
      A: #M0
    exportedExtensions: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M1
        requestedDeclaredExtensions
          A: #M0
[status] idle
''',
      updatedA: r'''
extension B on int {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      B: #M4
    exportMapId: #M5
    exportMap
      B: #M4
    exportedExtensions: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M5
        requestedDeclaredExtensions
          A: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getExtension_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getExtension('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension A on int {}
extension B on int {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensions
      A: #M0
      B: #M1
    exportMapId: #M2
    exportMap
      A: #M0
      B: #M1
    exportedExtensions: #M0 #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        requestedDeclaredExtensions
          A: #M0
[status] idle
''',
      updatedA: r'''
extension A on int {}
extension C on int {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensions
      A: #M0
      C: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      C: #M5
    exportedExtensions: #M0 #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getExtensionType_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getExtensionType('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        interface: #M3
          map
            it: #M2
          implemented
            it: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        requestedDeclaredExtensionTypes
          A: #M0
[status] idle
''',
      updatedA: r'''
extension type B(int it) {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      B: #M7
        declaredFields
          it: #M8
        declaredGetters
          it: #M9
        interface: #M10
          map
            it: #M9
          implemented
            it: #M9
    exportMapId: #M11
    exportMap
      B: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M11
        requestedDeclaredExtensionTypes
          A: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getExtensionType_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getExtensionType('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
extension type A(int it) {}
extension type B(int it) {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        interface: #M3
          map
            it: #M2
          implemented
            it: #M2
      B: #M4
        declaredFields
          it: #M5
        declaredGetters
          it: #M6
        interface: #M7
          map
            it: #M6
          implemented
            it: #M6
    exportMapId: #M8
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M9
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M10
        exportMapId: #M8
        requestedDeclaredExtensionTypes
          A: #M0
[status] idle
''',
      updatedA: r'''
extension type A(int it) {}
extension type C(int it) {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredExtensionTypes
      A: #M0
        declaredFields
          it: #M1
        declaredGetters
          it: #M2
        interface: #M3
          map
            it: #M2
          implemented
            it: #M2
      C: #M11
        declaredFields
          it: #M12
        declaredGetters
          it: #M13
        interface: #M14
          map
            it: #M13
          implemented
            it: #M13
    exportMapId: #M15
    exportMap
      A: #M0
      C: #M11
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getGetter_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getGetter('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
int get foo => 0;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      foo: #M0
    declaredVariables
      foo: #M1
    exportMapId: #M2
    exportMap
      foo: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        requestedDeclaredGetters
          foo: #M0
[status] idle
''',
      updatedA: r'''
int get bar => 0;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      bar: #M5
    declaredVariables
      bar: #M6
    exportMapId: #M7
    exportMap
      bar: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: foo
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M7
        requestedDeclaredGetters
          foo: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getGetter_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getGetter('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
int get foo => 0;
int get bar => 1;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      bar: #M0
      foo: #M1
    declaredVariables
      bar: #M2
      foo: #M3
    exportMapId: #M4
    exportMap
      bar: #M0
      foo: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        requestedDeclaredGetters
          foo: #M1
[status] idle
''',
      updatedA: r'''
int get foo => 0;
int get baz => 1;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      baz: #M7
      foo: #M1
    declaredVariables
      baz: #M8
      foo: #M3
    exportMapId: #M9
    exportMap
      baz: #M7
      foo: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getMixin_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getMixin('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        requestedDeclaredMixins
          A: #M0
[status] idle
''',
      updatedA: r'''
mixin B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      B: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      B: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M7
        requestedDeclaredMixins
          A: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getMixin_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getMixin('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
mixin B {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        requestedDeclaredMixins
          A: #M0
[status] idle
''',
      updatedA: r'''
mixin A {}
mixin C {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        interface: #M1
      C: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      C: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getName_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.name;
    });

    await _runChangeScenarioTA(
      initialA: r'''
library foo;
class A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    name: foo
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        name: foo
        libraryMetadataId: #M4
        exportMapId: #M2
[status] idle
''',
      updatedA: r'''
library bar;
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    name: bar
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryNameMismatch
    libraryUri: package:test/a.dart
    expected: foo
    actual: bar
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        name: bar
        libraryMetadataId: #M4
        exportMapId: #M2
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getName_noChange() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.name;
    });

    await _runChangeScenarioTA(
      initialA: r'''
library foo;
class A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    name: foo
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        name: foo
        libraryMetadataId: #M4
        exportMapId: #M2
[status] idle
''',
      updatedA: r'''
library foo;
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    name: foo
    declaredClasses
      B: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      B: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getSetter_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getSetter('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
set foo(int _) {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredSetters
      foo=: #M0
    declaredVariables
      foo: #M1
    exportMapId: #M2
    exportMap
      foo=: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        requestedDeclaredSetters
          foo=: #M0
[status] idle
''',
      updatedA: r'''
set bar(int _) {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredSetters
      bar=: #M5
    declaredVariables
      bar: #M6
    exportMapId: #M7
    exportMap
      bar=: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: foo=
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M7
        requestedDeclaredSetters
          foo=: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getSetter_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getSetter('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
set foo(int _) {}
set bar(int _) {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredSetters
      bar=: #M0
      foo=: #M1
    declaredVariables
      bar: #M2
      foo: #M3
    exportMapId: #M4
    exportMap
      bar=: #M0
      foo=: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M4
        requestedDeclaredSetters
          foo=: #M1
[status] idle
''',
      updatedA: r'''
set foo(int _) {}
set baz(int _) {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredSetters
      baz=: #M7
      foo=: #M1
    declaredVariables
      baz: #M8
      foo: #M3
    exportMapId: #M9
    exportMap
      baz=: #M7
      foo=: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getTopLevelFunction_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getTopLevelFunction('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
void foo() {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredFunctions
      foo: #M0
    exportMapId: #M1
    exportMap
      foo: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M1
        requestedDeclaredFunctions
          foo: #M0
[status] idle
''',
      updatedA: r'''
void bar() {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredFunctions
      bar: #M4
    exportMapId: #M5
    exportMap
      bar: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: foo
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M5
        requestedDeclaredFunctions
          foo: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getTopLevelFunction_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getTopLevelFunction('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
void foo() {}
void bar() {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredFunctions
      bar: #M0
      foo: #M1
    exportMapId: #M2
    exportMap
      bar: #M0
      foo: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        requestedDeclaredFunctions
          foo: #M1
[status] idle
''',
      updatedA: r'''
void foo() {}
void baz() {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredFunctions
      baz: #M5
      foo: #M1
    exportMapId: #M6
    exportMap
      baz: #M5
      foo: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getTopLevelVariable_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getTopLevelVariable('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
var foo = 0;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      foo: #M0
    declaredSetters
      foo=: #M1
    declaredVariables
      foo: #M2
    exportMapId: #M3
    exportMap
      foo: #M0
      foo=: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M3
        requestedDeclaredVariables
          foo: #M2
[status] idle
''',
      updatedA: r'''
var bar = 0;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      bar: #M6
    declaredSetters
      bar=: #M7
    declaredVariables
      bar: #M8
    exportMapId: #M9
    exportMap
      bar: #M6
      bar=: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: foo
    expectedId: #M2
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M9
        requestedDeclaredVariables
          foo: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getTopLevelVariable_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getTopLevelVariable('foo');
    });

    await _runChangeScenarioTA(
      initialA: r'''
var foo = 0;
var bar = 1;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      bar: #M0
      foo: #M1
    declaredSetters
      bar=: #M2
      foo=: #M3
    declaredVariables
      bar: #M4
      foo: #M5
    exportMapId: #M6
    exportMap
      bar: #M0
      bar=: #M2
      foo: #M1
      foo=: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M6
        requestedDeclaredVariables
          foo: #M5
[status] idle
''',
      updatedA: r'''
var foo = 0;
var baz = 1;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      baz: #M9
      foo: #M1
    declaredSetters
      baz=: #M10
      foo=: #M3
    declaredVariables
      baz: #M11
      foo: #M5
    exportMapId: #M12
    exportMap
      baz: #M9
      baz=: #M10
      foo: #M1
      foo=: #M3
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getTypeAlias_change() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getTypeAlias('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
typedef A = int;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredTypeAliases
      A: #M0
    exportMapId: #M1
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M1
        requestedDeclaredTypeAliases
          A: #M0
[status] idle
''',
      updatedA: r'''
typedef B = int;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredTypeAliases
      B: #M4
    exportMapId: #M5
    exportMap
      B: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M5
        requestedDeclaredTypeAliases
          A: <null>
[status] idle
''',
    );
  }

  test_dependency_libraryElement_getTypeAlias_change_other() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.getTypeAlias('A');
    });

    await _runChangeScenarioTA(
      initialA: r'''
typedef A = int;
typedef B = int;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredTypeAliases
      A: #M0
      B: #M1
    exportMapId: #M2
    exportMap
      A: #M0
      B: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        requestedDeclaredTypeAliases
          A: #M0
[status] idle
''',
      updatedA: r'''
typedef A = int;
typedef C = int;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredTypeAliases
      A: #M0
      C: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      C: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
''',
    );
  }

  test_dependency_libraryElement_isSynthetic() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.isSynthetic;
    });

    newFile('$testPackageLibPath/test.dart', r'''
import 'a.dart';
''');

    await _runChangeScenario(
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    flags: isSynthetic
    exportMapId: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M1
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        isSynthetic: true
        libraryMetadataId: #M2
        exportMapId: #M0
[status] idle
''',
      updateFiles: () {
        var a = newFile('$testPackageLibPath/a.dart', '');
        return [a];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    exportMapId: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryIsSyntheticMismatch
    libraryUri: package:test/a.dart
    expected: true
    actual: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M2
        exportMapId: #M0
[status] idle
''',
    );
  }

  test_dependency_libraryElement_languageVersion() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.languageVersion;
    });

    await _runChangeScenarioTA(
      initialA: r'''
// @dart = 3.8
class A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        languageVersion: <not-null>
        libraryMetadataId: #M4
        exportMapId: #M2
[status] idle
''',
      updatedA: r'''
// @dart = 3.9
class A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryLanguageVersionMismatch
    libraryUri: package:test/a.dart
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        languageVersion: <not-null>
        libraryMetadataId: #M4
        exportMapId: #M2
[status] idle
''',
    );
  }

  test_dependency_libraryElement_metadata() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.importedLibraries.first;
      library.metadata;
    });

    await _runChangeScenarioTA(
      initialA: r'''
@Deprecated('0')
library;
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    libraryMetadata: #M0
    exportMapId: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M2
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M0
        exportMapId: #M1
[status] idle
''',
      updatedA: r'''
@Deprecated('1')
library;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    libraryMetadata: #M3
    exportMapId: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  libraryMetadataMismatch
    libraryUri: package:test/a.dart
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        featureSet: <not-null>
        libraryMetadataId: #M3
        exportMapId: #M1
[status] idle
''',
    );
  }

  test_dependency_libraryElement_typeProvider_changeDartCore() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var library = state.singleUnit.libraryElement;
      library.typeProvider.intElement;
    });

    await _runChangeScenario(
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    flags: isSynthetic
    exportMapId: #M0
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
''',
      updateFiles: () {
        var core = sdkRoot.getChildAssumingFile('lib/core/core.dart');
        var newCode = core.readAsStringSync().replaceFirst(
          'abstract final class int extends num {',
          '@deprecated abstract final class int extends num {',
        );
        modifyFile2(core, newCode);
        return [core];
      },
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: dart:core
    name: int
    expectedId: #M1
    actualId: #M2
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
[status] idle
''',
    );
  }

  test_dependency_mixin_instanceField_add_hasNonFinalField() async {
    configuration
      ..withGetErrorsEvents = false
      ..withStreamResolvedUnitResults = false;

    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.hasNonFinalField;
    });

    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            hasNonFinalField: false
[status] idle
''',
      updatedA: r'''
mixin A {
  var foo = 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M5
        declaredGetters
          foo: #M6
        declaredSetters
          foo=: #M7
        interface: #M8
          map
            foo: #M6
            foo=: #M7
          implemented
            foo: #M6
            foo=: #M7
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceHasNonFinalFieldMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    expected: false
    actual: true
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            hasNonFinalField: true
[status] idle
''',
    );
  }

  test_dependency_mixin_instanceGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  int get foo => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  double get foo => 1.2;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M8
        declaredGetters
          foo: #M9
        interface: #M10
          map
            foo: #M9
          implemented
            foo: #M9
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
        interfaces
          A
            methods
              foo: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  int get foo => 0;
  int get bar => 0;
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredGetters
          bar: #M3
          foo: #M4
        interface: #M5
          map
            bar: #M3
            foo: #M4
          implemented
            bar: #M3
            foo: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
        interfaces
          A
            methods
              foo: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  int get foo => 0;
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M2
        declaredGetters
          foo: #M4
        interface: #M10
          map
            foo: #M4
          implemented
            foo: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceGetter_inherited_fromGeneric_on_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
mixin A<T> {
  T get foo {}
}

mixin B on A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M4
        interface: #M5
          map
            foo: #M2
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A<T> {
  T get foo {}
}

mixin B on A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M10
        interface: #M11
          map
            foo: #M2
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M13
    exportMapId: #M14
    exportMap
      f: #M13
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo: #M2
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceMethod_add_invoked() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        interfaces
          A
            methods
              foo: <null>
              foo=: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      35 +3 UNDEFINED_METHOD
''',
      updatedA: r'''
mixin A {
  int foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M6
        interface: #M7
          map
            foo: #M6
          implemented
            foo: #M6
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: <null>
    actualId: #M6
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M6
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M7
        interface: #M8
          map
            foo: #M7
          implemented
            foo: #M7
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M7
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  int foo() {}
  int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredMethods
          bar: #M1
          foo: #M2
        interface: #M3
          map
            bar: #M1
            foo: #M2
          implemented
            bar: #M1
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  int foo() {}
  double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredMethods
          bar: #M8
          foo: #M2
        interface: #M9
          map
            bar: #M8
            foo: #M2
          implemented
            bar: #M8
            foo: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceMethod_inherited_fromGeneric_implements_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
mixin A<T> {
  T foo() {}
}

mixin B implements A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          inherited
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A<T> {
  T foo() {}
}

mixin B implements A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M9
        interface: #M10
          map
            foo: #M1
          inherited
            foo: #M1
    exportMapId: #M11
    exportMap
      A: #M0
      B: #M9
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceMethod_inherited_fromGeneric_on_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
mixin A<T> {
  T foo() {}
}

mixin B on A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        interface: #M4
          map
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M5
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M6
    exportMapId: #M7
    exportMap
      f: #M6
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M5
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A<T> {
  T foo() {}
}

mixin B on A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M9
        interface: #M10
          map
            foo: #M1
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M11
    exportMap
      A: #M0
      B: #M9
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M12
    exportMapId: #M13
    exportMap
      f: #M12
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M3
    actualId: #M9
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M8
        exportMapId: #M11
        exportMap
          B: #M9
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          B
            methods
              foo: #M1
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceMethod_remove_invoked() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  void foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            methods
              foo: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        interface: #M7
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: <null>
        interfaces
          A
            methods
              foo: <null>
              foo=: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      35 +3 UNDEFINED_METHOD
''',
    );
  }

  test_dependency_mixin_instanceSetter_add_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M3
    exportMapId: #M4
    exportMap
      f: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      35 +3 UNDEFINED_SETTER
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredSetters
              foo=: <null>
            requestedDeclaredMethods
              foo: <null>
        interfaces
          A
            methods
              foo: <null>
              foo=: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      35 +3 UNDEFINED_SETTER
''',
      updatedA: r'''
mixin A {
  set foo(int _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M6
        declaredSetters
          foo=: #M7
        interface: #M8
          map
            foo=: #M7
          implemented
            foo=: #M7
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo=
    expectedId: <null>
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M6
        interfaces
          A
            methods
              foo=: #M7
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M8
        declaredSetters
          foo=: #M9
        interface: #M10
          map
            foo=: #M9
          implemented
            foo=: #M9
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
        interfaces
          A
            methods
              foo=: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  set foo(int _) {}
  set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredSetters
          bar=: #M3
          foo=: #M4
        interface: #M5
          map
            bar=: #M3
            foo=: #M4
          implemented
            bar=: #M3
            foo=: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
        interfaces
          A
            methods
              foo=: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  set foo(int _) {}
  set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          bar: #M10
          foo: #M2
        declaredSetters
          bar=: #M11
          foo=: #M4
        interface: #M12
          map
            bar=: #M11
            foo=: #M4
          implemented
            bar=: #M11
            foo=: #M4
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceSetter_inherited_fromGeneric_on_changeTypeArgument() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
mixin A<T> {
  set foo(T _) {}
}

mixin B on A<int> {}
''',
      testCode: r'''
import 'a.dart';
void f(B b) {
  b.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M4
        interface: #M5
          map
            foo=: #M2
          superImplemented
            [0]
              foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          B: #M4
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A<T> {
  set foo(T _) {}
}

mixin B on A<double> {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M10
        interface: #M11
          map
            foo=: #M2
          superImplemented
            [0]
              foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      f: #M13
    exportMapId: #M14
    exportMap
      f: #M13
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: B
    expectedId: #M4
    actualId: #M10
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M12
        exportMap
          B: #M10
          B=: <null>
        reExportDeprecatedOnly
          B: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          B
            methods
              foo=: #M2
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_instanceSetter_remove_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f(A a) {
  a.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
        interfaces
          A
            methods
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        interface: #M8
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      35 +3 UNDEFINED_SETTER
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredSetters
              foo=: <null>
            requestedDeclaredMethods
              foo: <null>
        interfaces
          A
            methods
              foo: <null>
              foo=: <null>
        exportedExtensions: []
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      35 +3 UNDEFINED_SETTER
''',
    );
  }

  test_dependency_mixin_it_add() async {
    await _runChangeScenarioTA(
      initialA: '',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    exportMapId: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M1
    exportMapId: #M2
    exportMap
      foo: #M1
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M0
        exportMap
          A: <null>
          A=: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      17 +1 UNDEFINED_CLASS
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M0
        exportMap
          A: <null>
          A=: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      17 +1 UNDEFINED_CLASS
''',
      updatedA: r'''
mixin A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M4
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: <null>
    actualId: #M4
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M7
    exportMapId: #M8
    exportMap
      foo: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M4
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: <null>
    actualId: #M4
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M3
        exportMapId: #M6
        exportMap
          A: #M4
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
    );
  }

  test_dependency_mixin_it_add_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M3
    exportMapId: #M4
    exportMap
      foo: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
      updatedA: r'''
mixin A {}
mixin B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        interface: #M1
      B: #M6
        interface: #M7
    exportMapId: #M8
    exportMap
      A: #M0
      B: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
    );
  }

  test_dependency_mixin_it_change() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
mixin B {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M5
    exportMapId: #M6
    exportMap
      foo: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
      updatedA: r'''
mixin A on B {}
mixin B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M8
        interface: #M9
      B: #M2
        interface: #M3
    exportMapId: #M10
    exportMap
      A: #M8
      B: #M2
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M8
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M11
    exportMapId: #M12
    exportMap
      foo: #M11
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M10
        exportMap
          A: #M8
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M10
        exportMap
          A: #M8
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
    );
  }

  test_dependency_mixin_it_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
mixin B {}
mixin C {}
''',
      testCode: r'''
import 'a.dart';
A foo() {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
      C: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
      B: #M2
      C: #M4
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M7
    exportMapId: #M8
    exportMap
      foo: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
      updatedA: r'''
mixin A {}
mixin B on C {}
mixin C {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        interface: #M1
      B: #M10
        interface: #M11
      C: #M4
        interface: #M5
    exportMapId: #M12
    exportMap
      A: #M0
      B: #M10
      C: #M4
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      19 +3 BODY_MIGHT_COMPLETE_NORMALLY
''',
    );
  }

  test_dependency_mixin_it_remove() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
''',
      testCode: r'''
import 'a.dart';
A foo() => throw 0;
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M3
    exportMapId: #M4
    exportMap
      foo: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: '',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    exportMapId: #M6
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M7
    exportMapId: #M8
    exportMap
      foo: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: <null>
          A=: <null>
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: <null>
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      17 +1 UNDEFINED_CLASS
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M6
        exportMap
          A: <null>
          A=: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      17 +1 UNDEFINED_CLASS
''',
    );
  }

  test_dependency_mixin_it_remove_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {}
mixin B {}
''',
      testCode: r'''
import 'a.dart';
A foo() => throw 0;
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        interface: #M1
      B: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M2
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M5
    exportMapId: #M6
    exportMap
      foo: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        interface: #M1
    exportMapId: #M8
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_staticGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  static int get foo {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredGetters
              foo: #M2
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  static double get foo {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M8
        declaredGetters
          foo: #M9
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
            requestedDeclaredGetters
              foo: #M9
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_staticGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  static int get foo {}
  static int get bar {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredGetters
          bar: #M3
          foo: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredGetters
              foo: #M4
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  static int get foo {}
  static double get bar {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          bar: #M10
          foo: #M2
        declaredGetters
          bar: #M11
          foo: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_staticMethod_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  static int foo() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M4
    exportMapId: #M5
    exportMap
      f: #M4
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M1
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  static double foo() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredMethods
          foo: #M7
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceMethodIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    methodName: foo
    expectedId: #M1
    actualId: #M7
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M3
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M7
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_staticMethod_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  static int foo() {}
  static int bar() {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo();
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredMethods
          bar: #M1
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredGetters
              foo: <null>
            requestedDeclaredMethods
              foo: #M2
        interfaces
          A
            requestedConstructors
              foo: <null>
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  static int foo() {}
  static double bar() {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredMethods
          bar: #M8
          foo: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_staticSetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  static set foo(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M5
    exportMapId: #M6
    exportMap
      f: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M1
            requestedDeclaredSetters
              foo=: #M2
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  static set foo(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          foo: #M8
        declaredSetters
          foo=: #M9
        interface: #M3
    exportMapId: #M4
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  instanceFieldIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    fieldName: foo
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M8
            requestedDeclaredSetters
              foo=: #M9
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_mixin_staticSetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
mixin A {
  static set foo(int _) {}
  static set bar(int _) {}
}
''',
      testCode: r'''
import 'a.dart';
void f() {
  A.foo = 0;
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredMixins
      A: #M0
        declaredFields
          bar: #M1
          foo: #M2
        declaredSetters
          bar=: #M3
          foo=: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      f: #M7
    exportMapId: #M8
    exportMap
      f: #M7
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M9
        exportMapId: #M6
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            requestedDeclaredFields
              foo: #M2
            requestedDeclaredSetters
              foo=: #M4
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
mixin A {
  static set foo(int _) {}
  static set bar(double _) {}
}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredMixins
      A: #M0
        declaredFields
          bar: #M10
          foo: #M2
        declaredSetters
          bar=: #M11
          foo=: #M4
        interface: #M5
    exportMapId: #M6
    exportMap
      A: #M0
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_notOpaque_baseClassImplementedOutsideOfLibrary() async {
    configuration.withStreamResolvedUnitResults = false;
    await _runChangeScenarioTA(
      initialA: r'''
base class A {}
''',
      testCode: r'''
import 'a.dart';
base class X implements A {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      X: #M3
        interface: #M4
    exportMapId: #M5
    exportMap
      X: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        interfaces
          A
            interfaceId: #M1
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M6
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
        instances
          A
            allDeclaredGetters: []
            allDeclaredSetters: []
            allDeclaredMethods: []
[status] idle
[future] getErrors T1
  ErrorsResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      41 +1 INVALID_USE_OF_TYPE_OUTSIDE_LIBRARY
''',
      updatedA: r'''
base class A {}
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M7
        interface: #M8
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M7
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      41 +1 INVALID_USE_OF_TYPE_OUTSIDE_LIBRARY
''',
    );
  }

  test_dependency_opaqueApiUse_firstFragment() async {
    _ManualRequirements.install((state) {
      var A = state.singleUnit.scopeInterfaceElement('A');
      A.firstFragment;
    });

    await _runChangeScenarioTA(
      initialA: r'''
class A {}
''',
      testCode: r'''
import 'a.dart';
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        interface: #M1
    exportMapId: #M2
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    exportMapId: #M3
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
    opaqueApiUses
      ClassElementImpl.firstFragment
        targetElementLibraryUri: package:test/a.dart
        targetElementName: A
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
      updatedA: r'''
class A {}
class B {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        interface: #M1
      B: #M5
        interface: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M5
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  opaqueApiUseFailure
    ClassElementImpl.firstFragment
      targetElementLibraryUri: package:test/a.dart
      targetElementName: A
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
    errors
      7 +8 UNUSED_IMPORT
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M7
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
    opaqueApiUses
      ClassElementImpl.firstFragment
        targetElementLibraryUri: package:test/a.dart
        targetElementName: A
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
    errors
      7 +8 UNUSED_IMPORT
''',
    );
  }

  test_dependency_topLevelFunction_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
int foo() {}
''',
      testCode: r'''
import 'a.dart';
final x = foo();
''',
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredFunctions
      foo: #M0
    exportMapId: #M1
    exportMap
      foo: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredGetters
      x: #M2
    declaredVariables
      x: #M3
    exportMapId: #M4
    exportMap
      x: #M2
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M1
        exportMap
          foo: #M0
          foo=: <null>
        reExportDeprecatedOnly
          foo: false
[status] idle
[future] getLibraryByUri T1
  library
    topLevelVariables
      final hasInitializer x
        type: int
''',
      updatedA: r'''
double foo() {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredFunctions
      foo: #M5
    exportMapId: #M6
    exportMap
      foo: #M5
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: foo
    expectedId: #M0
    actualId: #M5
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredGetters
      x: #M7
    declaredVariables
      x: #M8
    exportMapId: #M9
    exportMap
      x: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          foo: #M5
          foo=: <null>
        reExportDeprecatedOnly
          foo: false
[status] idle
[future] getLibraryByUri T2
  library
    topLevelVariables
      final hasInitializer x
        type: double
''',
    );
  }

  test_dependency_topLevelFunction_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
int foo() {}
int bar() {}
''',
      testCode: r'''
import 'a.dart';
final x = foo();
''',
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredFunctions
      bar: #M0
      foo: #M1
    exportMapId: #M2
    exportMap
      bar: #M0
      foo: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredGetters
      x: #M3
    declaredVariables
      x: #M4
    exportMapId: #M5
    exportMap
      x: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          foo: #M1
          foo=: <null>
        reExportDeprecatedOnly
          foo: false
[status] idle
[future] getLibraryByUri T1
  library
    topLevelVariables
      final hasInitializer x
        type: int
''',
      updatedA: r'''
int foo() {}
double bar() {}
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredFunctions
      bar: #M6
      foo: #M1
    exportMapId: #M7
    exportMap
      bar: #M6
      foo: #M1
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    topLevelVariables
      final hasInitializer x
        type: int
''',
    );
  }

  test_dependency_topLevelGetter_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
int get a => 0;
''',
      testCode: r'''
import 'a.dart';
final x = a;
''',
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredGetters
      x: #M3
    declaredVariables
      x: #M4
    exportMapId: #M5
    exportMap
      x: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          a: #M0
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T1
  library
    topLevelVariables
      final hasInitializer x
        type: int
''',
      updatedA: r'''
double get a => 1.2;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M6
    declaredVariables
      a: #M7
    exportMapId: #M8
    exportMap
      a: #M6
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: a
    expectedId: #M0
    actualId: #M6
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredGetters
      x: #M9
    declaredVariables
      x: #M10
    exportMapId: #M11
    exportMap
      x: #M9
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M8
        exportMap
          a: #M6
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T2
  library
    topLevelVariables
      final hasInitializer x
        type: double
''',
    );
  }

  test_dependency_topLevelGetter_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
int get a => 0;
int get b => 0;
''',
      testCode: r'''
import 'a.dart';
final x = a;
''',
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
      b: #M1
    declaredVariables
      a: #M2
      b: #M3
    exportMapId: #M4
    exportMap
      a: #M0
      b: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredGetters
      x: #M5
    declaredVariables
      x: #M6
    exportMapId: #M7
    exportMap
      x: #M5
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M4
        exportMap
          a: #M0
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T1
  library
    topLevelVariables
      final hasInitializer x
        type: int
''',
      updatedA: r'''
int get a => 0;
double get b => 1.2;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M0
      b: #M8
    declaredVariables
      a: #M2
      b: #M9
    exportMapId: #M10
    exportMap
      a: #M0
      b: #M8
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[status] idle
[future] getLibraryByUri T2
  library
    topLevelVariables
      final hasInitializer x
        type: int
''',
    );
  }

  test_dependency_topLevelVariable_change_exported() async {
    var a = newFile('$testPackageLibPath/a.dart', r'''
final a = 0;
''');

    newFile('$testPackageLibPath/b.dart', r'''
export 'a.dart';
''');

    // Uses exported `a`.
    newFile('$testPackageLibPath/test.dart', r'''
import 'b.dart';
final x = a;
''');

    await _runChangeScenario(
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H1
    exportMapId: #M3
    exportMap
      a: #M0
    reExportMap
      a: #M0
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/b.dart
        exports
          package:test/a.dart
            a: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    declaredGetters
      x: #M4
    declaredVariables
      x: #M5
    exportMapId: #M6
    exportMap
      x: #M4
  requirements
    libraries
      package:test/b.dart
        exportMapId: #M3
        exportMap
          a: #M0
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T1
  library
    topLevelVariables
      final hasInitializer x
        type: int
''',
      // Change the initializer, now `double`.
      updateFiles: () {
        modifyFile2(a, r'''
final a = 1.2;
''');
        return [a];
      },
      // Linked, `x` has type `double`.
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H3
    declaredGetters
      a: #M7
    declaredVariables
      a: #M8
    exportMapId: #M9
    exportMap
      a: #M7
  requirements
[operation] checkLinkedBundleRequirements
  package:test/b.dart
  exportIdMismatch
    fragmentUri: package:test/b.dart
    exportedUri: package:test/a.dart
    name: a
    expectedId: #M0
    actualId: #M7
[operation] linkLibraryCycle
  package:test/b.dart
    hashForRequirements: #H4
    exportMapId: #M10
    exportMap
      a: #M7
    reExportMap
      a: #M7
    exportedLibraryUris: package:test/a.dart
  requirements
    exportRequirements
      package:test/b.dart
        exports
          package:test/a.dart
            a: #M7
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/b.dart
    name: a
    expectedId: #M0
    actualId: #M7
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H5
    declaredGetters
      x: #M11
    declaredVariables
      x: #M12
    exportMapId: #M13
    exportMap
      x: #M11
  requirements
    libraries
      package:test/b.dart
        exportMapId: #M10
        exportMap
          a: #M7
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T2
  library
    topLevelVariables
      final hasInitializer x
        type: double
''',
    );
  }

  test_dependency_topLevelVariable_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
final a = 0;
''',
      testCode: r'''
import 'a.dart';
final x = a;
''',
      operation: _FineOperationGetTestLibrary(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredGetters
      a: #M0
    declaredVariables
      a: #M1
    exportMapId: #M2
    exportMap
      a: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredGetters
      x: #M3
    declaredVariables
      x: #M4
    exportMapId: #M5
    exportMap
      x: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          a: #M0
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T1
  library
    topLevelVariables
      final hasInitializer x
        type: int
''',
      // Change the initializer, now `double`.
      updatedA: r'''
final a = 1.2;
''',
      // Linked, `x` has type `double`.
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredGetters
      a: #M6
    declaredVariables
      a: #M7
    exportMapId: #M8
    exportMap
      a: #M6
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: a
    expectedId: #M0
    actualId: #M6
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredGetters
      x: #M9
    declaredVariables
      x: #M10
    exportMapId: #M11
    exportMap
      x: #M9
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M8
        exportMap
          a: #M6
          a=: <null>
        reExportDeprecatedOnly
          a: false
[status] idle
[future] getLibraryByUri T2
  library
    topLevelVariables
      final hasInitializer x
        type: double
''',
    );
  }

  test_dependency_typeAlias_change_notUsed() async {
    await _runChangeScenarioTA(
      initialA: r'''
typedef A = int;
typedef B = int;
''',
      testCode: r'''
import 'a.dart';
void foo(A _) {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredTypeAliases
      A: #M0
      B: #M1
    exportMapId: #M2
    exportMap
      A: #M0
      B: #M1
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M3
    exportMapId: #M4
    exportMap
      foo: #M3
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M5
        exportMapId: #M2
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
typedef A = int;
typedef B = double;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredTypeAliases
      A: #M0
      B: #M6
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M6
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] getErrorsFromBytes
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[status] idle
[future] getErrors T2
  ErrorsResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_typeAlias_change_referenced() async {
    await _runChangeScenarioTA(
      initialA: r'''
typedef A = int;
''',
      testCode: r'''
import 'a.dart';
void foo(A _) {}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredTypeAliases
      A: #M0
    exportMapId: #M1
    exportMap
      A: #M0
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M2
    exportMapId: #M3
    exportMap
      foo: #M2
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M1
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M1
        exportMap
          A: #M0
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
typedef A = double;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredTypeAliases
      A: #M5
    exportMapId: #M6
    exportMap
      A: #M5
  requirements
[operation] checkLinkedBundleRequirements
  package:test/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M5
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H3
    declaredFunctions
      foo: #M7
    exportMapId: #M8
    exportMap
      foo: #M7
  requirements
    libraries
      package:test/a.dart
        exportMapId: #M6
        exportMap
          A: #M5
          A=: <null>
        reExportDeprecatedOnly
          A: false
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  topLevelIdMismatch
    libraryUri: package:test/a.dart
    name: A
    expectedId: #M0
    actualId: #M5
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M4
        exportMapId: #M6
        exportMap
          A: #M5
          A=: <null>
        reExportDeprecatedOnly
          A: false
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_dependency_typeAlias_ofClass_constructor_change_invoked() async {
    await _runChangeScenarioTA(
      initialA: r'''
class A {
  A.named(int _);
}
typedef B = A;
''',
      testCode: r'''
import 'a.dart';
void foo() {
  B.named(0);
}
''',
      operation: _FineOperationTestFileGetErrors(),
      expectedInitialEvents: r'''
[status] working
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M1
        interface: #M2
    declaredTypeAliases
      B: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredFunctions
      foo: #M5
    exportMapId: #M6
    exportMap
      foo: #M5
  requirements
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #0
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          A
            requestedConstructors
              named: #M1
[status] idle
[future] getErrors T1
  ErrorsResult #1
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
      updatedA: r'''
class A {
  A.named(double _);
}
typedef B = A;
''',
      expectedUpdatedEvents: r'''
[status] working
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M0
        declaredConstructors
          named: #M8
        interface: #M2
    declaredTypeAliases
      B: #M3
    exportMapId: #M4
    exportMap
      A: #M0
      B: #M3
  requirements
[operation] reuseLinkedBundle
  package:test/test.dart
[operation] checkLibraryDiagnosticsRequirements
  library: /home/test/lib/test.dart
  interfaceConstructorIdMismatch
    libraryUri: package:test/a.dart
    interfaceName: A
    constructorName: named
    expectedId: #M1
    actualId: #M8
[operation] analyzeFile
  file: /home/test/lib/test.dart
  library: /home/test/lib/test.dart
[stream]
  ResolvedUnitResult #2
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: exists isLibrary
[operation] analyzedLibrary
  file: /home/test/lib/test.dart
  requirements
    libraries
      package:test/a.dart
        libraryMetadataId: #M7
        exportMapId: #M4
        exportMap
          B: #M3
          B=: <null>
        reExportDeprecatedOnly
          B: false
        interfaces
          A
            requestedConstructors
              named: #M8
[status] idle
[future] getErrors T2
  ErrorsResult #3
    path: /home/test/lib/test.dart
    uri: package:test/test.dart
    flags: isLibrary
''',
    );
  }

  test_manifest_baseName2_private2() async {
    newFile('$testPackageLibPath/a.dart', r'''
class A {
  void _foo() {}
}
''');

    await _runLibraryManifestScenario(
      initialCode: r'''
import 'a.dart';

class B extends A {
  int get _foo {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          _foo: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      B: #M4
        declaredFields
          _foo: #M5
        declaredGetters
          _foo: #M6
        interface: #M7
    exportMapId: #M8
    exportMap
      B: #M4
''',
      updatedCode: r'''
import 'a.dart';

class B extends A {
  int get _foo {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H2
    declaredClasses
      B: #M4
        declaredFields
          _foo: #M5
        declaredGetters
          _foo: #M6
        declaredMethods
          zzz: #M9
        interface: #M10
          map
            zzz: #M9
          implemented
            zzz: #M9
    exportMapId: #M8
    exportMap
      B: #M4
''',
    );
  }

  test_manifest_baseName2_private3() async {
    newFile('$testPackageLibPath/a.dart', r'''
import 'test.dart';

class B extends A {
  void _foo() {}
}
''');

    await _runLibraryManifestScenario(
      initialCode: r'''
import 'a.dart';

class A {
  int get _foo {}
}

class C extends B {
  set _foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      B: #M0
        declaredMethods
          _foo: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      B: #M0
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M4
        declaredFields
          _foo: #M5
        declaredGetters
          _foo: #M6
        interface: #M7
      C: #M8
        declaredFields
          _foo: #M9
        declaredSetters
          _foo=: #M10
        interface: #M11
    exportMapId: #M12
    exportMap
      A: #M4
      C: #M8
''',
      updatedCode: r'''
import 'a.dart';

class A {
  int get _foo {}
}

class C extends B {
  set _foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/a.dart
    hashForRequirements: #H0
    declaredClasses
      B: #M0
        declaredMethods
          _foo: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      B: #M0
  package:test/test.dart
    hashForRequirements: #H2
    declaredClasses
      A: #M4
        declaredFields
          _foo: #M5
        declaredGetters
          _foo: #M6
        interface: #M7
      C: #M8
        declaredFields
          _foo: #M9
        declaredSetters
          _foo=: #M10
        declaredMethods
          zzz: #M13
        interface: #M14
          map
            zzz: #M13
          implemented
            zzz: #M13
    exportMapId: #M12
    exportMap
      A: #M4
      C: #M8
''',
    );
  }

  test_manifest_baseName_declaredConstructor() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  A.foo();
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConstructors
          foo: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
''',
      updatedCode: r'''
class A {
  A.foo();
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredMethods
          zzz: #M4
        declaredConstructors
          foo: #M1
        interface: #M5
          map
            zzz: #M4
          implemented
            zzz: #M4
    exportMapId: #M3
    exportMap
      A: #M0
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredConstructor() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  A.foo();
  A.foo();
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConflicts
          foo: #M1
          foo=: #M1
        interface: #M2
    exportMapId: #M3
    exportMap
      A: #M0
''',
      updatedCode: r'''
class A {
  A.foo();
  A.foo();
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredConflicts
          foo: #M4
          foo=: #M4
        declaredMethods
          zzz: #M5
        interface: #M6
          map
            zzz: #M5
          implemented
            zzz: #M5
    exportMapId: #M3
    exportMap
      A: #M0
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  A.foo();
  int get foo {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredConstructors
          foo: #M3
        interface: #M4
          map
            foo: #M2
          implemented
            foo: #M2
    exportMapId: #M5
    exportMap
      A: #M0
''',
      updatedCode: r'''
class A {
  A.foo();
  int get foo {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredMethods
          zzz: #M6
        declaredConstructors
          foo: #M3
        interface: #M7
          map
            foo: #M2
            zzz: #M6
          implemented
            foo: #M2
            zzz: #M6
    exportMapId: #M5
    exportMap
      A: #M0
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredInstanceGetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  A.foo();
  int get foo {}
  int get foo {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConflicts
          foo: #M1
          foo=: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
''',
      updatedCode: r'''
class A {
  A.foo();
  int get foo {}
  int get foo {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredConflicts
          foo: #M4
          foo=: #M4
        declaredMethods
          zzz: #M5
        interface: #M6
          map
            foo: #M4
            zzz: #M5
          implemented
            foo: #M4
            zzz: #M5
    exportMapId: #M3
    exportMap
      A: #M0
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredInstanceSetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  A.foo();
  int get foo {}
  set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        declaredConstructors
          foo: #M4
        interface: #M5
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
    exportMapId: #M6
    exportMap
      A: #M0
''',
      updatedCode: r'''
class A {
  A.foo();
  int get foo {}
  set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        declaredMethods
          zzz: #M7
        declaredConstructors
          foo: #M4
        interface: #M8
          map
            foo: #M2
            foo=: #M3
            zzz: #M7
          implemented
            foo: #M2
            foo=: #M3
            zzz: #M7
    exportMapId: #M6
    exportMap
      A: #M0
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredInstanceSetter_declaredInstanceSetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  A.foo();
  int get foo {}
  set foo(int _) {}
  set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConflicts
          foo: #M1
          foo=: #M1
        interface: #M2
          map
            foo: #M1
            foo=: #M1
          implemented
            foo: #M1
            foo=: #M1
    exportMapId: #M3
    exportMap
      A: #M0
''',
      updatedCode: r'''
class A {
  A.foo();
  int get foo {}
  set foo(int _) {}
  set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredConflicts
          foo: #M4
          foo=: #M4
        declaredMethods
          zzz: #M5
        interface: #M6
          map
            foo: #M4
            foo=: #M4
            zzz: #M5
          implemented
            foo: #M4
            foo=: #M4
            zzz: #M5
    exportMapId: #M3
    exportMap
      A: #M0
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredInstanceSetter_inheritedInstanceGetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  int get foo {}
}

class B extends A {
  B.foo();
  int get foo {}
  set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M4
        declaredFields
          foo: #M5
        declaredGetters
          foo: #M6
        declaredSetters
          foo=: #M7
        declaredConstructors
          foo: #M8
        interface: #M9
          map
            foo: #M6
            foo=: #M7
          implemented
            foo: #M6
            foo=: #M7
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M10
    exportMap
      A: #M0
      B: #M4
''',
      updatedCode: r'''
class A {
  int get foo {}
}

class B extends A {
  B.foo();
  int get foo {}
  set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M4
        declaredFields
          foo: #M5
        declaredGetters
          foo: #M6
        declaredSetters
          foo=: #M7
        declaredMethods
          zzz: #M11
        declaredConstructors
          foo: #M8
        interface: #M12
          map
            foo: #M6
            foo=: #M7
            zzz: #M11
          implemented
            foo: #M6
            foo=: #M7
            zzz: #M11
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M10
    exportMap
      A: #M0
      B: #M4
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredInstanceSetter_inheritedInstanceGetter_inheritedInstanceSetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  int get foo {}
  set foo(int _) {}
}

class B extends A {
  B.foo();
  int get foo {}
  set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        interface: #M4
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
      B: #M5
        declaredFields
          foo: #M6
        declaredGetters
          foo: #M7
        declaredSetters
          foo=: #M8
        declaredConstructors
          foo: #M9
        interface: #M10
          map
            foo: #M7
            foo=: #M8
          implemented
            foo: #M7
            foo=: #M8
          superImplemented
            [0]
              foo: #M2
              foo=: #M3
          inherited
            foo: #M2
            foo=: #M3
    exportMapId: #M11
    exportMap
      A: #M0
      B: #M5
''',
      updatedCode: r'''
class A {
  int get foo {}
  set foo(int _) {}
}

class B extends A {
  B.foo();
  int get foo {}
  set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        interface: #M4
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
      B: #M5
        declaredFields
          foo: #M6
        declaredGetters
          foo: #M7
        declaredSetters
          foo=: #M8
        declaredMethods
          zzz: #M12
        declaredConstructors
          foo: #M9
        interface: #M13
          map
            foo: #M7
            foo=: #M8
            zzz: #M12
          implemented
            foo: #M7
            foo=: #M8
            zzz: #M12
          superImplemented
            [0]
              foo: #M2
              foo=: #M3
          inherited
            foo: #M2
            foo=: #M3
    exportMapId: #M11
    exportMap
      A: #M0
      B: #M5
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredInstanceSetter_inheritedInstanceMethod() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  void foo() {}
}

class B extends A {
  B.foo();
  int get foo {}
  set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        declaredFields
          foo: #M4
        declaredGetters
          foo: #M5
        declaredSetters
          foo=: #M6
        declaredConstructors
          foo: #M7
        interface: #M8
          map
            foo: #M5
            foo=: #M6
          implemented
            foo: #M5
            foo=: #M6
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M3
''',
      updatedCode: r'''
class A {
  void foo() {}
}

class B extends A {
  B.foo();
  int get foo {}
  set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        declaredFields
          foo: #M4
        declaredGetters
          foo: #M5
        declaredSetters
          foo=: #M6
        declaredMethods
          zzz: #M10
        declaredConstructors
          foo: #M7
        interface: #M11
          map
            foo: #M5
            foo=: #M6
            zzz: #M10
          implemented
            foo: #M5
            foo=: #M6
            zzz: #M10
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M9
    exportMap
      A: #M0
      B: #M3
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredInstanceSetter_inheritedInstanceSetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  set foo(int _) {}
}

class B extends A {
  B.foo();
  int get foo {}
  set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M4
        declaredFields
          foo: #M5
        declaredGetters
          foo: #M6
        declaredSetters
          foo=: #M7
        declaredConstructors
          foo: #M8
        interface: #M9
          map
            foo: #M6
            foo=: #M7
          implemented
            foo: #M6
            foo=: #M7
          superImplemented
            [0]
              foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M10
    exportMap
      A: #M0
      B: #M4
''',
      updatedCode: r'''
class A {
  set foo(int _) {}
}

class B extends A {
  B.foo();
  int get foo {}
  set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredSetters
          foo=: #M2
        interface: #M3
          map
            foo=: #M2
          implemented
            foo=: #M2
      B: #M4
        declaredFields
          foo: #M5
        declaredGetters
          foo: #M6
        declaredSetters
          foo=: #M7
        declaredMethods
          zzz: #M11
        declaredConstructors
          foo: #M8
        interface: #M12
          map
            foo: #M6
            foo=: #M7
            zzz: #M11
          implemented
            foo: #M6
            foo=: #M7
            zzz: #M11
          superImplemented
            [0]
              foo=: #M2
          inherited
            foo=: #M2
    exportMapId: #M10
    exportMap
      A: #M0
      B: #M4
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredStaticSetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  A.foo();
  int get foo {}
  static set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredConflicts
          foo: #M1
          foo=: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
    exportMapId: #M3
    exportMap
      A: #M0
''',
      updatedCode: r'''
class A {
  A.foo();
  int get foo {}
  static set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredConflicts
          foo: #M4
          foo=: #M4
        declaredMethods
          zzz: #M5
        interface: #M6
          map
            foo: #M4
            zzz: #M5
          implemented
            foo: #M4
            zzz: #M5
    exportMapId: #M3
    exportMap
      A: #M0
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredStaticSetter_inheritedInstanceGetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  int get foo {}
}

class B extends A {
  B.foo();
  int get foo {}
  static set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M4
        declaredConflicts
          foo: #M5
          foo=: #M5
        interface: #M6
          map
            foo: #M5
          implemented
            foo: #M5
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M4
''',
      updatedCode: r'''
class A {
  int get foo {}
}

class B extends A {
  B.foo();
  int get foo {}
  static set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        interface: #M3
          map
            foo: #M2
          implemented
            foo: #M2
      B: #M4
        declaredConflicts
          foo: #M8
          foo=: #M8
        declaredMethods
          zzz: #M9
        interface: #M10
          map
            foo: #M8
            zzz: #M9
          implemented
            foo: #M8
            zzz: #M9
          superImplemented
            [0]
              foo: #M2
          inherited
            foo: #M2
    exportMapId: #M7
    exportMap
      A: #M0
      B: #M4
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredStaticSetter_inheritedInstanceGetter_inheritedInstanceSetter() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  int get foo {}
  set foo(int _) {}
}

class B extends A {
  B.foo();
  int get foo {}
  static set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        interface: #M4
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
      B: #M5
        declaredConflicts
          foo: #M6
          foo=: #M6
        interface: #M7
          map
            foo: #M6
            foo=: #M3
          implemented
            foo: #M6
            foo=: #M3
          superImplemented
            [0]
              foo: #M2
              foo=: #M3
          inherited
            foo: #M2
            foo=: #M3
    exportMapId: #M8
    exportMap
      A: #M0
      B: #M5
''',
      updatedCode: r'''
class A {
  int get foo {}
  set foo(int _) {}
}

class B extends A {
  B.foo();
  int get foo {}
  static set foo(int _) {}
  void zzz() {}
}
''',
      expectedUpdatedEvents: r'''
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H1
    declaredClasses
      A: #M0
        declaredFields
          foo: #M1
        declaredGetters
          foo: #M2
        declaredSetters
          foo=: #M3
        interface: #M4
          map
            foo: #M2
            foo=: #M3
          implemented
            foo: #M2
            foo=: #M3
      B: #M5
        declaredConflicts
          foo: #M9
          foo=: #M9
        declaredMethods
          zzz: #M10
        interface: #M11
          map
            foo: #M9
            foo=: #M3
            zzz: #M10
          implemented
            foo: #M9
            foo=: #M3
            zzz: #M10
          superImplemented
            [0]
              foo: #M2
              foo=: #M3
          inherited
            foo: #M2
            foo=: #M3
    exportMapId: #M8
    exportMap
      A: #M0
      B: #M5
''',
    );
  }

  test_manifest_baseName_declaredConstructor_declaredInstanceGetter_declaredStaticSetter_inheritedInstanceMethod() async {
    await _runLibraryManifestScenario(
      initialCode: r'''
class A {
  void foo() {}
}

class B extends A {
  B.foo();
  int get foo {}
  static set foo(int _) {}
}
''',
      expectedInitialEvents: r'''
[operation] linkLibraryCycle SDK
[operation] linkLibraryCycle
  package:test/test.dart
    hashForRequirements: #H0
    declaredClasses
      A: #M0
        declaredMethods
          foo: #M1
        interface: #M2
          map
            foo: #M1
          implemented
            foo: #M1
      B: #M3
        declaredConflicts
          foo: #M4
          foo=: #M4
        interface: #M5
          map
            foo: #M4
          implemented
            foo: #M4
          superImplemented
            [0]
              foo: #M1
          inherited
            foo: #M1
    exportMapId: #M6
    exportMap
  