import 'package:pub_semver/pub_semver.dart';
import 'package:pubspec_lock_parse/src/package_description.dart';
import 'package:test/test.dart';

import 'test_utils.dart';

void main() {
  test('sdks', () {
    final value = parse('''
      # Generated by pub
      # See https://dart.dev/tools/pub/glossary#lockfile
      packages:
        package_a:
          dependency: transitive
          description:
            name: package_a
            url: "https://pub.dartlang.org"
          source: hosted
          version: "1.2.3"
      sdks:
        dart: ">=2.7.0 <3.0.0"
    ''');
    expect(value.sdks['dart'], VersionConstraint.parse('>=2.7.0 <3.0.0'));
  });

  group('package types -', () {
    test('hosted', () {
      final value = parse('''
        packages:
          package_a:
            dependency: transitive
            description:
              name: package_a
              url: "https://pub.dartlang.org"
            source: hosted
            version: "1.2.3"
        sdks:
          dart: ">=2.7.0 <3.0.0"
      ''');

      final package = value.packages['package_a']!;
      expect(package.dependency, 'transitive');
      expect(package.description is HostedPackageDescription, isTrue);
      expect(package.version, Version.parse('1.2.3'));

      final packageDescription =
          package.description as HostedPackageDescription;
      expect(packageDescription.name, 'package_a');
      expect(packageDescription.url, 'https://pub.dartlang.org');
    });

    test('git', () {
      final value = parse('''
        packages:
          package_a:
            dependency: direct overridden
            description:
              path: "../path"
              ref: "some-branch-name"
              resolved-ref: "abcdefg"
              url: "git@github.com:SomeOrg/SomeRepo.git"
            source: git
            version: "1.2.3"
        sdks:
          dart: ">=2.7.0 <3.0.0"
      ''');

      final package = value.packages['package_a']!;
      expect(package.dependency, 'direct overridden');
      expect(package.description is GitPackageDescription, isTrue);
      expect(package.version, Version.parse('1.2.3'));

      final packageDescription = package.description as GitPackageDescription;
      expect(packageDescription.path, '../path');
      expect(packageDescription.ref, 'some-branch-name');
      expect(packageDescription.resolvedRef, 'abcdefg');
      expect(packageDescription.url, 'git@github.com:SomeOrg/SomeRepo.git');
    });

    test('path', () {
      final value = parse('''
        packages:
          package_a:
            dependency: direct
            description:
              path: ../path
              relative: true
            source: path
            version: "1.2.3"
        sdks:
          dart: ">=2.7.0 <3.0.0"
      ''');

      final package = value.packages['package_a']!;
      expect(package.dependency, 'direct');
      expect(package.description is PathPackageDescription, isTrue);
      expect(package.version, Version.parse('1.2.3'));

      final packageDescription = package.description as PathPackageDescription;
      expect(packageDescription.path, '../path');
      expect(packageDescription.relative, isTrue);
    });

    test('sdk', () {
      final value = parse('''
        packages:
          package_a:
            dependency: direct
            description: flutter
            source: sdk
            version: "0.0.0"
        sdks:
          dart: ">=2.7.0 <3.0.0"
      ''');

      final package = value.packages['package_a']!;
      expect(package.dependency, 'direct');
      expect(package.description is SdkPackageDescription, isTrue);
      expect(package.version, Version.parse('0.0.0'));

      final packageDescription = package.description as SdkPackageDescription;
      expect(packageDescription.description, 'flutter');
    });
  });
}
