package lying.fengfeng.foodrecords.repository

import StatisticPermissionState
import android.annotation.SuppressLint
import android.app.Application
import android.content.Context
import android.content.SharedPreferences
import androidx.room.Room
import androidx.room.migration.Migration
import androidx.sqlite.db.SupportSQLiteDatabase
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import lying.fengfeng.foodrecords.Constants.DB_NAME_FOOD_INFO
import lying.fengfeng.foodrecords.Constants.DB_NAME_FOOD_TYPE_INFO
import lying.fengfeng.foodrecords.Constants.DB_NAME_SHELF_LIFE_INFO
import lying.fengfeng.foodrecords.Constants.SP_NAME
import lying.fengfeng.foodrecords.R
import lying.fengfeng.foodrecords.entities.FoodInfo
import lying.fengfeng.foodrecords.entities.FoodTypeInfo
import lying.fengfeng.foodrecords.entities.ShelfLifeInfo
import lying.fengfeng.foodrecords.ui.theme.ThemeOptions
import java.text.SimpleDateFormat
import java.util.Locale

object AppRepo {

    private lateinit var app: Application
    private lateinit var sp: SharedPreferences

    private lateinit var foodInfoDB: FoodInfoDatabase
    private lateinit var foodInfoDao: FoodInfoDao

    private lateinit var typeInfoDB: FoodTypeInfoDatabase
    private lateinit var typeInfoDao: FoodTypeInfoDao

    private lateinit var shelfLifeDB: ShelfLifeInfoDatabase
    private lateinit var shelfLifeDao: ShelfLifeInfoDao

    private val MIGRATION_1_2 = object : Migration(1, 2) {
        override fun migrate(db: SupportSQLiteDatabase) {
            db.execSQL("ALTER TABLE FoodInfo ADD COLUMN expirationDate TEXT NOT NULL DEFAULT '' ")
            db.execSQL("ALTER TABLE FoodInfo ADD COLUMN tips TEXT NOT NULL DEFAULT '' ")
        }
    }
    private val MIGRATION_2_3 = object : Migration(2, 3) {
        @SuppressLint("Range")
        override fun migrate(db: SupportSQLiteDatabase) {
            val cursor = db.query("SELECT uuid, productionDate, expirationDate FROM FoodInfo",
                emptyArray())
            if (cursor.moveToFirst()) {
                do {
                    val uuid = cursor.getString(cursor.getColumnIndex("uuid"))
                    val productionDate = cursor.getString(cursor.getColumnIndex("productionDate"))
                    val expirationDate = cursor.getString(cursor.getColumnIndex("expirationDate"))

                    val dateFormatter = SimpleDateFormat("yy-MM-dd", Locale.getDefault())
                    val productionDateTimestamp = dateFormatter.parse(productionDate)?.time ?: 0
                    val expirationDateTimestamp = if (expirationDate == "--") 0 else dateFormatter.parse(expirationDate)?.time ?: 0
                    val updateQuery = "UPDATE FoodInfo SET productionDate = ?, expirationDate = ? " +
                            "WHERE uuid = ?"

                    db.execSQL(updateQuery, arrayOf(productionDateTimestamp, expirationDateTimestamp,
                        uuid))

                } while (cursor.moveToNext())
            }
            cursor.close()
        }
    }
    private val MIGRATION_3_4 = object : Migration(3, 4) {
        override fun migrate(db: SupportSQLiteDatabase) {
            db.execSQL("ALTER TABLE FoodInfo ADD COLUMN amount INTEGER NOT NULL DEFAULT 1")
        }
    }

    fun init(application: Application) {

        app = application
        sp = application.getSharedPreferences(SP_NAME, Context.MODE_PRIVATE)

        foodInfoDB = Room.databaseBuilder(app, FoodInfoDatabase::class.java, DB_NAME_FOOD_INFO)
            .addMigrations(MIGRATION_1_2)
            .addMigrations(MIGRATION_2_3)
            .addMigrations(MIGRATION_3_4)
            .build()
        foodInfoDao = foodInfoDB.foodInfoDao()

        typeInfoDB = Room.databaseBuilder(app, FoodTypeInfoDatabase::class.java, DB_NAME_FOOD_TYPE_INFO).build()
        typeInfoDao = typeInfoDB.foodTypeInfoDao()

        shelfLifeDB = Room.databaseBuilder(app, ShelfLifeInfoDatabase::class.java, DB_NAME_SHELF_LIFE_INFO).build()
        shelfLifeDao = shelfLifeDB.shelfLifeDao()

        if (!sp.getBoolean("hasInitialized", false)) {
            addInitializedData()
        }

        // 检查版本更新并重置新UI试用状态 后面也许会改成一个更新通告的弹窗
        checkVersionUpdateAndResetUITried()
    }

    fun getPicturePath(uuid: String): String {
        return app.filesDir.absolutePath + "/" + uuid
    }

    fun addFoodInfo(foodInfo: FoodInfo) {
        CoroutineScope(Dispatchers.IO).launch {
            foodInfoDao.insert(foodInfo)
        }
    }

    fun getAllFoodInfo(): List<FoodInfo> {
        return foodInfoDao.getAll()
    }

    fun removeFoodInfo(foodInfo: FoodInfo) {
        foodInfoDao.remove(foodInfo)
    }

    fun getAllTypeInfo(): List<FoodTypeInfo> {
        return typeInfoDao.getAll()
    }

    fun addTypeInfo(typeInfo: FoodTypeInfo) {
        CoroutineScope(Dispatchers.IO).launch {
            typeInfoDao.insert(typeInfo)
        }
    }

    fun removeTypeInfo(typeInfo: FoodTypeInfo) {
        CoroutineScope(Dispatchers.IO).launch {
            typeInfoDao.remove(typeInfo)
        }
    }

    fun getAllShelfLifeInfo(): List<ShelfLifeInfo> {
        return shelfLifeDao.getAll().sortedBy { it.life.toInt() }
    }

    fun addShelfLifeInfo(shelfLifeInfo: ShelfLifeInfo) {
        CoroutineScope(Dispatchers.IO).launch {
            shelfLifeDao.insert(shelfLifeInfo)
        }
    }

    fun removeShelfLifeInfo(shelfLifeInfo: ShelfLifeInfo) {
        CoroutineScope(Dispatchers.IO).launch {
            shelfLifeDao.remove(shelfLifeInfo)
        }
    }

    fun isNotificationEnabled(): Boolean {
        return sp.getBoolean("notification_enabled", false)
    }

    fun setNotificationEnabled(boolean: Boolean) {
        sp.edit().putBoolean("notification_enabled", boolean).apply()
    }

    fun getDaysBeforeNotification(): Int {
        return sp.getInt("days_before_notification", 3)
    }

    fun setDaysBeforeNotification(days: Int) {
        sp.edit().putInt("days_before_notification", days).apply()
    }

    fun setNextNotificationMillis(time: Long) {
        sp.edit().putLong("next_notification_time", time).apply()
    }

    fun getNextNotificationMillis(): Long {
        return sp.getLong("next_notification_time", -1)
    }

    fun setDateFormat(format: String) {
        sp.edit().putString("date_format", format).apply()
    }

    fun getDateFormat(): String {
        return sp.getString("date_format", "yy-MM-dd") ?: "yy-MM-dd"
    }

    fun setThemeOption(option: ThemeOptions) {
        sp.edit().putInt("theme_option", option.int).apply()
    }

    fun getThemeOption(): ThemeOptions {
        val themeValue = sp.getInt("theme_option", 0)
        return ThemeOptions.fromInt(themeValue)
    }

    fun setIsNewUI(isNewUI: Boolean) {
        sp.edit().putBoolean("is_new_ui", isNewUI).apply()
    }

    fun isNewUI(): Boolean {
        return sp.getBoolean("is_new_ui", true)
    }

    fun setNewUITried(isTried: Boolean) {
        sp.edit().putBoolean("is_new_ui_tried", isTried).apply()
    }

    fun isNewUITried(): Boolean {
        return sp.getBoolean("is_new_ui_tried", false)
    }

    fun setIsExtraColumnLayout(isExtraColumn: Boolean) {
        sp.edit().putBoolean("is_extra_column_layout", isExtraColumn).apply()
    }

    fun isExtraColumnLayout(): Boolean {
        return sp.getBoolean("is_extra_column_layout", false)
    }

    fun setPhoneHomePermissionState(state: Int) {
        sp.edit().putInt("phone_home_permission_state", state).apply()
    }

    fun getPhoneHomePermissionState(): Int {
        return sp.getInt("phone_home_permission_state", StatisticPermissionState.INIT.ordinal)
    }

    fun getOrCreateUuid(): String {
        val uuid = sp.getString("uuid", null)
        if (uuid != null) {
            return uuid
        }
        val newUuid = java.util.UUID.randomUUID().toString()
        sp.edit().putString("uuid", newUuid).apply()
        return newUuid
    }

    private fun addInitializedData() {
        CoroutineScope(Dispatchers.IO).launch {

            typeInfoDao.insert(FoodTypeInfo(type = app.getString(R.string.type_fruits_or_vegetables)))
            typeInfoDao.insert(FoodTypeInfo(type = app.getString(R.string.type_meat)))
            typeInfoDao.insert(FoodTypeInfo(type = app.getString(R.string.type_milk)))
            typeInfoDao.insert(FoodTypeInfo(type = app.getString(R.string.type_seafood)))
            typeInfoDao.insert(FoodTypeInfo(type = app.getString(R.string.type_cereal)))
            typeInfoDao.insert(FoodTypeInfo(type = app.getString(R.string.type_can)))
            typeInfoDao.insert(FoodTypeInfo(type = app.getString(R.string.type_condiment)))

            shelfLifeDao.insert(ShelfLifeInfo(life = "3"))
            shelfLifeDao.insert(ShelfLifeInfo(life = "7"))
            shelfLifeDao.insert(ShelfLifeInfo(life = "14"))
            shelfLifeDao.insert(ShelfLifeInfo(life = "30"))
            shelfLifeDao.insert(ShelfLifeInfo(life = "90"))
            shelfLifeDao.insert(ShelfLifeInfo(life = "180"))
            shelfLifeDao.insert(ShelfLifeInfo(life = "360"))

            sp.edit().putBoolean("hasInitialized", true).commit()
        }
    }

    private fun checkVersionUpdateAndResetUITried() {
        try {
            val packageInfo = app.packageManager.getPackageInfo(app.packageName, 0)
            val currentVersionCode = packageInfo.longVersionCode
            val savedVersionCode = sp.getLong("last_version_code", -1L)

            if (savedVersionCode != -1L && savedVersionCode < currentVersionCode) {
                // 版本更新了
                sp.edit()
                    .putBoolean("is_new_ui_tried", false)
                    .putLong("last_version_code", currentVersionCode)
                    .apply()
            } else if (savedVersionCode == -1L) {
                // 首次安装，保存当前版本号
                sp.edit()
                    .putLong("last_version_code", currentVersionCode)
                    .apply()
            }
        } catch (e: Exception) {
            // do nothing
        }
    }
}
