/**
 * Copyright (C) 2025  Antonio Tari
 *
 * This file is a part of Power Ampache 2
 * Ampache Android client application
 * @author Antonio Tari
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package luci.sixsixsix.powerampache2.data.local.datasource

import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.mapNotNull
import luci.sixsixsix.mrlog.MrLog
import luci.sixsixsix.mrlog.MrLog.invoke
import luci.sixsixsix.powerampache2.data.local.MusicDatabase
import luci.sixsixsix.powerampache2.data.local.entities.PlaylistSongEntity
import luci.sixsixsix.powerampache2.data.local.entities.toAlbum
import luci.sixsixsix.powerampache2.data.local.entities.toPlaylist
import luci.sixsixsix.powerampache2.data.local.entities.toPlaylistEntity
import luci.sixsixsix.powerampache2.data.local.entities.toSong
import luci.sixsixsix.powerampache2.di.LocalDataSource
import luci.sixsixsix.powerampache2.domain.datasource.PlaylistsDbDataSource
import luci.sixsixsix.powerampache2.domain.models.AmpacheModel
import luci.sixsixsix.powerampache2.domain.models.Playlist
import luci.sixsixsix.powerampache2.domain.models.PlaylistSongItem
import luci.sixsixsix.powerampache2.domain.models.Song
import luci.sixsixsix.powerampache2.domain.models.isSmartPlaylist
import javax.inject.Inject
import javax.inject.Singleton
import kotlin.collections.map

@LocalDataSource
@Singleton
class PlaylistsDbDataSourceImpl @Inject constructor(db: MusicDatabase): PlaylistsDbDataSource {
    private val dao = db.dao

    override val playlistsFlow: Flow<List<Playlist>>
        get() = dao.playlistsFlow().filterNotNull().mapNotNull { list -> list.map { it.toPlaylist() } }

    override suspend fun getPlaylists(query: String): List<Playlist> =
        dao.searchPlaylists(query).map { it.toPlaylist() }

    override fun getPlaylist(id: String) =
        dao.playlistFlow(id).filterNotNull().mapNotNull { it.toPlaylist() }

    override suspend fun savePlaylistsToDb(
        playlists: List<Playlist>,
        username: String,
        serverUrl: String,
        shouldClearBeforeAdding: Boolean
    ) {
        if (shouldClearBeforeAdding) {
            // avoid clearing if lists are equal, insert will already replace the old versions
            dao.clearPlaylists()
        }

        dao.insertPlaylists(
            playlists.map { it.toPlaylistEntity(username = username, serverUrl = serverUrl) }
        )
    }

    override suspend fun savePlaylistSongsToDb(
        songs: List<Song>,
        playlistId: String,
        username: String,
        serverUrl: String
    ) {
        dao.clearPlaylistSongs(playlistId)
        dao.insertPlaylistSongs(PlaylistSongEntity.newEntries(songs, playlistId, username = username, serverUrl = serverUrl))
    }

    override suspend fun savePlaylistSongRefsToDb(
        songRefs: List<PlaylistSongItem>,
        playlistId: String,
        username: String,
        serverUrl: String
    ) {
        dao.clearPlaylistSongs(playlistId)
        dao.insertPlaylistSongs(
            PlaylistSongEntity.newEntriesFromPlaylistSongItems(
                songRefs,
                playlistId,
                username = username,
                serverUrl = serverUrl
            )
        )
    }

    /**
     * Smartlists are never fetched from db (unless in offline-mode, see OfflineDataSource) to avoid
     * flickering UI, since they are generated by the backend each time.
     */
    override suspend fun getSongsFromPlaylist(playlist: Playlist): List<Song> =
        if (!playlist.isSmartPlaylist())
            dao.getSongsFromPlaylist(playlist.id).map { it.toSong() }
        else listOf()
}
