/**
 * Copyright (C) 2024  Antonio Tari
 *
 * This file is a part of Power Ampache 2
 * Ampache Android client application
 * @author Antonio Tari
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package luci.sixsixsix.powerampache2.presentation.screens_detail.song_detail.components

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.background
import androidx.compose.foundation.basicMarquee
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Cast
import androidx.compose.material3.BottomSheetScaffoldState
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.IconButtonColors
import androidx.compose.material3.IconButtonDefaults
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.runtime.snapshotFlow
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.media3.common.util.UnstableApi
import coil.compose.AsyncImage
import kotlinx.coroutines.launch
import luci.sixsixsix.powerampache2.R
import luci.sixsixsix.powerampache2.common.fontDimensionResource
import luci.sixsixsix.powerampache2.domain.models.Song
import luci.sixsixsix.powerampache2.domain.models.totalTime
import luci.sixsixsix.powerampache2.domain.plugin.info.PluginSongData
import luci.sixsixsix.powerampache2.presentation.common.LikeButton
import luci.sixsixsix.powerampache2.presentation.dialogs.AddToPlaylistOrQueueDialog
import luci.sixsixsix.powerampache2.presentation.dialogs.AddToPlaylistOrQueueDialogOpen
import luci.sixsixsix.powerampache2.presentation.dialogs.AddToPlaylistOrQueueDialogViewModel
import luci.sixsixsix.powerampache2.presentation.dialogs.ShareDialog
import luci.sixsixsix.powerampache2.presentation.dialogs.info.InfoDialogSong
import luci.sixsixsix.powerampache2.presentation.navigation.Ampache2NavGraphs
import luci.sixsixsix.powerampache2.presentation.screens.main.viewmodel.MainEvent
import luci.sixsixsix.powerampache2.presentation.screens.main.viewmodel.MainViewModel

@androidx.annotation.OptIn(UnstableApi::class)
@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun SongDetailContent(
    mainScaffoldState: BottomSheetScaffoldState,
    modifier: Modifier = Modifier,
    mainViewModel: MainViewModel,
    pluginSong: PluginSongData?,
    isChromecastPluginInstalled: Boolean,
    addToPlaylistOrQueueDialogViewModel: AddToPlaylistOrQueueDialogViewModel
) {
    val currentSongState by mainViewModel.currentSongStateFlow().collectAsState()
    val scope = rememberCoroutineScope()
    val buttonsTint = MaterialTheme.colorScheme.onSurfaceVariant

    var playlistsDialogOpen by remember { mutableStateOf(AddToPlaylistOrQueueDialogOpen(false)) }
    if (playlistsDialogOpen.isOpen) {
        if (playlistsDialogOpen.songs.isNotEmpty()) {
            AddToPlaylistOrQueueDialog(playlistsDialogOpen.songs,
                onDismissRequest = {
                    playlistsDialogOpen = AddToPlaylistOrQueueDialogOpen(false)
                },
                mainViewModel = mainViewModel,
                viewModel = addToPlaylistOrQueueDialogViewModel
            )
        }
    }

    var infoDialogOpen by remember { mutableStateOf(false) }
    if (infoDialogOpen) {
        currentSongState?.let { song ->
            InfoDialogSong(song = song, pluginSong) {
                infoDialogOpen = false
            }
        }
    }

    var isOffline by remember { mutableStateOf(false) }

    var songToShare: Song? by remember { mutableStateOf(null) }
    AnimatedVisibility(songToShare != null) {
        songToShare?.let { songS ->
            ShareDialog(
                onShareWeb = {
                    mainViewModel.onEvent(MainEvent.OnShareSongWebUrl(songS))
                    songToShare = null
                },
                onSharePowerAmpache = {
                    mainViewModel.onEvent(MainEvent.OnShareSong(songS))
                    songToShare = null
                },
                onDismissRequest = {
                    songToShare = null
                }
            )
        }
    }

    var isImageScaleFit by remember { mutableStateOf(false) }

    val coverImage = currentSongState?.imageUrl ?: pluginSong?.imageUrl?.let { imageUrl ->
        imageUrl.ifBlank { pluginSong.imageAlbum.ifBlank { pluginSong.imageArtist } }
    }

    Column(
        modifier = modifier
            .fillMaxSize()
            .background(MaterialTheme.colorScheme.surface)
    ) {
        Box(modifier = Modifier.weight(1f).fillMaxWidth(),
            contentAlignment = Alignment.TopEnd
        ) {
            SongAlbumCoverArt(
                modifier = Modifier
                    //.weight(1f)
                    .fillMaxWidth()
                    .clickable { isImageScaleFit = !isImageScaleFit },
                artUrl = coverImage,
                contentDescription = currentSongState?.title,
                isImageScaleFit = isImageScaleFit,
                onSwipeLeft = { mainViewModel.onEvent(MainEvent.SkipNext) },
                onSwipeRight = { mainViewModel.onEvent(MainEvent.SkipPrevious) }
            )

            if (isChromecastPluginInstalled) {
                IconButton(
                    modifier = Modifier.padding(8.dp),
                    colors = IconButtonDefaults.iconButtonColors(
                        containerColor = MaterialTheme.colorScheme.onBackground.copy(alpha = 0.5f),
                        contentColor = MaterialTheme.colorScheme.background
                    ),
                    onClick = { mainViewModel.onEvent(MainEvent.OnCastPress) }) {
                    Icon(imageVector = Icons.Default.Cast, contentDescription = "Cast")
                }
            }
        }

        Spacer(modifier = Modifier.height(16.dp))

        // Title and artist, like button
        Box(Modifier.padding(horizontal = dimensionResource(id = R.dimen.player_screen_title_padding))) {
            Column {
                val artistName = if (currentSongState?.artists?.isNotEmpty() == true) {
                    currentSongState?.artists?.joinToString { it.name } ?: currentSongState?.artist?.name
                } else {
                    currentSongState?.artist?.name
                }
                Text(
                    text = artistName ?: "",
                    fontWeight = FontWeight.SemiBold,
                    fontSize = fontDimensionResource(id = R.dimen.player_artistName_size),
                    color = MaterialTheme.colorScheme.onSurfaceVariant,
                    overflow = TextOverflow.Ellipsis,
                    maxLines = 1,
                    modifier = Modifier
                        .fillMaxWidth()
                        .basicMarquee()
                )
                Spacer(modifier = Modifier.height(2.dp))

                Text(
                    text = currentSongState?.title ?: "",
                    fontWeight = FontWeight.Normal,
                    fontSize = fontDimensionResource(id = R.dimen.player_songTitle_size),
                    color = MaterialTheme.colorScheme.onSurface,
                    overflow = TextOverflow.Ellipsis,
                    maxLines = 1,
                    modifier = Modifier.fillMaxWidth()
                )
            }

            LikeButton(
                modifier = Modifier
                    .align(Alignment.BottomEnd)
                    .size(36.dp),
                isLikeLoading = mainViewModel.state.isLikeLoading,
                isFavourite = currentSongState?.flag == 1
            ) {
                mainViewModel.onEvent(MainEvent.FavouriteSong)
            }
        }

        Spacer(modifier = Modifier.height(16.dp))

        currentSongState?.let { song ->
            mainViewModel.isOfflineSong(song) {
                isOffline = it
            }
            SongDetailButtonRow(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(horizontal = dimensionResource(id = R.dimen.player_screen_padding)),
                isOffline = isOffline,
                tint = buttonsTint
            ) { event ->
                when(event) {
                    SongDetailButtonEvents.SHARE_SONG -> {
                        songToShare = song
                    }
                    SongDetailButtonEvents.DOWNLOAD_SONG ->
                        mainViewModel.onEvent(MainEvent.OnDownloadSong(song))
                    SongDetailButtonEvents.ADD_SONG_TO_PLAYLIST_OR_QUEUE ->
                        playlistsDialogOpen = AddToPlaylistOrQueueDialogOpen(true, listOf(song))
                    SongDetailButtonEvents.GO_TO_ALBUM ->
                        currentSongState?.album?.id?.let { albumId ->
                            Ampache2NavGraphs.navigateToAlbum(albumId = albumId)
                            scope.launch {
                                mainScaffoldState.bottomSheetState.partialExpand()
                            }
                        }
                    SongDetailButtonEvents.GO_TO_ARTIST ->
                        currentSongState?.artist?.id?.let { artistId ->
                            Ampache2NavGraphs.navigateToArtist(artistId = artistId)
                            scope.launch {
                                mainScaffoldState.bottomSheetState.partialExpand()
                            }
                        }
                    SongDetailButtonEvents.SHOW_INFO ->
                        infoDialogOpen = true
                    SongDetailButtonEvents.DELETE_DOWNLOADED_SONG -> {
                        mainViewModel.onEvent(MainEvent.OnDownloadedSongDelete(song = song))
                        // TODO BREAKING_RULE anti-pattern. verify the song is actually deleted
                        isOffline = false
                    }
                }
            }
        }

        Spacer(modifier = Modifier.height(12.dp))

        SongDetailPlayerBar(
            progress = mainViewModel.progress,
            durationStr = currentSongState?.totalTime() ?: "",
            progressStr = mainViewModel.progressStr,
            isPlaying = mainViewModel.isPlaying,
            isPlayLoading = mainViewModel.isPlayLoading(),
            shuffleOn = mainViewModel.shuffleOn,
            repeatMode = mainViewModel.repeatMode,
            isBuffering = mainViewModel.isBuffering,
            modifier = Modifier
                .fillMaxWidth()
                .wrapContentHeight()
                .padding(horizontal = dimensionResource(id = R.dimen.player_screen_title_padding))
        ) { event ->
            mainViewModel.onEvent(event)
        }

        Spacer(modifier = Modifier.height(16.dp))
    }
}

@Composable
fun SongAlbumCoverArt(
    modifier: Modifier,
    artUrl: String?,
    contentDescription: String?,
    onSwipeLeft: () -> Unit,
    onSwipeRight: () -> Unit,
    isImageScaleFit: Boolean = true
) {
    var currentPage by remember { mutableIntStateOf(1500) }
    val pagerState = rememberPagerState(
        initialPage = currentPage,
        pageCount = { 3000 }
    )
    LaunchedEffect(pagerState) {
        snapshotFlow { pagerState.currentPage }.collect { page ->
            if (page > currentPage) {
                onSwipeLeft()
            } else if (page < currentPage) {
                onSwipeRight()
            }
            currentPage = page
        }
    }

    HorizontalPager(
        modifier = modifier,
        state = pagerState
    ) {
        AsyncImage(
            modifier = Modifier
                //.weight(1f)
                .fillMaxWidth(),
            model = artUrl,
            contentScale = if (isImageScaleFit) ContentScale.Fit else ContentScale.Crop,
            placeholder = painterResource(id = R.drawable.placeholder_album),
            error = painterResource(id = R.drawable.placeholder_album),
            contentDescription = contentDescription,
        )
    }
}
