/**
 * Copyright (C) 2025  Antonio Tari
 *
 * This file is a part of Power Ampache 2
 * Ampache Android client application
 * @author Antonio Tari
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package luci.sixsixsix.powerampache2.data.offlinemode

import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.mapNotNull
import luci.sixsixsix.powerampache2.data.local.MusicDatabase
import luci.sixsixsix.powerampache2.data.local.entities.toArtist
import luci.sixsixsix.powerampache2.data.local.entities.toSong
import luci.sixsixsix.powerampache2.di.OfflineModeDataSource
import luci.sixsixsix.powerampache2.domain.datasource.ArtistsOfflineModeDataSource
import luci.sixsixsix.powerampache2.domain.models.Artist
import luci.sixsixsix.powerampache2.domain.models.Genre
import luci.sixsixsix.powerampache2.domain.models.Song
import javax.inject.Inject

@OfflineModeDataSource
class ArtistsOfflineDataSourceImpl @Inject constructor(
    db: MusicDatabase,
): ArtistsOfflineModeDataSource {
    private val dao = db.dao

    override val recommendedFlow: Flow<List<Artist>>
        get() = dao.getRecommendedOfflineArtists().mapNotNull { list -> list.map { it.toArtist() } }

    override suspend fun getArtist(artistId: String): Artist? =
        dao.generateOfflineArtist(artistId)?.toArtist()

    override suspend fun getArtists(owner: String, query: String): List<Artist> =
        dao.generateOfflineArtists(owner, query).map { it.toArtist() }

    override suspend fun getArtistsByGenre(genreId: Genre): List<Artist> {
        TODO("Not yet implemented")
    }

    override suspend fun likeArtist(id: String, like: Boolean): Boolean {
        TODO("Not yet implemented")
    }

    override suspend fun getMostPlayedArtists(): List<Artist> =
        dao.getMostPlayedOfflineArtists().map { it.toArtist() }

    override suspend fun getSongsFromArtist(artistId: String): List<Song> =
        dao.getOfflineSongsFromArtist(artistId).map { it.toSong() }

    override suspend fun getRecommendedArtists(baseArtistId: String): List<Artist> =
        dao.getRecommendedOfflineArtists(baseArtistId).map { it.toArtist() }
}
