/**
 * Copyright (C) 2024  Antonio Tari
 *
 * This file is a part of Power Ampache 2
 * Ampache Android client application
 * @author Antonio Tari
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package luci.sixsixsix.powerampache2.data

import android.app.Application
import androidx.media3.common.PlaybackException
import androidx.media3.common.util.UnstableApi
import androidx.media3.datasource.HttpDataSource
import com.google.gson.Gson
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.FlowCollector
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.launch
import luci.sixsixsix.mrlog.L
import luci.sixsixsix.powerampache2.R
import luci.sixsixsix.powerampache2.common.Resource
import luci.sixsixsix.powerampache2.domain.datasource.DbDataSource
import luci.sixsixsix.powerampache2.domain.errors.AmpachePreferenceException
import luci.sixsixsix.powerampache2.domain.errors.ErrorHandler
import luci.sixsixsix.powerampache2.domain.errors.ErrorType
import luci.sixsixsix.powerampache2.domain.errors.MusicException
import luci.sixsixsix.powerampache2.domain.errors.ScrobbleException
import luci.sixsixsix.powerampache2.domain.errors.ServerUrlNotInitializedException
import luci.sixsixsix.powerampache2.domain.errors.UserNotEnabledException
import luci.sixsixsix.powerampache2.domain.utils.ConfigProvider
import luci.sixsixsix.powerampache2.player.MusicPlaylistManager
import retrofit2.HttpException
import java.io.IOException
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
class ErrorHandlerImpl @Inject constructor(
    private val playlistManager: MusicPlaylistManager,
    private val db: DbDataSource,
    configProvider: ConfigProvider,
    private val applicationContext: Application,
    applicationCoroutineScope: CoroutineScope
): ErrorHandler {
    private var isErrorHandlingEnabled = configProvider.ENABLE_ERROR_LOG

    init {
        applicationCoroutineScope.launch {
            db.settingsFlow
                .map { it.enableRemoteLogging }
                .distinctUntilChanged()
                .collectLatest {
                    if (it != isErrorHandlingEnabled) {
                        isErrorHandlingEnabled = it
                        L.e("ERROR hand enabled? $isErrorHandlingEnabled")
                    }
                }
        }
    }

    @androidx.annotation.OptIn(UnstableApi::class)
    override suspend fun <T> invoke(
        label: String,
        e: Throwable,
        fc: FlowCollector<Resource<T>>?,
        onError: (message: String, e: Throwable) -> Unit
    ) {
        // Blocking errors for server url not initialized
        // this is an exception generated by the interceptor for when a url has not yet been
        // initialized by the user
        if (e is MusicException && e.musicError.isServerUrlNotInitialized()) {
            L("ServerUrlNotInitializedException")
            fc?.emit(Resource.Loading(false))
            return
        }

        val exceptionString = e.printStackTrace()

        var readableMessage: String? = null
        StringBuilder(label)
            .append(if (label.isBlank()) "" else " - ")
            .append(
                when (e) {
                    is UserNotEnabledException -> {
                        readableMessage = applicationContext.getString(R.string.error_user_notEnabled)
                        "PlaybackException \n $exceptionString"
                    }
                    is HttpDataSource.InvalidResponseCodeException -> {
                        readableMessage = "${applicationContext.getString(R.string.error_cannotConnect)}\nResponse code: ${e.responseCode}"
                        "HttpDataSource.InvalidResponseCodeException \n$label\n $exceptionString"
                    }

                    is HttpDataSource.HttpDataSourceException -> {
                        readableMessage = applicationContext.getString(R.string.error_cannotConnect)
                        "HttpDataSource.HttpDataSourceException \n$readableMessage\n $exceptionString"
                    }

                    is PlaybackException -> {
                        readableMessage =
                            if (e.errorCode != PlaybackException.ERROR_CODE_PARSING_CONTAINER_UNSUPPORTED)
                                "Error Code: ${e.errorCode}. ${applicationContext.getString(R.string.error_playback_exception)}\n$exceptionString"
                            else ""

                        "PlaybackException \n$readableMessage\n $exceptionString"
                    }

                    is IOException -> {
                        readableMessage = applicationContext.getString(R.string.error_io_exception)
                        "cannot load data IOException $exceptionString"
                    }

                    is HttpException -> {
                        readableMessage = e.localizedMessage
                        "cannot load data HttpException $exceptionString"
                    }

                    is AmpachePreferenceException -> {
                        readableMessage = applicationContext.getString(R.string.error_cannotEditPreference)
                        "$readableMessage $exceptionString"
                    }

                    is ServerUrlNotInitializedException ->
                        "ServerUrlNotInitializedException $exceptionString"

                    is ScrobbleException -> {
                        readableMessage = ""
                        ""
                    }
                    is MusicException -> {
                        when (e.musicError.getErrorType()) {
                            ErrorType.ACCOUNT -> {
                                // clear session and try to autologin using the saved credentials
                                db.clearSession()
                                readableMessage = e.musicError.errorMessage
                            }

                            ErrorType.EMPTY ->
                                readableMessage =
                                    applicationContext.getString(R.string.error_empty_result)

                            ErrorType.DUPLICATE ->
                                readableMessage =
                                    applicationContext.getString(R.string.error_duplicate)

                            ErrorType.Other ->
                                readableMessage = e.musicError.errorMessage

                            ErrorType.SYSTEM ->
                                readableMessage = e.musicError.errorMessage
                        }
                        e.musicError.toString()
                    }

                    else -> {
                        readableMessage = e.localizedMessage
                        "generic exception $exceptionString"
                    }
                }
            ).toString().apply {
                // check on error on the emitted data for detailed logging
                fc?.emit(Resource.Error(message = this, exception = e))
                // log and report error here
                logError(e, this)
                playlistManager.updateErrorLogMessage(this)
                // readable message here
                readableMessage?.let {
                    // TODO find a better way to not show verbose info
                    //  ie. session expired for timestamp
                    if (e is HttpException || e is IOException) {
                        playlistManager.updateUserMessage(applicationContext.getString(R.string.error_offline))
                    } else if (!readableMessage.lowercase().contains("timestamp") &&
                        !readableMessage.lowercase().contains("expired") &&
                        !readableMessage.lowercase().contains("session")) {
                        playlistManager.updateUserMessage(readableMessage)
                    } else if (e is UserNotEnabledException) {
                        playlistManager.updateUserMessage(readableMessage)
                    }
                }
                onError(this, e)
                L.e(readableMessage, e)
            }
    }

    override suspend fun logError(e: Throwable, message: String) = logError(message = "${e.stackTraceToString()}\n$message")

    override suspend fun logError(message: String) {
        try {
            // TODO DISABLING HTTP ERROR REPORTING for now
//            if (isErrorHandlingEnabled && !configProvider.URL_ERROR_LOG.isNullOrBlank()) {
//               api.sendErrorReport(apiPasteCode = "${getVersionInfoString(applicationContext)}\n$message")
//            }
            if (isErrorHandlingEnabled) {
                playlistManager.updateErrorLogMessage(message)
            }
        } catch (e: Exception) {
            if (e is HttpException)
                L.e(e.stackTraceToString(), e.message(), e.localizedMessage, e.code(), e.response())
            else
                L.e(e.stackTraceToString(), e.localizedMessage)
        }
    }
}
