package tools;
/*
Version 1.0, 16-08-2021, First release by Livio (javalc6@gmail.com)

IMPORTANT NOTICE, please read:

This software is licensed under the terms of the GNU GENERAL PUBLIC LICENSE,
please read the enclosed file license.txt or https://www.gnu.org/licenses/old-licenses/gpl-2.0-standalone.html

Note that this software is freeware and it is not designed, licensed or intended
for use in mission critical, life support and military purposes.

The use of this software is at the risk of the user.
*/
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;
import java.util.zip.GZIPInputStream;

import android.os.Build;
import android.util.Log;

public final class WebFetch {
    private String proxy_host = null;
	private int proxy_port = -1;
	private String user_agent = null;
    private String ETag = null;
    private String LastMod = null;
    private URL server = null;
	private final String tag = "WebFetch";
    private InputStream inStream;
    private String accept = null;

    public HttpURLConnection connect(String urlName) throws IOException {//http
        server = new URL(urlName);
        try { // check if proxy is set
            List<Proxy> proxies = ProxySelector.getDefault().select(new URI(urlName));
            if (!proxies.isEmpty()) {
                Proxy px = proxies.get(0);
                if (!px.equals(Proxy.NO_PROXY)) {
                    InetSocketAddress addr = (InetSocketAddress) px.address();
                    if (addr != null) {
                        proxy_host = addr.getHostName();
                        proxy_port = addr.getPort();
                    }
                }
            }
        } catch (URISyntaxException e) {// do nothing
        } catch (IllegalArgumentException e) {// do nothing - generated by bad proxy settings
            System.out.println("IllegalArgumentException raised by connect()");
        }
        HttpURLConnection conn;
        if (proxy_host != null) {
            Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxy_host, proxy_port));//http
            conn = (HttpURLConnection) server.openConnection(proxy);//http
        } else {
            conn = (HttpURLConnection) server.openConnection();//http
        }
		conn.setReadTimeout(15000);//avoid infinite timeout
        conn.setRequestProperty("Host", server.getHost());//http
        if ((accept != null) && (!accept.isEmpty()))
            conn.setRequestProperty("Accept", accept);
        conn.addRequestProperty("Accept-Encoding", "gzip,deflate");//http
        if ((ETag != null) && (!ETag.isEmpty()))
            conn.setRequestProperty("If-None-Match", ETag);
        if ((LastMod != null) && (!LastMod.isEmpty()))
            conn.setRequestProperty("If-Modified-Since", LastMod);
		if (user_agent == null)//http
			user_agent = "WebFetch/1 (Linux; U; Android " + Build.VERSION.RELEASE +//http
			"; " + Build.MODEL + "/" + Build.MANUFACTURER + ")";//http
        conn.setRequestProperty("User-Agent", user_agent);//http
        conn.setInstanceFollowRedirects(false);//let's handle redirects ourself
        return conn;
    }

    public WebResponse fetchURL(String urlName, int requestId) throws IOException {
        return fetchURL(urlName, requestId, null);
    }

    // fetchURL(String urlName)
    public WebResponse fetchURL(String urlName, int requestId, String[] custom_header) throws IOException {//added custom_header to support optiona custom HTTP header
        HttpURLConnection conn = connect(urlName); //http
		conn.setReadTimeout(15000);//avoid infinite timeout
        if (custom_header != null)
            conn.addRequestProperty(custom_header[0], custom_header[1]);//http

        WebResponse resp = openStream(conn, requestId);
        if (inStream != null) {
            BufferedReader is;
            if (!resp.coding.isEmpty())
                is = new BufferedReader(new InputStreamReader(inStream, resp.coding));
            else is = new BufferedReader(new InputStreamReader(inStream));
            StringBuilder result = new StringBuilder(8192);
            String line;
            while ((line = is.readLine())!= null) {
                result.append(line);
            }
            resp.result = result;
            is.close();
            closeStream();
        }
        conn.disconnect();

        return resp;
    }

// openStream(String urlName, int requestId)
	public WebResponse openStream(HttpURLConnection conn, int requestId) throws IOException {
        int responseCode;
// the following block of code retries connection in case of remote server closing the connection without any reason
// some proxies send TCP/IP FIN in unexpected way, causing RST
		int n = 0;
  	  	do {
            responseCode = conn.getResponseCode();//http
//            Log.i(tag,"responseCode: "+responseCode);
            if ((responseCode == HttpURLConnection.HTTP_MOVED_TEMP) ||
                    (responseCode == HttpURLConnection.HTTP_MOVED_PERM) ||
                    (responseCode == 307) ||//response code 307 aggiunto per gestire https://www.lequipe.fr/Xml/actu_rss.xml
                    (responseCode == 308)) {//response code 308 aggiunto per gestire http://economico.sapo.pt/rss/ultimas
                conn.disconnect(); //http
// rewrite URL with Location
                String urlName = conn.getHeaderField("Location"); //http
                if (urlName.startsWith("/")) {//controlla se il path è relativo, in tal caso renderlo assoluto
                    urlName = new URL(server, urlName).toExternalForm();
                }
                Log.i(tag,"Location: "+urlName);
                n++;
                if (n > 6) // too many retries, aumentato a 6 per gestire https://www.lequipe.fr/Xml/actu_rss.xml
                    return new WebResponse(null, null, null, responseCode, conn.getResponseMessage(), requestId);
                conn = connect(urlName); // retry a new connection
            } else break; // break in case of final response
  	  	} while (true);


//		Log.i(tag,"getContentType: "+entity.getContentType());
		String ctype, cenc = "", contype = "";
        if (conn.getContentType() != null) //http
            contype = conn.getContentType();//http
		int hc = contype.indexOf(';');
		if (hc == -1) {
			ctype = contype.trim();
		} else {
			ctype = contype.substring(0, hc).trim();
			int charset = contype.indexOf("charset=");
			if (charset != -1)	{
				cenc = contype.substring(charset+8).trim();
			}
		}
        String diagnostic = conn.getResponseMessage();
        if (responseCode == HttpURLConnection.HTTP_SEE_OTHER)
            diagnostic += " " + conn.getHeaderField("Location");
        inStream = (responseCode >= 400) ? conn.getErrorStream() : conn.getInputStream();//http
        if (inStream == null)//http
            return new WebResponse(null, null, null, responseCode, diagnostic, requestId);//http
        if ((conn.getContentEncoding() != null) && ("gzip".equals(conn.getContentEncoding()))) {//http
            try {
                inStream = new GZIPInputStream(inStream);
            } catch (java.io.EOFException ex) { //workaround for android bug: https://code.google.com/p/android/issues/detail?id=58637
// do nothing				
            }
        }
		
        String etag = conn.getHeaderField("ETag");
		String lastmod = conn.getHeaderField("Last-Modified");
        if ((etag != null) || (lastmod != null))
            return new WebResponse(null, ctype, cenc, responseCode, diagnostic, requestId, etag, lastmod);
        else {
			return new WebResponse(null, ctype, cenc, responseCode, diagnostic, requestId);
		}
	}

    // getStream()
    public InputStream getStream() {
        return inStream;
    }

    // closeStream()
    public void closeStream() throws IOException {
        if (inStream != null)
            inStream.close();
    }


    public void set_UA(String user_agent) {
        this.user_agent = user_agent;
	}
    public void set_ETag(String ETag) {
        this.ETag = ETag;
    }
    public void set_LastMod(String LastMod) {
        this.LastMod = LastMod;
    }
    public void set_Accept(String accept) {
        this.accept = accept;
    }
}
