"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.run = exports.runExecutor = exports.validateProject = exports.printRunHelp = void 0;
const tslib_1 = require("tslib");
const yargsParser = require("yargs-parser");
const params_1 = require("../shared/params");
const print_help_1 = require("../shared/print-help");
const workspace_1 = require("../shared/workspace");
const chalk = require("chalk");
const logger_1 = require("../shared/logger");
function throwInvalidInvocation() {
    throw new Error(`Specify the project name and the target (e.g., nx run proj:build)`);
}
function parseRunOpts(cwd, args, defaultProjectName) {
    const runOptions = params_1.convertToCamelCase(yargsParser(args, {
        boolean: ['help', 'prod'],
        string: ['configuration', 'project'],
        alias: {
            c: 'configuration',
        },
    }));
    const help = runOptions.help;
    if (!runOptions._ || !runOptions._[0]) {
        throwInvalidInvocation();
    }
    // eslint-disable-next-line prefer-const
    let [project, target, configuration] = runOptions._[0].split(':');
    if (!project && defaultProjectName) {
        logger_1.logger.debug(`No project name specified. Using default project : ${chalk.bold(defaultProjectName)}`);
        project = defaultProjectName;
    }
    if (runOptions.configuration) {
        configuration = runOptions.configuration;
    }
    if (runOptions.prod) {
        configuration = 'production';
    }
    if (runOptions.project) {
        project = runOptions.project;
    }
    if (!project || !target) {
        throwInvalidInvocation();
    }
    const res = { project, target, configuration, help, runOptions };
    delete runOptions['help'];
    delete runOptions['_'];
    delete runOptions['c'];
    delete runOptions['configuration'];
    delete runOptions['prod'];
    delete runOptions['project'];
    return res;
}
function printRunHelp(opts, schema) {
    print_help_1.printHelp(`nx run ${opts.project}:${opts.target}`, schema);
}
exports.printRunHelp = printRunHelp;
function validateProject(workspace, projectName) {
    const project = workspace.projects[projectName];
    if (!project) {
        throw new Error(`Could not find project "${projectName}"`);
    }
}
exports.validateProject = validateProject;
function isPromise(v) {
    var _a;
    return typeof ((_a = v) === null || _a === void 0 ? void 0 : _a.then) === 'function';
}
function isAsyncIterator(v) {
    var _a;
    return typeof ((_a = v) === null || _a === void 0 ? void 0 : _a[Symbol.asyncIterator]) === 'function';
}
function promiseToIterator(v) {
    return tslib_1.__asyncGenerator(this, arguments, function* promiseToIterator_1() {
        yield yield tslib_1.__await(yield tslib_1.__await(v));
    });
}
function iteratorToProcessStatusCode(i) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        let success;
        // This is a workaround to fix an issue that only happens with
        // the @angular-devkit/build-angular:browser builder. Starting
        // on version 12.0.1, a SASS compilation implementation was
        // introduced making use of workers and it's unref()-ing the worker
        // too early, causing the process to exit early in environments
        // like CI or when running Docker builds.
        const keepProcessAliveInterval = setInterval(() => { }, 1000);
        try {
            let prev;
            let current;
            do {
                prev = current;
                current = yield i.next();
            } while (!current.done);
            success =
                current.value !== undefined || !prev
                    ? current.value.success
                    : prev.value.success;
            return success ? 0 : 1;
        }
        finally {
            clearInterval(keepProcessAliveInterval);
        }
    });
}
function createImplicitTargetConfig(proj, targetName) {
    return {
        executor: '@nrwl/workspace:run-script',
        options: {
            script: targetName,
        },
    };
}
function runExecutorInternal({ project, target, configuration, }, options, root, cwd, workspace, isVerbose, printHelp) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        validateProject(workspace, project);
        const ws = new workspace_1.Workspaces(root);
        const proj = workspace.projects[project];
        const targetConfig = proj.targets
            ? proj.targets[target]
            : createImplicitTargetConfig(proj, target);
        if (!targetConfig) {
            throw new Error(`NX Cannot find target '${target}' for project '${project}'`);
        }
        configuration = configuration !== null && configuration !== void 0 ? configuration : targetConfig.defaultConfiguration;
        const [nodeModule, executor] = targetConfig.executor.split(':');
        const { schema, implementationFactory } = ws.readExecutor(nodeModule, executor);
        if (printHelp) {
            printRunHelp({ project, target }, schema);
            process.exit(0);
        }
        const combinedOptions = params_1.combineOptionsForExecutor(options, configuration, targetConfig, schema, project, ws.relativeCwd(cwd), isVerbose);
        if (ws.isNxExecutor(nodeModule, executor)) {
            const implementation = implementationFactory();
            const r = implementation(combinedOptions, {
                root,
                target: targetConfig,
                workspace,
                projectName: project,
                targetName: target,
                configurationName: configuration,
                cwd,
                isVerbose,
            });
            if (isPromise(r)) {
                return promiseToIterator(r);
            }
            else if (isAsyncIterator(r)) {
                return r;
            }
            else {
                throw new TypeError(`NX Executor "${targetConfig.executor}" should return either a Promise or an AsyncIterator`);
            }
        }
        else {
            require('../compat/compat');
            const observable = yield (yield Promise.resolve().then(() => require('./ngcli-adapter'))).scheduleTarget(root, {
                project,
                target,
                configuration,
                runOptions: combinedOptions,
                executor: targetConfig.executor,
            }, isVerbose);
            const { eachValueFrom } = require('rxjs-for-await');
            return eachValueFrom(observable);
        }
    });
}
/**
 * Loads and invokes executor.
 *
 * This is analogous to invoking executor from the terminal, with the exception
 * that the params aren't parsed from the string, but instead provided parsed already.
 *
 * Apart from that, it works the same way:
 *
 * - it will load the workspace configuration
 * - it will resolve the target
 * - it will load the executor and the schema
 * - it will load the options for the appropriate configuration
 * - it will run the validations and will set the default
 * - and, of course, it will invoke the executor
 *
 * Example:
 *
 * ```typescript
 * for await (const s of await runExecutor({project: 'myproj', target: 'serve'}, {watch: true}, context)) {
 *   // s.success
 * }
 * ```
 *
 * Note that the return value is a promise of an iterator, so you need to await before iterating over it.
 */
function runExecutor(targetDescription, options, context) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        return yield runExecutorInternal(targetDescription, options, context.root, context.cwd, context.workspace, context.isVerbose, false);
    });
}
exports.runExecutor = runExecutor;
function run(cwd, root, args, isVerbose) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const ws = new workspace_1.Workspaces(root);
        return params_1.handleErrors(isVerbose, () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const workspace = ws.readWorkspaceConfiguration();
            const defaultProjectName = ws.calculateDefaultProjectName(cwd, workspace);
            const opts = parseRunOpts(cwd, args, defaultProjectName);
            return iteratorToProcessStatusCode(yield runExecutorInternal(opts, opts.runOptions, root, cwd, workspace, isVerbose, opts.help));
        }));
    });
}
exports.run = run;
//# sourceMappingURL=run.js.map