"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.outputsHashesMatch = exports.recordOutputsHash = void 0;
const tslib_1 = require("tslib");
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
const fastGlob = require("fast-glob");
const workspace_root_1 = require("../../utils/workspace-root");
const logger_1 = require("./logger");
const collapse_expanded_outputs_1 = require("../../utils/collapse-expanded-outputs");
const watcher = require('@parcel/watcher');
/**
 * We cannot start a file watcher right away because the IO write that has just
 * completed hasn't triggered the event yet.
 *
 * If we start the file watcher without the delay, we will see an invalidation
 * of the write that is the write itself.
 */
const FILE_WATCHING_DELAY = 1000;
const errors = {};
const subscriptions = {};
const recordedHashes = {};
const numberOfExpandedOutputs = {};
function recordOutputsHash(_outputs, hash) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const outputs = yield normalizeOutputs(_outputs);
        numberOfExpandedOutputs[hash] = outputs.length;
        yield removeSubscriptionsForOutputs(outputs);
        // skip any recording. errors disable the optimization with the file restoration.
        if (anyErrorsAssociatedWithOutputs(outputs)) {
            return;
        }
        for (const output of outputs) {
            recordedHashes[output] = hash;
        }
        yield createSubscriptionsForOutputs(outputs);
    });
}
exports.recordOutputsHash = recordOutputsHash;
function outputsHashesMatch(_outputs, hash) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const outputs = yield normalizeOutputs(_outputs);
        if (outputs.length !== numberOfExpandedOutputs[hash])
            return false;
        for (const output of outputs) {
            if (recordedHashes[output] !== hash)
                return false;
        }
        return true;
    });
}
exports.outputsHashesMatch = outputsHashesMatch;
function anyErrorsAssociatedWithOutputs(outputs) {
    for (const output of outputs) {
        if (errors[output])
            return true;
    }
    return false;
}
/**
 * A subscription that starts in FILE_WATCHING_DELAY but is
 * cancelable before that as well.
 */
function delayedSubscription(output) {
    let subscription;
    const handle = setTimeout(() => tslib_1.__awaiter(this, void 0, void 0, function* () {
        subscription = watcher.subscribe(output, (err) => {
            if (err) {
                logger_1.serverLogger.watcherLog(`File watcher '${output}' threw an error.`);
                console.error(err);
                errors[output] = err;
                if (subscriptions[output]) {
                    subscriptions[output].unsubscribe();
                }
            }
            recordedHashes[output] = undefined;
        });
    }), FILE_WATCHING_DELAY);
    return {
        unsubscribe: () => tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (subscription) {
                yield (yield subscription).unsubscribe();
            }
            else {
                clearTimeout(handle);
            }
        }),
    };
}
// reduce files to folders and expand globs
function normalizeOutputs(outputs) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        return (yield Promise.all(outputs.map((o) => normalizeOutput(o, true)))).flat();
    });
}
function normalizeOutput(output, expand) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        try {
            const res = yield (0, fs_extra_1.lstat)((0, path_1.join)(workspace_root_1.workspaceRoot, output));
            if (res.isFile()) {
                return [(0, path_1.dirname)(output)];
            }
            else if (res.isDirectory()) {
                return [output];
            }
            else {
                return [];
            }
        }
        catch (e) {
            if (expand) {
                const expanded = (0, collapse_expanded_outputs_1.collapseExpandedOutputs)(yield fastGlob(output, { cwd: workspace_root_1.workspaceRoot, dot: true }));
                return (yield Promise.all(expanded.map((e) => normalizeOutput(e, false)))).flat();
            }
            else {
                return [];
            }
        }
    });
}
function createSubscriptionsForOutputs(normalizedOutputs) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        for (const output of normalizedOutputs) {
            subscriptions[output] = delayedSubscription(output);
        }
    });
}
function removeSubscriptionsForOutputs(outputs) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const unsubscribes = [];
        for (const output of outputs) {
            if (subscriptions[output]) {
                unsubscribes.push(subscriptions[output].unsubscribe());
            }
            delete subscriptions[output];
        }
        yield Promise.all(unsubscribes);
    });
}
//# sourceMappingURL=output-watcher.js.map