/*!
 * (C) Ionic http://ionicframework.com - MIT License
 */
import { Component, Host, Listen, Prop, h } from '@stencil/core';
import { getIonMode } from '../../global/ionic-global';
import { safeCall } from '../../utils/overlays';
import { getClassMap } from '../../utils/theme';
/**
 * @internal
 */
export class SelectPopover {
  constructor() {
    /**
     * An array of options for the popover
     */
    this.options = [];
  }
  onSelect(ev) {
    this.setChecked(ev);
    this.callOptionHandler(ev);
  }
  findOptionFromEvent(ev) {
    const { options } = this;
    return options.find((o) => o.value === ev.target.value);
  }
  /**
   * When an option is selected we need to get the value(s)
   * of the selected option(s) and return it in the option
   * handler
   */
  callOptionHandler(ev) {
    const option = this.findOptionFromEvent(ev);
    const values = this.getValues(ev);
    if (option === null || option === void 0 ? void 0 : option.handler) {
      safeCall(option.handler, values);
    }
  }
  /**
   * This is required when selecting a radio that is already
   * selected because it will not trigger the ionChange event
   * but we still want to close the popover
   */
  rbClick(ev) {
    this.callOptionHandler(ev);
  }
  setChecked(ev) {
    const { multiple } = this;
    const option = this.findOptionFromEvent(ev);
    // this is a popover with checkboxes (multiple value select)
    // we need to set the checked value for this option
    if (multiple && option) {
      option.checked = ev.detail.checked;
    }
  }
  getValues(ev) {
    const { multiple, options } = this;
    if (multiple) {
      // this is a popover with checkboxes (multiple value select)
      // return an array of all the checked values
      return options.filter((o) => o.checked).map((o) => o.value);
    }
    // this is a popover with radio buttons (single value select)
    // return the value that was clicked, otherwise undefined
    const option = this.findOptionFromEvent(ev);
    return option ? option.value : undefined;
  }
  renderOptions(options) {
    const { multiple } = this;
    switch (multiple) {
      case true:
        return this.renderCheckboxOptions(options);
      default:
        return this.renderRadioOptions(options);
    }
  }
  renderCheckboxOptions(options) {
    return options.map((option) => (h("ion-item", { class: getClassMap(option.cssClass) },
      h("ion-checkbox", { slot: "start", value: option.value, disabled: option.disabled, checked: option.checked }),
      h("ion-label", null, option.text))));
  }
  renderRadioOptions(options) {
    const checked = options.filter((o) => o.checked).map((o) => o.value)[0];
    return (h("ion-radio-group", { value: checked }, options.map((option) => (h("ion-item", { class: getClassMap(option.cssClass) },
      h("ion-label", null, option.text),
      h("ion-radio", { value: option.value, disabled: option.disabled, onClick: (ev) => this.rbClick(ev) }))))));
  }
  render() {
    const { header, message, options, subHeader } = this;
    const hasSubHeaderOrMessage = subHeader !== undefined || message !== undefined;
    return (h(Host, { class: getIonMode(this) },
      h("ion-list", null,
        header !== undefined && h("ion-list-header", null, header),
        hasSubHeaderOrMessage && (h("ion-item", null,
          h("ion-label", { class: "ion-text-wrap" },
            subHeader !== undefined && h("h3", null, subHeader),
            message !== undefined && h("p", null, message)))),
        this.renderOptions(options))));
  }
  static get is() { return "ion-select-popover"; }
  static get encapsulation() { return "scoped"; }
  static get originalStyleUrls() { return {
    "ios": ["select-popover.ios.scss"],
    "md": ["select-popover.md.scss"]
  }; }
  static get styleUrls() { return {
    "ios": ["select-popover.ios.css"],
    "md": ["select-popover.md.css"]
  }; }
  static get properties() { return {
    "header": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string | undefined",
        "references": {}
      },
      "required": false,
      "optional": true,
      "docs": {
        "tags": [],
        "text": "The header text of the popover"
      },
      "attribute": "header",
      "reflect": false
    },
    "subHeader": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string | undefined",
        "references": {}
      },
      "required": false,
      "optional": true,
      "docs": {
        "tags": [],
        "text": "The subheader text of the popover"
      },
      "attribute": "sub-header",
      "reflect": false
    },
    "message": {
      "type": "string",
      "mutable": false,
      "complexType": {
        "original": "string",
        "resolved": "string | undefined",
        "references": {}
      },
      "required": false,
      "optional": true,
      "docs": {
        "tags": [],
        "text": "The text content of the popover body"
      },
      "attribute": "message",
      "reflect": false
    },
    "multiple": {
      "type": "boolean",
      "mutable": false,
      "complexType": {
        "original": "boolean",
        "resolved": "boolean | undefined",
        "references": {}
      },
      "required": false,
      "optional": true,
      "docs": {
        "tags": [],
        "text": "If true, the select accepts multiple values"
      },
      "attribute": "multiple",
      "reflect": false
    },
    "options": {
      "type": "unknown",
      "mutable": false,
      "complexType": {
        "original": "SelectPopoverOption[]",
        "resolved": "SelectPopoverOption[]",
        "references": {
          "SelectPopoverOption": {
            "location": "import",
            "path": "../../interface"
          }
        }
      },
      "required": false,
      "optional": false,
      "docs": {
        "tags": [],
        "text": "An array of options for the popover"
      },
      "defaultValue": "[]"
    }
  }; }
  static get listeners() { return [{
      "name": "ionChange",
      "method": "onSelect",
      "target": undefined,
      "capture": false,
      "passive": false
    }]; }
}
