import { __awaiter } from "tslib";
import { bindLifecycleEvents } from '../../providers/angular-delegate';
import { computeStackId, destroyView, getUrl, insertView, isTabSwitch, toSegments, } from './stack-utils';
export class StackController {
    constructor(tabsPrefix, containerEl, router, navCtrl, zone, location) {
        this.containerEl = containerEl;
        this.router = router;
        this.navCtrl = navCtrl;
        this.zone = zone;
        this.location = location;
        this.views = [];
        this.skipTransition = false;
        this.nextId = 0;
        this.tabsPrefix = tabsPrefix !== undefined ? toSegments(tabsPrefix) : undefined;
    }
    createView(ref, activatedRoute) {
        var _a;
        const url = getUrl(this.router, activatedRoute);
        const element = (_a = ref === null || ref === void 0 ? void 0 : ref.location) === null || _a === void 0 ? void 0 : _a.nativeElement;
        const unlistenEvents = bindLifecycleEvents(this.zone, ref.instance, element);
        return {
            id: this.nextId++,
            stackId: computeStackId(this.tabsPrefix, url),
            unlistenEvents,
            element,
            ref,
            url,
        };
    }
    getExistingView(activatedRoute) {
        const activatedUrlKey = getUrl(this.router, activatedRoute);
        const view = this.views.find((vw) => vw.url === activatedUrlKey);
        if (view) {
            view.ref.changeDetectorRef.reattach();
        }
        return view;
    }
    setActive(enteringView) {
        var _a, _b;
        const consumeResult = this.navCtrl.consumeTransition();
        let { direction, animation, animationBuilder } = consumeResult;
        const leavingView = this.activeView;
        const tabSwitch = isTabSwitch(enteringView, leavingView);
        if (tabSwitch) {
            direction = 'back';
            animation = undefined;
        }
        const viewsSnapshot = this.views.slice();
        let currentNavigation;
        const router = this.router;
        // Angular >= 7.2.0
        if (router.getCurrentNavigation) {
            currentNavigation = router.getCurrentNavigation();
            // Angular < 7.2.0
        }
        else if ((_a = router.navigations) === null || _a === void 0 ? void 0 : _a.value) {
            currentNavigation = router.navigations.value;
        }
        /**
         * If the navigation action
         * sets `replaceUrl: true`
         * then we need to make sure
         * we remove the last item
         * from our views stack
         */
        if ((_b = currentNavigation === null || currentNavigation === void 0 ? void 0 : currentNavigation.extras) === null || _b === void 0 ? void 0 : _b.replaceUrl) {
            if (this.views.length > 0) {
                this.views.splice(-1, 1);
            }
        }
        const reused = this.views.includes(enteringView);
        const views = this.insertView(enteringView, direction);
        // Trigger change detection before transition starts
        // This will call ngOnInit() the first time too, just after the view
        // was attached to the dom, but BEFORE the transition starts
        if (!reused) {
            enteringView.ref.changeDetectorRef.detectChanges();
        }
        /**
         * If we are going back from a page that
         * was presented using a custom animation
         * we should default to using that
         * unless the developer explicitly
         * provided another animation.
         */
        const customAnimation = enteringView.animationBuilder;
        if (animationBuilder === undefined && direction === 'back' && !tabSwitch && customAnimation !== undefined) {
            animationBuilder = customAnimation;
        }
        /**
         * Save any custom animation so that navigating
         * back will use this custom animation by default.
         */
        if (leavingView) {
            leavingView.animationBuilder = animationBuilder;
        }
        // Wait until previous transitions finish
        return this.zone.runOutsideAngular(() => {
            return this.wait(() => {
                // disconnect leaving page from change detection to
                // reduce jank during the page transition
                if (leavingView) {
                    leavingView.ref.changeDetectorRef.detach();
                }
                // In case the enteringView is the same as the leavingPage we need to reattach()
                enteringView.ref.changeDetectorRef.reattach();
                return this.transition(enteringView, leavingView, animation, this.canGoBack(1), false, animationBuilder)
                    .then(() => cleanupAsync(enteringView, views, viewsSnapshot, this.location, this.zone))
                    .then(() => ({
                    enteringView,
                    direction,
                    animation,
                    tabSwitch,
                }));
            });
        });
    }
    canGoBack(deep, stackId = this.getActiveStackId()) {
        return this.getStack(stackId).length > deep;
    }
    pop(deep, stackId = this.getActiveStackId()) {
        return this.zone.run(() => {
            var _a, _b;
            const views = this.getStack(stackId);
            if (views.length <= deep) {
                return Promise.resolve(false);
            }
            const view = views[views.length - deep - 1];
            let url = view.url;
            const viewSavedData = view.savedData;
            if (viewSavedData) {
                const primaryOutlet = viewSavedData.get('primary');
                if ((_b = (_a = primaryOutlet === null || primaryOutlet === void 0 ? void 0 : primaryOutlet.route) === null || _a === void 0 ? void 0 : _a._routerState) === null || _b === void 0 ? void 0 : _b.snapshot.url) {
                    url = primaryOutlet.route._routerState.snapshot.url;
                }
            }
            const { animationBuilder } = this.navCtrl.consumeTransition();
            return this.navCtrl.navigateBack(url, Object.assign(Object.assign({}, view.savedExtras), { animation: animationBuilder })).then(() => true);
        });
    }
    startBackTransition() {
        const leavingView = this.activeView;
        if (leavingView) {
            const views = this.getStack(leavingView.stackId);
            const enteringView = views[views.length - 2];
            const customAnimation = enteringView.animationBuilder;
            return this.wait(() => {
                return this.transition(enteringView, // entering view
                leavingView, // leaving view
                'back', this.canGoBack(2), true, customAnimation);
            });
        }
        return Promise.resolve();
    }
    endBackTransition(shouldComplete) {
        if (shouldComplete) {
            this.skipTransition = true;
            this.pop(1);
        }
        else if (this.activeView) {
            cleanup(this.activeView, this.views, this.views, this.location, this.zone);
        }
    }
    getLastUrl(stackId) {
        const views = this.getStack(stackId);
        return views.length > 0 ? views[views.length - 1] : undefined;
    }
    /**
     * @internal
     */
    getRootUrl(stackId) {
        const views = this.getStack(stackId);
        return views.length > 0 ? views[0] : undefined;
    }
    getActiveStackId() {
        return this.activeView ? this.activeView.stackId : undefined;
    }
    hasRunningTask() {
        return this.runningTask !== undefined;
    }
    destroy() {
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        this.containerEl = undefined;
        this.views.forEach(destroyView);
        this.activeView = undefined;
        this.views = [];
    }
    getStack(stackId) {
        return this.views.filter((v) => v.stackId === stackId);
    }
    insertView(enteringView, direction) {
        this.activeView = enteringView;
        this.views = insertView(this.views, enteringView, direction);
        return this.views.slice();
    }
    transition(enteringView, leavingView, direction, showGoBack, progressAnimation, animationBuilder) {
        if (this.skipTransition) {
            this.skipTransition = false;
            return Promise.resolve(false);
        }
        if (leavingView === enteringView) {
            return Promise.resolve(false);
        }
        const enteringEl = enteringView ? enteringView.element : undefined;
        const leavingEl = leavingView ? leavingView.element : undefined;
        const containerEl = this.containerEl;
        if (enteringEl && enteringEl !== leavingEl) {
            enteringEl.classList.add('ion-page');
            enteringEl.classList.add('ion-page-invisible');
            if (enteringEl.parentElement !== containerEl) {
                containerEl.appendChild(enteringEl);
            }
            if (containerEl.commit) {
                return containerEl.commit(enteringEl, leavingEl, {
                    deepWait: true,
                    duration: direction === undefined ? 0 : undefined,
                    direction,
                    showGoBack,
                    progressAnimation,
                    animationBuilder,
                });
            }
        }
        return Promise.resolve(false);
    }
    wait(task) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.runningTask !== undefined) {
                yield this.runningTask;
                this.runningTask = undefined;
            }
            const promise = (this.runningTask = task());
            promise.finally(() => (this.runningTask = undefined));
            return promise;
        });
    }
}
const cleanupAsync = (activeRoute, views, viewsSnapshot, location, zone) => {
    if (typeof requestAnimationFrame === 'function') {
        return new Promise((resolve) => {
            requestAnimationFrame(() => {
                cleanup(activeRoute, views, viewsSnapshot, location, zone);
                resolve();
            });
        });
    }
    return Promise.resolve();
};
const cleanup = (activeRoute, views, viewsSnapshot, location, zone) => {
    /**
     * Re-enter the Angular zone when destroying page components. This will allow
     * lifecycle events (`ngOnDestroy`) to be run inside the Angular zone.
     */
    zone.run(() => viewsSnapshot.filter((view) => !views.includes(view)).forEach(destroyView));
    views.forEach((view) => {
        /**
         * In the event that a user navigated multiple
         * times in rapid succession, we want to make sure
         * we don't pre-emptively detach a view while
         * it is in mid-transition.
         *
         * In this instance we also do not care about query
         * params or fragments as it will be the same view regardless
         */
        const locationWithoutParams = location.path().split('?')[0];
        const locationWithoutFragment = locationWithoutParams.split('#')[0];
        if (view !== activeRoute && view.url !== locationWithoutFragment) {
            const element = view.element;
            element.setAttribute('aria-hidden', 'true');
            element.classList.add('ion-page-hidden');
            view.ref.changeDetectorRef.detach();
        }
    });
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhY2stY29udHJvbGxlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy9kaXJlY3RpdmVzL25hdmlnYXRpb24vc3RhY2stY29udHJvbGxlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBS0EsT0FBTyxFQUFFLG1CQUFtQixFQUFFLE1BQU0sa0NBQWtDLENBQUM7QUFHdkUsT0FBTyxFQUdMLGNBQWMsRUFDZCxXQUFXLEVBQ1gsTUFBTSxFQUNOLFVBQVUsRUFDVixXQUFXLEVBQ1gsVUFBVSxHQUNYLE1BQU0sZUFBZSxDQUFDO0FBRXZCLE1BQU0sT0FBTyxlQUFlO0lBUTFCLFlBQ0UsVUFBOEIsRUFDdEIsV0FBdUMsRUFDdkMsTUFBYyxFQUNkLE9BQXNCLEVBQ3RCLElBQVksRUFDWixRQUFrQjtRQUpsQixnQkFBVyxHQUFYLFdBQVcsQ0FBNEI7UUFDdkMsV0FBTSxHQUFOLE1BQU0sQ0FBUTtRQUNkLFlBQU8sR0FBUCxPQUFPLENBQWU7UUFDdEIsU0FBSSxHQUFKLElBQUksQ0FBUTtRQUNaLGFBQVEsR0FBUixRQUFRLENBQVU7UUFicEIsVUFBSyxHQUFnQixFQUFFLENBQUM7UUFFeEIsbUJBQWMsR0FBRyxLQUFLLENBQUM7UUFHdkIsV0FBTSxHQUFHLENBQUMsQ0FBQztRQVVqQixJQUFJLENBQUMsVUFBVSxHQUFHLFVBQVUsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO0lBQ2xGLENBQUM7SUFFRCxVQUFVLENBQUMsR0FBc0IsRUFBRSxjQUE4Qjs7UUFDL0QsTUFBTSxHQUFHLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDaEQsTUFBTSxPQUFPLEdBQUcsTUFBQSxHQUFHLGFBQUgsR0FBRyx1QkFBSCxHQUFHLENBQUUsUUFBUSwwQ0FBRSxhQUE0QixDQUFDO1FBQzVELE1BQU0sY0FBYyxHQUFHLG1CQUFtQixDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUM3RSxPQUFPO1lBQ0wsRUFBRSxFQUFFLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDakIsT0FBTyxFQUFFLGNBQWMsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEdBQUcsQ0FBQztZQUM3QyxjQUFjO1lBQ2QsT0FBTztZQUNQLEdBQUc7WUFDSCxHQUFHO1NBQ0osQ0FBQztJQUNKLENBQUM7SUFFRCxlQUFlLENBQUMsY0FBOEI7UUFDNUMsTUFBTSxlQUFlLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDNUQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxHQUFHLEtBQUssZUFBZSxDQUFDLENBQUM7UUFDakUsSUFBSSxJQUFJLEVBQUU7WUFDUixJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLFFBQVEsRUFBRSxDQUFDO1NBQ3ZDO1FBQ0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsU0FBUyxDQUFDLFlBQXVCOztRQUMvQixNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLENBQUM7UUFDdkQsSUFBSSxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUUsZ0JBQWdCLEVBQUUsR0FBRyxhQUFhLENBQUM7UUFDL0QsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQztRQUNwQyxNQUFNLFNBQVMsR0FBRyxXQUFXLENBQUMsWUFBWSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ3pELElBQUksU0FBUyxFQUFFO1lBQ2IsU0FBUyxHQUFHLE1BQU0sQ0FBQztZQUNuQixTQUFTLEdBQUcsU0FBUyxDQUFDO1NBQ3ZCO1FBRUQsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUV6QyxJQUFJLGlCQUFpQixDQUFDO1FBRXRCLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxNQUFhLENBQUM7UUFFbEMsbUJBQW1CO1FBQ25CLElBQUksTUFBTSxDQUFDLG9CQUFvQixFQUFFO1lBQy9CLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1lBRWxELGtCQUFrQjtTQUNuQjthQUFNLElBQUksTUFBQSxNQUFNLENBQUMsV0FBVywwQ0FBRSxLQUFLLEVBQUU7WUFDcEMsaUJBQWlCLEdBQUcsTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUM7U0FDOUM7UUFFRDs7Ozs7O1dBTUc7UUFDSCxJQUFJLE1BQUEsaUJBQWlCLGFBQWpCLGlCQUFpQix1QkFBakIsaUJBQWlCLENBQUUsTUFBTSwwQ0FBRSxVQUFVLEVBQUU7WUFDekMsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3pCLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO2FBQzFCO1NBQ0Y7UUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUNqRCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLFlBQVksRUFBRSxTQUFTLENBQUMsQ0FBQztRQUV2RCxvREFBb0Q7UUFDcEQsb0VBQW9FO1FBQ3BFLDREQUE0RDtRQUM1RCxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ1gsWUFBWSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxhQUFhLEVBQUUsQ0FBQztTQUNwRDtRQUVEOzs7Ozs7V0FNRztRQUNILE1BQU0sZUFBZSxHQUFHLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQztRQUN0RCxJQUFJLGdCQUFnQixLQUFLLFNBQVMsSUFBSSxTQUFTLEtBQUssTUFBTSxJQUFJLENBQUMsU0FBUyxJQUFJLGVBQWUsS0FBSyxTQUFTLEVBQUU7WUFDekcsZ0JBQWdCLEdBQUcsZUFBZSxDQUFDO1NBQ3BDO1FBRUQ7OztXQUdHO1FBQ0gsSUFBSSxXQUFXLEVBQUU7WUFDZixXQUFXLENBQUMsZ0JBQWdCLEdBQUcsZ0JBQWdCLENBQUM7U0FDakQ7UUFFRCx5Q0FBeUM7UUFDekMsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRTtZQUN0QyxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFO2dCQUNwQixtREFBbUQ7Z0JBQ25ELHlDQUF5QztnQkFDekMsSUFBSSxXQUFXLEVBQUU7b0JBQ2YsV0FBVyxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEVBQUUsQ0FBQztpQkFDNUM7Z0JBQ0QsZ0ZBQWdGO2dCQUNoRixZQUFZLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLFFBQVEsRUFBRSxDQUFDO2dCQUU5QyxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsWUFBWSxFQUFFLFdBQVcsRUFBRSxTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsZ0JBQWdCLENBQUM7cUJBQ3JHLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxZQUFZLENBQUMsWUFBWSxFQUFFLEtBQUssRUFBRSxhQUFhLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7cUJBQ3RGLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO29CQUNYLFlBQVk7b0JBQ1osU0FBUztvQkFDVCxTQUFTO29CQUNULFNBQVM7aUJBQ1YsQ0FBQyxDQUFDLENBQUM7WUFDUixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELFNBQVMsQ0FBQyxJQUFZLEVBQUUsT0FBTyxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsRUFBRTtRQUN2RCxPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQztJQUM5QyxDQUFDO0lBRUQsR0FBRyxDQUFDLElBQVksRUFBRSxPQUFPLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1FBQ2pELE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFOztZQUN4QixNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ3JDLElBQUksS0FBSyxDQUFDLE1BQU0sSUFBSSxJQUFJLEVBQUU7Z0JBQ3hCLE9BQU8sT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUMvQjtZQUNELE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLENBQUMsQ0FBQztZQUM1QyxJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDO1lBRW5CLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDckMsSUFBSSxhQUFhLEVBQUU7Z0JBQ2pCLE1BQU0sYUFBYSxHQUFHLGFBQWEsQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUM7Z0JBQ25ELElBQUksTUFBQSxNQUFBLGFBQWEsYUFBYixhQUFhLHVCQUFiLGFBQWEsQ0FBRSxLQUFLLDBDQUFFLFlBQVksMENBQUUsUUFBUSxDQUFDLEdBQUcsRUFBRTtvQkFDcEQsR0FBRyxHQUFHLGFBQWEsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUM7aUJBQ3JEO2FBQ0Y7WUFDRCxNQUFNLEVBQUUsZ0JBQWdCLEVBQUUsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLENBQUM7WUFDOUQsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxHQUFHLGtDQUFPLElBQUksQ0FBQyxXQUFXLEtBQUUsU0FBUyxFQUFFLGdCQUFnQixJQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQy9HLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELG1CQUFtQjtRQUNqQixNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDO1FBQ3BDLElBQUksV0FBVyxFQUFFO1lBQ2YsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUM7WUFDakQsTUFBTSxZQUFZLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDN0MsTUFBTSxlQUFlLEdBQUcsWUFBWSxDQUFDLGdCQUFnQixDQUFDO1lBRXRELE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7Z0JBQ3BCLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FDcEIsWUFBWSxFQUFFLGdCQUFnQjtnQkFDOUIsV0FBVyxFQUFFLGVBQWU7Z0JBQzVCLE1BQU0sRUFDTixJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUNqQixJQUFJLEVBQ0osZUFBZSxDQUNoQixDQUFDO1lBQ0osQ0FBQyxDQUFDLENBQUM7U0FDSjtRQUNELE9BQU8sT0FBTyxDQUFDLE9BQU8sRUFBRSxDQUFDO0lBQzNCLENBQUM7SUFFRCxpQkFBaUIsQ0FBQyxjQUF1QjtRQUN2QyxJQUFJLGNBQWMsRUFBRTtZQUNsQixJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQztZQUMzQixJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ2I7YUFBTSxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDMUIsT0FBTyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzVFO0lBQ0gsQ0FBQztJQUVELFVBQVUsQ0FBQyxPQUFnQjtRQUN6QixNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7SUFDaEUsQ0FBQztJQUVEOztPQUVHO0lBQ0gsVUFBVSxDQUFDLE9BQWdCO1FBQ3pCLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDckMsT0FBTyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7SUFDakQsQ0FBQztJQUVELGdCQUFnQjtRQUNkLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUMvRCxDQUFDO0lBRUQsY0FBYztRQUNaLE9BQU8sSUFBSSxDQUFDLFdBQVcsS0FBSyxTQUFTLENBQUM7SUFDeEMsQ0FBQztJQUVELE9BQU87UUFDTCxvRUFBb0U7UUFDcEUsSUFBSSxDQUFDLFdBQVcsR0FBRyxTQUFVLENBQUM7UUFDOUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDaEMsSUFBSSxDQUFDLFVBQVUsR0FBRyxTQUFTLENBQUM7UUFDNUIsSUFBSSxDQUFDLEtBQUssR0FBRyxFQUFFLENBQUM7SUFDbEIsQ0FBQztJQUVPLFFBQVEsQ0FBQyxPQUEyQjtRQUMxQyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxLQUFLLE9BQU8sQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFTyxVQUFVLENBQUMsWUFBdUIsRUFBRSxTQUEwQjtRQUNwRSxJQUFJLENBQUMsVUFBVSxHQUFHLFlBQVksQ0FBQztRQUMvQixJQUFJLENBQUMsS0FBSyxHQUFHLFVBQVUsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFlBQVksRUFBRSxTQUFTLENBQUMsQ0FBQztRQUM3RCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDNUIsQ0FBQztJQUVPLFVBQVUsQ0FDaEIsWUFBbUMsRUFDbkMsV0FBa0MsRUFDbEMsU0FBeUMsRUFDekMsVUFBbUIsRUFDbkIsaUJBQTBCLEVBQzFCLGdCQUFtQztRQUVuQyxJQUFJLElBQUksQ0FBQyxjQUFjLEVBQUU7WUFDdkIsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUM7WUFDNUIsT0FBTyxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQy9CO1FBQ0QsSUFBSSxXQUFXLEtBQUssWUFBWSxFQUFFO1lBQ2hDLE9BQU8sT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUMvQjtRQUNELE1BQU0sVUFBVSxHQUFHLFlBQVksQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ25FLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ2hFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUM7UUFDckMsSUFBSSxVQUFVLElBQUksVUFBVSxLQUFLLFNBQVMsRUFBRTtZQUMxQyxVQUFVLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUNyQyxVQUFVLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1lBQy9DLElBQUksVUFBVSxDQUFDLGFBQWEsS0FBSyxXQUFXLEVBQUU7Z0JBQzVDLFdBQVcsQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLENBQUM7YUFDckM7WUFFRCxJQUFLLFdBQW1CLENBQUMsTUFBTSxFQUFFO2dCQUMvQixPQUFPLFdBQVcsQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFLFNBQVMsRUFBRTtvQkFDL0MsUUFBUSxFQUFFLElBQUk7b0JBQ2QsUUFBUSxFQUFFLFNBQVMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztvQkFDakQsU0FBUztvQkFDVCxVQUFVO29CQUNWLGlCQUFpQjtvQkFDakIsZ0JBQWdCO2lCQUNqQixDQUFDLENBQUM7YUFDSjtTQUNGO1FBQ0QsT0FBTyxPQUFPLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFYSxJQUFJLENBQUksSUFBc0I7O1lBQzFDLElBQUksSUFBSSxDQUFDLFdBQVcsS0FBSyxTQUFTLEVBQUU7Z0JBQ2xDLE1BQU0sSUFBSSxDQUFDLFdBQVcsQ0FBQztnQkFDdkIsSUFBSSxDQUFDLFdBQVcsR0FBRyxTQUFTLENBQUM7YUFDOUI7WUFDRCxNQUFNLE9BQU8sR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUM1QyxPQUFPLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLFdBQVcsR0FBRyxTQUFTLENBQUMsQ0FBQyxDQUFDO1lBQ3RELE9BQU8sT0FBTyxDQUFDO1FBQ2pCLENBQUM7S0FBQTtDQUNGO0FBRUQsTUFBTSxZQUFZLEdBQUcsQ0FDbkIsV0FBc0IsRUFDdEIsS0FBa0IsRUFDbEIsYUFBMEIsRUFDMUIsUUFBa0IsRUFDbEIsSUFBWSxFQUNaLEVBQUU7SUFDRixJQUFJLE9BQVEscUJBQTZCLEtBQUssVUFBVSxFQUFFO1FBQ3hELE9BQU8sSUFBSSxPQUFPLENBQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRTtZQUNuQyxxQkFBcUIsQ0FBQyxHQUFHLEVBQUU7Z0JBQ3pCLE9BQU8sQ0FBQyxXQUFXLEVBQUUsS0FBSyxFQUFFLGFBQWEsRUFBRSxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7Z0JBQzNELE9BQU8sRUFBRSxDQUFDO1lBQ1osQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztLQUNKO0lBQ0QsT0FBTyxPQUFPLENBQUMsT0FBTyxFQUFFLENBQUM7QUFDM0IsQ0FBQyxDQUFDO0FBRUYsTUFBTSxPQUFPLEdBQUcsQ0FDZCxXQUFzQixFQUN0QixLQUFrQixFQUNsQixhQUEwQixFQUMxQixRQUFrQixFQUNsQixJQUFZLEVBQ1osRUFBRTtJQUNGOzs7T0FHRztJQUNILElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7SUFFM0YsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFO1FBQ3JCOzs7Ozs7OztXQVFHO1FBQ0gsTUFBTSxxQkFBcUIsR0FBRyxRQUFRLENBQUMsSUFBSSxFQUFFLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzVELE1BQU0sdUJBQXVCLEdBQUcscUJBQXFCLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXBFLElBQUksSUFBSSxLQUFLLFdBQVcsSUFBSSxJQUFJLENBQUMsR0FBRyxLQUFLLHVCQUF1QixFQUFFO1lBQ2hFLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUM7WUFDN0IsT0FBTyxDQUFDLFlBQVksQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFDNUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsaUJBQWlCLENBQUMsQ0FBQztZQUN6QyxJQUFJLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLE1BQU0sRUFBRSxDQUFDO1NBQ3JDO0lBQ0gsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBMb2NhdGlvbiB9IGZyb20gJ0Bhbmd1bGFyL2NvbW1vbic7XG5pbXBvcnQgeyBDb21wb25lbnRSZWYsIE5nWm9uZSB9IGZyb20gJ0Bhbmd1bGFyL2NvcmUnO1xuaW1wb3J0IHsgQWN0aXZhdGVkUm91dGUsIFJvdXRlciB9IGZyb20gJ0Bhbmd1bGFyL3JvdXRlcic7XG5pbXBvcnQgeyBBbmltYXRpb25CdWlsZGVyLCBSb3V0ZXJEaXJlY3Rpb24gfSBmcm9tICdAaW9uaWMvY29yZSc7XG5cbmltcG9ydCB7IGJpbmRMaWZlY3ljbGVFdmVudHMgfSBmcm9tICcuLi8uLi9wcm92aWRlcnMvYW5ndWxhci1kZWxlZ2F0ZSc7XG5pbXBvcnQgeyBOYXZDb250cm9sbGVyIH0gZnJvbSAnLi4vLi4vcHJvdmlkZXJzL25hdi1jb250cm9sbGVyJztcblxuaW1wb3J0IHtcbiAgUm91dGVWaWV3LFxuICBTdGFja0V2ZW50LFxuICBjb21wdXRlU3RhY2tJZCxcbiAgZGVzdHJveVZpZXcsXG4gIGdldFVybCxcbiAgaW5zZXJ0VmlldyxcbiAgaXNUYWJTd2l0Y2gsXG4gIHRvU2VnbWVudHMsXG59IGZyb20gJy4vc3RhY2stdXRpbHMnO1xuXG5leHBvcnQgY2xhc3MgU3RhY2tDb250cm9sbGVyIHtcbiAgcHJpdmF0ZSB2aWV3czogUm91dGVWaWV3W10gPSBbXTtcbiAgcHJpdmF0ZSBydW5uaW5nVGFzaz86IFByb21pc2U8YW55PjtcbiAgcHJpdmF0ZSBza2lwVHJhbnNpdGlvbiA9IGZhbHNlO1xuICBwcml2YXRlIHRhYnNQcmVmaXg6IHN0cmluZ1tdIHwgdW5kZWZpbmVkO1xuICBwcml2YXRlIGFjdGl2ZVZpZXc6IFJvdXRlVmlldyB8IHVuZGVmaW5lZDtcbiAgcHJpdmF0ZSBuZXh0SWQgPSAwO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHRhYnNQcmVmaXg6IHN0cmluZyB8IHVuZGVmaW5lZCxcbiAgICBwcml2YXRlIGNvbnRhaW5lckVsOiBIVE1MSW9uUm91dGVyT3V0bGV0RWxlbWVudCxcbiAgICBwcml2YXRlIHJvdXRlcjogUm91dGVyLFxuICAgIHByaXZhdGUgbmF2Q3RybDogTmF2Q29udHJvbGxlcixcbiAgICBwcml2YXRlIHpvbmU6IE5nWm9uZSxcbiAgICBwcml2YXRlIGxvY2F0aW9uOiBMb2NhdGlvblxuICApIHtcbiAgICB0aGlzLnRhYnNQcmVmaXggPSB0YWJzUHJlZml4ICE9PSB1bmRlZmluZWQgPyB0b1NlZ21lbnRzKHRhYnNQcmVmaXgpIDogdW5kZWZpbmVkO1xuICB9XG5cbiAgY3JlYXRlVmlldyhyZWY6IENvbXBvbmVudFJlZjxhbnk+LCBhY3RpdmF0ZWRSb3V0ZTogQWN0aXZhdGVkUm91dGUpOiBSb3V0ZVZpZXcge1xuICAgIGNvbnN0IHVybCA9IGdldFVybCh0aGlzLnJvdXRlciwgYWN0aXZhdGVkUm91dGUpO1xuICAgIGNvbnN0IGVsZW1lbnQgPSByZWY/LmxvY2F0aW9uPy5uYXRpdmVFbGVtZW50IGFzIEhUTUxFbGVtZW50O1xuICAgIGNvbnN0IHVubGlzdGVuRXZlbnRzID0gYmluZExpZmVjeWNsZUV2ZW50cyh0aGlzLnpvbmUsIHJlZi5pbnN0YW5jZSwgZWxlbWVudCk7XG4gICAgcmV0dXJuIHtcbiAgICAgIGlkOiB0aGlzLm5leHRJZCsrLFxuICAgICAgc3RhY2tJZDogY29tcHV0ZVN0YWNrSWQodGhpcy50YWJzUHJlZml4LCB1cmwpLFxuICAgICAgdW5saXN0ZW5FdmVudHMsXG4gICAgICBlbGVtZW50LFxuICAgICAgcmVmLFxuICAgICAgdXJsLFxuICAgIH07XG4gIH1cblxuICBnZXRFeGlzdGluZ1ZpZXcoYWN0aXZhdGVkUm91dGU6IEFjdGl2YXRlZFJvdXRlKTogUm91dGVWaWV3IHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBhY3RpdmF0ZWRVcmxLZXkgPSBnZXRVcmwodGhpcy5yb3V0ZXIsIGFjdGl2YXRlZFJvdXRlKTtcbiAgICBjb25zdCB2aWV3ID0gdGhpcy52aWV3cy5maW5kKCh2dykgPT4gdncudXJsID09PSBhY3RpdmF0ZWRVcmxLZXkpO1xuICAgIGlmICh2aWV3KSB7XG4gICAgICB2aWV3LnJlZi5jaGFuZ2VEZXRlY3RvclJlZi5yZWF0dGFjaCgpO1xuICAgIH1cbiAgICByZXR1cm4gdmlldztcbiAgfVxuXG4gIHNldEFjdGl2ZShlbnRlcmluZ1ZpZXc6IFJvdXRlVmlldyk6IFByb21pc2U8U3RhY2tFdmVudD4ge1xuICAgIGNvbnN0IGNvbnN1bWVSZXN1bHQgPSB0aGlzLm5hdkN0cmwuY29uc3VtZVRyYW5zaXRpb24oKTtcbiAgICBsZXQgeyBkaXJlY3Rpb24sIGFuaW1hdGlvbiwgYW5pbWF0aW9uQnVpbGRlciB9ID0gY29uc3VtZVJlc3VsdDtcbiAgICBjb25zdCBsZWF2aW5nVmlldyA9IHRoaXMuYWN0aXZlVmlldztcbiAgICBjb25zdCB0YWJTd2l0Y2ggPSBpc1RhYlN3aXRjaChlbnRlcmluZ1ZpZXcsIGxlYXZpbmdWaWV3KTtcbiAgICBpZiAodGFiU3dpdGNoKSB7XG4gICAgICBkaXJlY3Rpb24gPSAnYmFjayc7XG4gICAgICBhbmltYXRpb24gPSB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgY29uc3Qgdmlld3NTbmFwc2hvdCA9IHRoaXMudmlld3Muc2xpY2UoKTtcblxuICAgIGxldCBjdXJyZW50TmF2aWdhdGlvbjtcblxuICAgIGNvbnN0IHJvdXRlciA9IHRoaXMucm91dGVyIGFzIGFueTtcblxuICAgIC8vIEFuZ3VsYXIgPj0gNy4yLjBcbiAgICBpZiAocm91dGVyLmdldEN1cnJlbnROYXZpZ2F0aW9uKSB7XG4gICAgICBjdXJyZW50TmF2aWdhdGlvbiA9IHJvdXRlci5nZXRDdXJyZW50TmF2aWdhdGlvbigpO1xuXG4gICAgICAvLyBBbmd1bGFyIDwgNy4yLjBcbiAgICB9IGVsc2UgaWYgKHJvdXRlci5uYXZpZ2F0aW9ucz8udmFsdWUpIHtcbiAgICAgIGN1cnJlbnROYXZpZ2F0aW9uID0gcm91dGVyLm5hdmlnYXRpb25zLnZhbHVlO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIElmIHRoZSBuYXZpZ2F0aW9uIGFjdGlvblxuICAgICAqIHNldHMgYHJlcGxhY2VVcmw6IHRydWVgXG4gICAgICogdGhlbiB3ZSBuZWVkIHRvIG1ha2Ugc3VyZVxuICAgICAqIHdlIHJlbW92ZSB0aGUgbGFzdCBpdGVtXG4gICAgICogZnJvbSBvdXIgdmlld3Mgc3RhY2tcbiAgICAgKi9cbiAgICBpZiAoY3VycmVudE5hdmlnYXRpb24/LmV4dHJhcz8ucmVwbGFjZVVybCkge1xuICAgICAgaWYgKHRoaXMudmlld3MubGVuZ3RoID4gMCkge1xuICAgICAgICB0aGlzLnZpZXdzLnNwbGljZSgtMSwgMSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgcmV1c2VkID0gdGhpcy52aWV3cy5pbmNsdWRlcyhlbnRlcmluZ1ZpZXcpO1xuICAgIGNvbnN0IHZpZXdzID0gdGhpcy5pbnNlcnRWaWV3KGVudGVyaW5nVmlldywgZGlyZWN0aW9uKTtcblxuICAgIC8vIFRyaWdnZXIgY2hhbmdlIGRldGVjdGlvbiBiZWZvcmUgdHJhbnNpdGlvbiBzdGFydHNcbiAgICAvLyBUaGlzIHdpbGwgY2FsbCBuZ09uSW5pdCgpIHRoZSBmaXJzdCB0aW1lIHRvbywganVzdCBhZnRlciB0aGUgdmlld1xuICAgIC8vIHdhcyBhdHRhY2hlZCB0byB0aGUgZG9tLCBidXQgQkVGT1JFIHRoZSB0cmFuc2l0aW9uIHN0YXJ0c1xuICAgIGlmICghcmV1c2VkKSB7XG4gICAgICBlbnRlcmluZ1ZpZXcucmVmLmNoYW5nZURldGVjdG9yUmVmLmRldGVjdENoYW5nZXMoKTtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBJZiB3ZSBhcmUgZ29pbmcgYmFjayBmcm9tIGEgcGFnZSB0aGF0XG4gICAgICogd2FzIHByZXNlbnRlZCB1c2luZyBhIGN1c3RvbSBhbmltYXRpb25cbiAgICAgKiB3ZSBzaG91bGQgZGVmYXVsdCB0byB1c2luZyB0aGF0XG4gICAgICogdW5sZXNzIHRoZSBkZXZlbG9wZXIgZXhwbGljaXRseVxuICAgICAqIHByb3ZpZGVkIGFub3RoZXIgYW5pbWF0aW9uLlxuICAgICAqL1xuICAgIGNvbnN0IGN1c3RvbUFuaW1hdGlvbiA9IGVudGVyaW5nVmlldy5hbmltYXRpb25CdWlsZGVyO1xuICAgIGlmIChhbmltYXRpb25CdWlsZGVyID09PSB1bmRlZmluZWQgJiYgZGlyZWN0aW9uID09PSAnYmFjaycgJiYgIXRhYlN3aXRjaCAmJiBjdXN0b21BbmltYXRpb24gIT09IHVuZGVmaW5lZCkge1xuICAgICAgYW5pbWF0aW9uQnVpbGRlciA9IGN1c3RvbUFuaW1hdGlvbjtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBTYXZlIGFueSBjdXN0b20gYW5pbWF0aW9uIHNvIHRoYXQgbmF2aWdhdGluZ1xuICAgICAqIGJhY2sgd2lsbCB1c2UgdGhpcyBjdXN0b20gYW5pbWF0aW9uIGJ5IGRlZmF1bHQuXG4gICAgICovXG4gICAgaWYgKGxlYXZpbmdWaWV3KSB7XG4gICAgICBsZWF2aW5nVmlldy5hbmltYXRpb25CdWlsZGVyID0gYW5pbWF0aW9uQnVpbGRlcjtcbiAgICB9XG5cbiAgICAvLyBXYWl0IHVudGlsIHByZXZpb3VzIHRyYW5zaXRpb25zIGZpbmlzaFxuICAgIHJldHVybiB0aGlzLnpvbmUucnVuT3V0c2lkZUFuZ3VsYXIoKCkgPT4ge1xuICAgICAgcmV0dXJuIHRoaXMud2FpdCgoKSA9PiB7XG4gICAgICAgIC8vIGRpc2Nvbm5lY3QgbGVhdmluZyBwYWdlIGZyb20gY2hhbmdlIGRldGVjdGlvbiB0b1xuICAgICAgICAvLyByZWR1Y2UgamFuayBkdXJpbmcgdGhlIHBhZ2UgdHJhbnNpdGlvblxuICAgICAgICBpZiAobGVhdmluZ1ZpZXcpIHtcbiAgICAgICAgICBsZWF2aW5nVmlldy5yZWYuY2hhbmdlRGV0ZWN0b3JSZWYuZGV0YWNoKCk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gSW4gY2FzZSB0aGUgZW50ZXJpbmdWaWV3IGlzIHRoZSBzYW1lIGFzIHRoZSBsZWF2aW5nUGFnZSB3ZSBuZWVkIHRvIHJlYXR0YWNoKClcbiAgICAgICAgZW50ZXJpbmdWaWV3LnJlZi5jaGFuZ2VEZXRlY3RvclJlZi5yZWF0dGFjaCgpO1xuXG4gICAgICAgIHJldHVybiB0aGlzLnRyYW5zaXRpb24oZW50ZXJpbmdWaWV3LCBsZWF2aW5nVmlldywgYW5pbWF0aW9uLCB0aGlzLmNhbkdvQmFjaygxKSwgZmFsc2UsIGFuaW1hdGlvbkJ1aWxkZXIpXG4gICAgICAgICAgLnRoZW4oKCkgPT4gY2xlYW51cEFzeW5jKGVudGVyaW5nVmlldywgdmlld3MsIHZpZXdzU25hcHNob3QsIHRoaXMubG9jYXRpb24sIHRoaXMuem9uZSkpXG4gICAgICAgICAgLnRoZW4oKCkgPT4gKHtcbiAgICAgICAgICAgIGVudGVyaW5nVmlldyxcbiAgICAgICAgICAgIGRpcmVjdGlvbixcbiAgICAgICAgICAgIGFuaW1hdGlvbixcbiAgICAgICAgICAgIHRhYlN3aXRjaCxcbiAgICAgICAgICB9KSk7XG4gICAgICB9KTtcbiAgICB9KTtcbiAgfVxuXG4gIGNhbkdvQmFjayhkZWVwOiBudW1iZXIsIHN0YWNrSWQgPSB0aGlzLmdldEFjdGl2ZVN0YWNrSWQoKSk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLmdldFN0YWNrKHN0YWNrSWQpLmxlbmd0aCA+IGRlZXA7XG4gIH1cblxuICBwb3AoZGVlcDogbnVtYmVyLCBzdGFja0lkID0gdGhpcy5nZXRBY3RpdmVTdGFja0lkKCkpOiBQcm9taXNlPGJvb2xlYW4+IHtcbiAgICByZXR1cm4gdGhpcy56b25lLnJ1bigoKSA9PiB7XG4gICAgICBjb25zdCB2aWV3cyA9IHRoaXMuZ2V0U3RhY2soc3RhY2tJZCk7XG4gICAgICBpZiAodmlld3MubGVuZ3RoIDw9IGRlZXApIHtcbiAgICAgICAgcmV0dXJuIFByb21pc2UucmVzb2x2ZShmYWxzZSk7XG4gICAgICB9XG4gICAgICBjb25zdCB2aWV3ID0gdmlld3Nbdmlld3MubGVuZ3RoIC0gZGVlcCAtIDFdO1xuICAgICAgbGV0IHVybCA9IHZpZXcudXJsO1xuXG4gICAgICBjb25zdCB2aWV3U2F2ZWREYXRhID0gdmlldy5zYXZlZERhdGE7XG4gICAgICBpZiAodmlld1NhdmVkRGF0YSkge1xuICAgICAgICBjb25zdCBwcmltYXJ5T3V0bGV0ID0gdmlld1NhdmVkRGF0YS5nZXQoJ3ByaW1hcnknKTtcbiAgICAgICAgaWYgKHByaW1hcnlPdXRsZXQ/LnJvdXRlPy5fcm91dGVyU3RhdGU/LnNuYXBzaG90LnVybCkge1xuICAgICAgICAgIHVybCA9IHByaW1hcnlPdXRsZXQucm91dGUuX3JvdXRlclN0YXRlLnNuYXBzaG90LnVybDtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgY29uc3QgeyBhbmltYXRpb25CdWlsZGVyIH0gPSB0aGlzLm5hdkN0cmwuY29uc3VtZVRyYW5zaXRpb24oKTtcbiAgICAgIHJldHVybiB0aGlzLm5hdkN0cmwubmF2aWdhdGVCYWNrKHVybCwgeyAuLi52aWV3LnNhdmVkRXh0cmFzLCBhbmltYXRpb246IGFuaW1hdGlvbkJ1aWxkZXIgfSkudGhlbigoKSA9PiB0cnVlKTtcbiAgICB9KTtcbiAgfVxuXG4gIHN0YXJ0QmFja1RyYW5zaXRpb24oKTogUHJvbWlzZTxib29sZWFuPiB8IFByb21pc2U8dm9pZD4ge1xuICAgIGNvbnN0IGxlYXZpbmdWaWV3ID0gdGhpcy5hY3RpdmVWaWV3O1xuICAgIGlmIChsZWF2aW5nVmlldykge1xuICAgICAgY29uc3Qgdmlld3MgPSB0aGlzLmdldFN0YWNrKGxlYXZpbmdWaWV3LnN0YWNrSWQpO1xuICAgICAgY29uc3QgZW50ZXJpbmdWaWV3ID0gdmlld3Nbdmlld3MubGVuZ3RoIC0gMl07XG4gICAgICBjb25zdCBjdXN0b21BbmltYXRpb24gPSBlbnRlcmluZ1ZpZXcuYW5pbWF0aW9uQnVpbGRlcjtcblxuICAgICAgcmV0dXJuIHRoaXMud2FpdCgoKSA9PiB7XG4gICAgICAgIHJldHVybiB0aGlzLnRyYW5zaXRpb24oXG4gICAgICAgICAgZW50ZXJpbmdWaWV3LCAvLyBlbnRlcmluZyB2aWV3XG4gICAgICAgICAgbGVhdmluZ1ZpZXcsIC8vIGxlYXZpbmcgdmlld1xuICAgICAgICAgICdiYWNrJyxcbiAgICAgICAgICB0aGlzLmNhbkdvQmFjaygyKSxcbiAgICAgICAgICB0cnVlLFxuICAgICAgICAgIGN1c3RvbUFuaW1hdGlvblxuICAgICAgICApO1xuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoKTtcbiAgfVxuXG4gIGVuZEJhY2tUcmFuc2l0aW9uKHNob3VsZENvbXBsZXRlOiBib29sZWFuKTogdm9pZCB7XG4gICAgaWYgKHNob3VsZENvbXBsZXRlKSB7XG4gICAgICB0aGlzLnNraXBUcmFuc2l0aW9uID0gdHJ1ZTtcbiAgICAgIHRoaXMucG9wKDEpO1xuICAgIH0gZWxzZSBpZiAodGhpcy5hY3RpdmVWaWV3KSB7XG4gICAgICBjbGVhbnVwKHRoaXMuYWN0aXZlVmlldywgdGhpcy52aWV3cywgdGhpcy52aWV3cywgdGhpcy5sb2NhdGlvbiwgdGhpcy56b25lKTtcbiAgICB9XG4gIH1cblxuICBnZXRMYXN0VXJsKHN0YWNrSWQ/OiBzdHJpbmcpOiBSb3V0ZVZpZXcgfCB1bmRlZmluZWQge1xuICAgIGNvbnN0IHZpZXdzID0gdGhpcy5nZXRTdGFjayhzdGFja0lkKTtcbiAgICByZXR1cm4gdmlld3MubGVuZ3RoID4gMCA/IHZpZXdzW3ZpZXdzLmxlbmd0aCAtIDFdIDogdW5kZWZpbmVkO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgZ2V0Um9vdFVybChzdGFja0lkPzogc3RyaW5nKTogUm91dGVWaWV3IHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCB2aWV3cyA9IHRoaXMuZ2V0U3RhY2soc3RhY2tJZCk7XG4gICAgcmV0dXJuIHZpZXdzLmxlbmd0aCA+IDAgPyB2aWV3c1swXSA6IHVuZGVmaW5lZDtcbiAgfVxuXG4gIGdldEFjdGl2ZVN0YWNrSWQoKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5hY3RpdmVWaWV3ID8gdGhpcy5hY3RpdmVWaWV3LnN0YWNrSWQgOiB1bmRlZmluZWQ7XG4gIH1cblxuICBoYXNSdW5uaW5nVGFzaygpOiBib29sZWFuIHtcbiAgICByZXR1cm4gdGhpcy5ydW5uaW5nVGFzayAhPT0gdW5kZWZpbmVkO1xuICB9XG5cbiAgZGVzdHJveSgpOiB2b2lkIHtcbiAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLW5vbi1udWxsLWFzc2VydGlvblxuICAgIHRoaXMuY29udGFpbmVyRWwgPSB1bmRlZmluZWQhO1xuICAgIHRoaXMudmlld3MuZm9yRWFjaChkZXN0cm95Vmlldyk7XG4gICAgdGhpcy5hY3RpdmVWaWV3ID0gdW5kZWZpbmVkO1xuICAgIHRoaXMudmlld3MgPSBbXTtcbiAgfVxuXG4gIHByaXZhdGUgZ2V0U3RhY2soc3RhY2tJZDogc3RyaW5nIHwgdW5kZWZpbmVkKSB7XG4gICAgcmV0dXJuIHRoaXMudmlld3MuZmlsdGVyKCh2KSA9PiB2LnN0YWNrSWQgPT09IHN0YWNrSWQpO1xuICB9XG5cbiAgcHJpdmF0ZSBpbnNlcnRWaWV3KGVudGVyaW5nVmlldzogUm91dGVWaWV3LCBkaXJlY3Rpb246IFJvdXRlckRpcmVjdGlvbikge1xuICAgIHRoaXMuYWN0aXZlVmlldyA9IGVudGVyaW5nVmlldztcbiAgICB0aGlzLnZpZXdzID0gaW5zZXJ0Vmlldyh0aGlzLnZpZXdzLCBlbnRlcmluZ1ZpZXcsIGRpcmVjdGlvbik7XG4gICAgcmV0dXJuIHRoaXMudmlld3Muc2xpY2UoKTtcbiAgfVxuXG4gIHByaXZhdGUgdHJhbnNpdGlvbihcbiAgICBlbnRlcmluZ1ZpZXc6IFJvdXRlVmlldyB8IHVuZGVmaW5lZCxcbiAgICBsZWF2aW5nVmlldzogUm91dGVWaWV3IHwgdW5kZWZpbmVkLFxuICAgIGRpcmVjdGlvbjogJ2ZvcndhcmQnIHwgJ2JhY2snIHwgdW5kZWZpbmVkLFxuICAgIHNob3dHb0JhY2s6IGJvb2xlYW4sXG4gICAgcHJvZ3Jlc3NBbmltYXRpb246IGJvb2xlYW4sXG4gICAgYW5pbWF0aW9uQnVpbGRlcj86IEFuaW1hdGlvbkJ1aWxkZXJcbiAgKSB7XG4gICAgaWYgKHRoaXMuc2tpcFRyYW5zaXRpb24pIHtcbiAgICAgIHRoaXMuc2tpcFRyYW5zaXRpb24gPSBmYWxzZTtcbiAgICAgIHJldHVybiBQcm9taXNlLnJlc29sdmUoZmFsc2UpO1xuICAgIH1cbiAgICBpZiAobGVhdmluZ1ZpZXcgPT09IGVudGVyaW5nVmlldykge1xuICAgICAgcmV0dXJuIFByb21pc2UucmVzb2x2ZShmYWxzZSk7XG4gICAgfVxuICAgIGNvbnN0IGVudGVyaW5nRWwgPSBlbnRlcmluZ1ZpZXcgPyBlbnRlcmluZ1ZpZXcuZWxlbWVudCA6IHVuZGVmaW5lZDtcbiAgICBjb25zdCBsZWF2aW5nRWwgPSBsZWF2aW5nVmlldyA/IGxlYXZpbmdWaWV3LmVsZW1lbnQgOiB1bmRlZmluZWQ7XG4gICAgY29uc3QgY29udGFpbmVyRWwgPSB0aGlzLmNvbnRhaW5lckVsO1xuICAgIGlmIChlbnRlcmluZ0VsICYmIGVudGVyaW5nRWwgIT09IGxlYXZpbmdFbCkge1xuICAgICAgZW50ZXJpbmdFbC5jbGFzc0xpc3QuYWRkKCdpb24tcGFnZScpO1xuICAgICAgZW50ZXJpbmdFbC5jbGFzc0xpc3QuYWRkKCdpb24tcGFnZS1pbnZpc2libGUnKTtcbiAgICAgIGlmIChlbnRlcmluZ0VsLnBhcmVudEVsZW1lbnQgIT09IGNvbnRhaW5lckVsKSB7XG4gICAgICAgIGNvbnRhaW5lckVsLmFwcGVuZENoaWxkKGVudGVyaW5nRWwpO1xuICAgICAgfVxuXG4gICAgICBpZiAoKGNvbnRhaW5lckVsIGFzIGFueSkuY29tbWl0KSB7XG4gICAgICAgIHJldHVybiBjb250YWluZXJFbC5jb21taXQoZW50ZXJpbmdFbCwgbGVhdmluZ0VsLCB7XG4gICAgICAgICAgZGVlcFdhaXQ6IHRydWUsXG4gICAgICAgICAgZHVyYXRpb246IGRpcmVjdGlvbiA9PT0gdW5kZWZpbmVkID8gMCA6IHVuZGVmaW5lZCxcbiAgICAgICAgICBkaXJlY3Rpb24sXG4gICAgICAgICAgc2hvd0dvQmFjayxcbiAgICAgICAgICBwcm9ncmVzc0FuaW1hdGlvbixcbiAgICAgICAgICBhbmltYXRpb25CdWlsZGVyLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIFByb21pc2UucmVzb2x2ZShmYWxzZSk7XG4gIH1cblxuICBwcml2YXRlIGFzeW5jIHdhaXQ8VD4odGFzazogKCkgPT4gUHJvbWlzZTxUPik6IFByb21pc2U8VD4ge1xuICAgIGlmICh0aGlzLnJ1bm5pbmdUYXNrICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGF3YWl0IHRoaXMucnVubmluZ1Rhc2s7XG4gICAgICB0aGlzLnJ1bm5pbmdUYXNrID0gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBjb25zdCBwcm9taXNlID0gKHRoaXMucnVubmluZ1Rhc2sgPSB0YXNrKCkpO1xuICAgIHByb21pc2UuZmluYWxseSgoKSA9PiAodGhpcy5ydW5uaW5nVGFzayA9IHVuZGVmaW5lZCkpO1xuICAgIHJldHVybiBwcm9taXNlO1xuICB9XG59XG5cbmNvbnN0IGNsZWFudXBBc3luYyA9IChcbiAgYWN0aXZlUm91dGU6IFJvdXRlVmlldyxcbiAgdmlld3M6IFJvdXRlVmlld1tdLFxuICB2aWV3c1NuYXBzaG90OiBSb3V0ZVZpZXdbXSxcbiAgbG9jYXRpb246IExvY2F0aW9uLFxuICB6b25lOiBOZ1pvbmVcbikgPT4ge1xuICBpZiAodHlwZW9mIChyZXF1ZXN0QW5pbWF0aW9uRnJhbWUgYXMgYW55KSA9PT0gJ2Z1bmN0aW9uJykge1xuICAgIHJldHVybiBuZXcgUHJvbWlzZTx2b2lkPigocmVzb2x2ZSkgPT4ge1xuICAgICAgcmVxdWVzdEFuaW1hdGlvbkZyYW1lKCgpID0+IHtcbiAgICAgICAgY2xlYW51cChhY3RpdmVSb3V0ZSwgdmlld3MsIHZpZXdzU25hcHNob3QsIGxvY2F0aW9uLCB6b25lKTtcbiAgICAgICAgcmVzb2x2ZSgpO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH1cbiAgcmV0dXJuIFByb21pc2UucmVzb2x2ZSgpO1xufTtcblxuY29uc3QgY2xlYW51cCA9IChcbiAgYWN0aXZlUm91dGU6IFJvdXRlVmlldyxcbiAgdmlld3M6IFJvdXRlVmlld1tdLFxuICB2aWV3c1NuYXBzaG90OiBSb3V0ZVZpZXdbXSxcbiAgbG9jYXRpb246IExvY2F0aW9uLFxuICB6b25lOiBOZ1pvbmVcbikgPT4ge1xuICAvKipcbiAgICogUmUtZW50ZXIgdGhlIEFuZ3VsYXIgem9uZSB3aGVuIGRlc3Ryb3lpbmcgcGFnZSBjb21wb25lbnRzLiBUaGlzIHdpbGwgYWxsb3dcbiAgICogbGlmZWN5Y2xlIGV2ZW50cyAoYG5nT25EZXN0cm95YCkgdG8gYmUgcnVuIGluc2lkZSB0aGUgQW5ndWxhciB6b25lLlxuICAgKi9cbiAgem9uZS5ydW4oKCkgPT4gdmlld3NTbmFwc2hvdC5maWx0ZXIoKHZpZXcpID0+ICF2aWV3cy5pbmNsdWRlcyh2aWV3KSkuZm9yRWFjaChkZXN0cm95VmlldykpO1xuXG4gIHZpZXdzLmZvckVhY2goKHZpZXcpID0+IHtcbiAgICAvKipcbiAgICAgKiBJbiB0aGUgZXZlbnQgdGhhdCBhIHVzZXIgbmF2aWdhdGVkIG11bHRpcGxlXG4gICAgICogdGltZXMgaW4gcmFwaWQgc3VjY2Vzc2lvbiwgd2Ugd2FudCB0byBtYWtlIHN1cmVcbiAgICAgKiB3ZSBkb24ndCBwcmUtZW1wdGl2ZWx5IGRldGFjaCBhIHZpZXcgd2hpbGVcbiAgICAgKiBpdCBpcyBpbiBtaWQtdHJhbnNpdGlvbi5cbiAgICAgKlxuICAgICAqIEluIHRoaXMgaW5zdGFuY2Ugd2UgYWxzbyBkbyBub3QgY2FyZSBhYm91dCBxdWVyeVxuICAgICAqIHBhcmFtcyBvciBmcmFnbWVudHMgYXMgaXQgd2lsbCBiZSB0aGUgc2FtZSB2aWV3IHJlZ2FyZGxlc3NcbiAgICAgKi9cbiAgICBjb25zdCBsb2NhdGlvbldpdGhvdXRQYXJhbXMgPSBsb2NhdGlvbi5wYXRoKCkuc3BsaXQoJz8nKVswXTtcbiAgICBjb25zdCBsb2NhdGlvbldpdGhvdXRGcmFnbWVudCA9IGxvY2F0aW9uV2l0aG91dFBhcmFtcy5zcGxpdCgnIycpWzBdO1xuXG4gICAgaWYgKHZpZXcgIT09IGFjdGl2ZVJvdXRlICYmIHZpZXcudXJsICE9PSBsb2NhdGlvbldpdGhvdXRGcmFnbWVudCkge1xuICAgICAgY29uc3QgZWxlbWVudCA9IHZpZXcuZWxlbWVudDtcbiAgICAgIGVsZW1lbnQuc2V0QXR0cmlidXRlKCdhcmlhLWhpZGRlbicsICd0cnVlJyk7XG4gICAgICBlbGVtZW50LmNsYXNzTGlzdC5hZGQoJ2lvbi1wYWdlLWhpZGRlbicpO1xuICAgICAgdmlldy5yZWYuY2hhbmdlRGV0ZWN0b3JSZWYuZGV0YWNoKCk7XG4gICAgfVxuICB9KTtcbn07XG4iXX0=