/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { EMPTY, from, of } from 'rxjs';
import { concatMap, map, mergeMap, takeLast, tap } from 'rxjs/operators';
import { inheritedParamsDataResolve } from '../router_state';
import { wrapIntoObservable } from '../utils/collection';
import { getToken } from '../utils/preactivation';
export function resolveData(paramsInheritanceStrategy, moduleInjector) {
    return mergeMap(t => {
        const { targetSnapshot, guards: { canActivateChecks } } = t;
        if (!canActivateChecks.length) {
            return of(t);
        }
        let canActivateChecksResolved = 0;
        return from(canActivateChecks)
            .pipe(concatMap(check => runResolve(check.route, targetSnapshot, paramsInheritanceStrategy, moduleInjector)), tap(() => canActivateChecksResolved++), takeLast(1), mergeMap(_ => canActivateChecksResolved === canActivateChecks.length ? of(t) : EMPTY));
    });
}
function runResolve(futureARS, futureRSS, paramsInheritanceStrategy, moduleInjector) {
    const resolve = futureARS._resolve;
    return resolveNode(resolve, futureARS, futureRSS, moduleInjector)
        .pipe(map((resolvedData) => {
        futureARS._resolvedData = resolvedData;
        futureARS.data = {
            ...futureARS.data,
            ...inheritedParamsDataResolve(futureARS, paramsInheritanceStrategy).resolve
        };
        return null;
    }));
}
function resolveNode(resolve, futureARS, futureRSS, moduleInjector) {
    const keys = getDataKeys(resolve);
    if (keys.length === 0) {
        return of({});
    }
    const data = {};
    return from(keys).pipe(mergeMap(key => getResolver(resolve[key], futureARS, futureRSS, moduleInjector)
        .pipe(tap((value) => {
        data[key] = value;
    }))), takeLast(1), mergeMap(() => {
        // Ensure all resolvers returned values, otherwise don't emit any "next" and just complete
        // the chain which will cancel navigation
        if (getDataKeys(data).length === keys.length) {
            return of(data);
        }
        return EMPTY;
    }));
}
function getDataKeys(obj) {
    return [...Object.keys(obj), ...Object.getOwnPropertySymbols(obj)];
}
function getResolver(injectionToken, futureARS, futureRSS, moduleInjector) {
    const resolver = getToken(injectionToken, futureARS, moduleInjector);
    return resolver.resolve ? wrapIntoObservable(resolver.resolve(futureARS, futureRSS)) :
        wrapIntoObservable(resolver(futureARS, futureRSS));
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb2x2ZV9kYXRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvcm91dGVyL3NyYy9vcGVyYXRvcnMvcmVzb2x2ZV9kYXRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRztBQUdILE9BQU8sRUFBQyxLQUFLLEVBQUUsSUFBSSxFQUF3QyxFQUFFLEVBQUMsTUFBTSxNQUFNLENBQUM7QUFDM0UsT0FBTyxFQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxHQUFHLEVBQUMsTUFBTSxnQkFBZ0IsQ0FBQztBQUl2RSxPQUFPLEVBQXlCLDBCQUEwQixFQUFzQixNQUFNLGlCQUFpQixDQUFDO0FBQ3hHLE9BQU8sRUFBQyxrQkFBa0IsRUFBQyxNQUFNLHFCQUFxQixDQUFDO0FBQ3ZELE9BQU8sRUFBQyxRQUFRLEVBQUMsTUFBTSx3QkFBd0IsQ0FBQztBQUVoRCxNQUFNLFVBQVUsV0FBVyxDQUN2Qix5QkFBK0MsRUFDL0MsY0FBd0I7SUFDMUIsT0FBTyxRQUFRLENBQUMsQ0FBQyxDQUFDLEVBQUU7UUFDbEIsTUFBTSxFQUFDLGNBQWMsRUFBRSxNQUFNLEVBQUUsRUFBQyxpQkFBaUIsRUFBQyxFQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRXhELElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEVBQUU7WUFDN0IsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDZDtRQUNELElBQUkseUJBQXlCLEdBQUcsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDO2FBQ3pCLElBQUksQ0FDRCxTQUFTLENBQ0wsS0FBSyxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQ2YsS0FBSyxDQUFDLEtBQUssRUFBRSxjQUFlLEVBQUUseUJBQXlCLEVBQUUsY0FBYyxDQUFDLENBQUMsRUFDakYsR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDLHlCQUF5QixFQUFFLENBQUMsRUFDdEMsUUFBUSxDQUFDLENBQUMsQ0FBQyxFQUNYLFFBQVEsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUF5QixLQUFLLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FDeEYsQ0FBQztJQUNSLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELFNBQVMsVUFBVSxDQUNmLFNBQWlDLEVBQUUsU0FBOEIsRUFDakUseUJBQStDLEVBQUUsY0FBd0I7SUFDM0UsTUFBTSxPQUFPLEdBQUcsU0FBUyxDQUFDLFFBQVEsQ0FBQztJQUNuQyxPQUFPLFdBQVcsQ0FBQyxPQUFPLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFBRSxjQUFjLENBQUM7U0FDNUQsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQWlCLEVBQUUsRUFBRTtRQUM5QixTQUFTLENBQUMsYUFBYSxHQUFHLFlBQVksQ0FBQztRQUN2QyxTQUFTLENBQUMsSUFBSSxHQUFHO1lBQ2YsR0FBRyxTQUFTLENBQUMsSUFBSTtZQUNqQixHQUFHLDBCQUEwQixDQUFDLFNBQVMsRUFBRSx5QkFBeUIsQ0FBQyxDQUFDLE9BQU87U0FDNUUsQ0FBQztRQUNGLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUNWLENBQUM7QUFFRCxTQUFTLFdBQVcsQ0FDaEIsT0FBb0IsRUFBRSxTQUFpQyxFQUFFLFNBQThCLEVBQ3ZGLGNBQXdCO0lBQzFCLE1BQU0sSUFBSSxHQUFHLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNsQyxJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQ3JCLE9BQU8sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0tBQ2Y7SUFDRCxNQUFNLElBQUksR0FBOEIsRUFBRSxDQUFDO0lBQzNDLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FDbEIsUUFBUSxDQUNKLEdBQUcsQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLGNBQWMsQ0FBQztTQUMxRCxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBVSxFQUFFLEVBQUU7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztJQUNwQixDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQ25CLFFBQVEsQ0FBQyxDQUFDLENBQUMsRUFDWCxRQUFRLENBQUMsR0FBRyxFQUFFO1FBQ1osMEZBQTBGO1FBQzFGLHlDQUF5QztRQUN6QyxJQUFJLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLEtBQUssSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUM1QyxPQUFPLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNqQjtRQUNELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQyxDQUFDLENBQ0wsQ0FBQztBQUNKLENBQUM7QUFFRCxTQUFTLFdBQVcsQ0FBQyxHQUFXO0lBQzlCLE9BQU8sQ0FBQyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsR0FBRyxNQUFNLENBQUMscUJBQXFCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztBQUNyRSxDQUFDO0FBRUQsU0FBUyxXQUFXLENBQ2hCLGNBQW1CLEVBQUUsU0FBaUMsRUFBRSxTQUE4QixFQUN0RixjQUF3QjtJQUMxQixNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsY0FBYyxFQUFFLFNBQVMsRUFBRSxjQUFjLENBQUMsQ0FBQztJQUNyRSxPQUFPLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM1RCxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDLENBQUM7QUFDL0UsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQge0luamVjdG9yfSBmcm9tICdAYW5ndWxhci9jb3JlJztcbmltcG9ydCB7RU1QVFksIGZyb20sIE1vbm9UeXBlT3BlcmF0b3JGdW5jdGlvbiwgT2JzZXJ2YWJsZSwgb2Z9IGZyb20gJ3J4anMnO1xuaW1wb3J0IHtjb25jYXRNYXAsIG1hcCwgbWVyZ2VNYXAsIHRha2VMYXN0LCB0YXB9IGZyb20gJ3J4anMvb3BlcmF0b3JzJztcblxuaW1wb3J0IHtSZXNvbHZlRGF0YX0gZnJvbSAnLi4vY29uZmlnJztcbmltcG9ydCB7TmF2aWdhdGlvblRyYW5zaXRpb259IGZyb20gJy4uL3JvdXRlcic7XG5pbXBvcnQge0FjdGl2YXRlZFJvdXRlU25hcHNob3QsIGluaGVyaXRlZFBhcmFtc0RhdGFSZXNvbHZlLCBSb3V0ZXJTdGF0ZVNuYXBzaG90fSBmcm9tICcuLi9yb3V0ZXJfc3RhdGUnO1xuaW1wb3J0IHt3cmFwSW50b09ic2VydmFibGV9IGZyb20gJy4uL3V0aWxzL2NvbGxlY3Rpb24nO1xuaW1wb3J0IHtnZXRUb2tlbn0gZnJvbSAnLi4vdXRpbHMvcHJlYWN0aXZhdGlvbic7XG5cbmV4cG9ydCBmdW5jdGlvbiByZXNvbHZlRGF0YShcbiAgICBwYXJhbXNJbmhlcml0YW5jZVN0cmF0ZWd5OiAnZW1wdHlPbmx5J3wnYWx3YXlzJyxcbiAgICBtb2R1bGVJbmplY3RvcjogSW5qZWN0b3IpOiBNb25vVHlwZU9wZXJhdG9yRnVuY3Rpb248TmF2aWdhdGlvblRyYW5zaXRpb24+IHtcbiAgcmV0dXJuIG1lcmdlTWFwKHQgPT4ge1xuICAgIGNvbnN0IHt0YXJnZXRTbmFwc2hvdCwgZ3VhcmRzOiB7Y2FuQWN0aXZhdGVDaGVja3N9fSA9IHQ7XG5cbiAgICBpZiAoIWNhbkFjdGl2YXRlQ2hlY2tzLmxlbmd0aCkge1xuICAgICAgcmV0dXJuIG9mKHQpO1xuICAgIH1cbiAgICBsZXQgY2FuQWN0aXZhdGVDaGVja3NSZXNvbHZlZCA9IDA7XG4gICAgcmV0dXJuIGZyb20oY2FuQWN0aXZhdGVDaGVja3MpXG4gICAgICAgIC5waXBlKFxuICAgICAgICAgICAgY29uY2F0TWFwKFxuICAgICAgICAgICAgICAgIGNoZWNrID0+IHJ1blJlc29sdmUoXG4gICAgICAgICAgICAgICAgICAgIGNoZWNrLnJvdXRlLCB0YXJnZXRTbmFwc2hvdCEsIHBhcmFtc0luaGVyaXRhbmNlU3RyYXRlZ3ksIG1vZHVsZUluamVjdG9yKSksXG4gICAgICAgICAgICB0YXAoKCkgPT4gY2FuQWN0aXZhdGVDaGVja3NSZXNvbHZlZCsrKSxcbiAgICAgICAgICAgIHRha2VMYXN0KDEpLFxuICAgICAgICAgICAgbWVyZ2VNYXAoXyA9PiBjYW5BY3RpdmF0ZUNoZWNrc1Jlc29sdmVkID09PSBjYW5BY3RpdmF0ZUNoZWNrcy5sZW5ndGggPyBvZih0KSA6IEVNUFRZKSxcbiAgICAgICAgKTtcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHJ1blJlc29sdmUoXG4gICAgZnV0dXJlQVJTOiBBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90LCBmdXR1cmVSU1M6IFJvdXRlclN0YXRlU25hcHNob3QsXG4gICAgcGFyYW1zSW5oZXJpdGFuY2VTdHJhdGVneTogJ2VtcHR5T25seSd8J2Fsd2F5cycsIG1vZHVsZUluamVjdG9yOiBJbmplY3Rvcikge1xuICBjb25zdCByZXNvbHZlID0gZnV0dXJlQVJTLl9yZXNvbHZlO1xuICByZXR1cm4gcmVzb2x2ZU5vZGUocmVzb2x2ZSwgZnV0dXJlQVJTLCBmdXR1cmVSU1MsIG1vZHVsZUluamVjdG9yKVxuICAgICAgLnBpcGUobWFwKChyZXNvbHZlZERhdGE6IGFueSkgPT4ge1xuICAgICAgICBmdXR1cmVBUlMuX3Jlc29sdmVkRGF0YSA9IHJlc29sdmVkRGF0YTtcbiAgICAgICAgZnV0dXJlQVJTLmRhdGEgPSB7XG4gICAgICAgICAgLi4uZnV0dXJlQVJTLmRhdGEsXG4gICAgICAgICAgLi4uaW5oZXJpdGVkUGFyYW1zRGF0YVJlc29sdmUoZnV0dXJlQVJTLCBwYXJhbXNJbmhlcml0YW5jZVN0cmF0ZWd5KS5yZXNvbHZlXG4gICAgICAgIH07XG4gICAgICAgIHJldHVybiBudWxsO1xuICAgICAgfSkpO1xufVxuXG5mdW5jdGlvbiByZXNvbHZlTm9kZShcbiAgICByZXNvbHZlOiBSZXNvbHZlRGF0YSwgZnV0dXJlQVJTOiBBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90LCBmdXR1cmVSU1M6IFJvdXRlclN0YXRlU25hcHNob3QsXG4gICAgbW9kdWxlSW5qZWN0b3I6IEluamVjdG9yKTogT2JzZXJ2YWJsZTxhbnk+IHtcbiAgY29uc3Qga2V5cyA9IGdldERhdGFLZXlzKHJlc29sdmUpO1xuICBpZiAoa2V5cy5sZW5ndGggPT09IDApIHtcbiAgICByZXR1cm4gb2Yoe30pO1xuICB9XG4gIGNvbnN0IGRhdGE6IHtbazogc3RyaW5nfHN5bWJvbF06IGFueX0gPSB7fTtcbiAgcmV0dXJuIGZyb20oa2V5cykucGlwZShcbiAgICAgIG1lcmdlTWFwKFxuICAgICAgICAgIGtleSA9PiBnZXRSZXNvbHZlcihyZXNvbHZlW2tleV0sIGZ1dHVyZUFSUywgZnV0dXJlUlNTLCBtb2R1bGVJbmplY3RvcilcbiAgICAgICAgICAgICAgICAgICAgIC5waXBlKHRhcCgodmFsdWU6IGFueSkgPT4ge1xuICAgICAgICAgICAgICAgICAgICAgICBkYXRhW2tleV0gPSB2YWx1ZTtcbiAgICAgICAgICAgICAgICAgICAgIH0pKSksXG4gICAgICB0YWtlTGFzdCgxKSxcbiAgICAgIG1lcmdlTWFwKCgpID0+IHtcbiAgICAgICAgLy8gRW5zdXJlIGFsbCByZXNvbHZlcnMgcmV0dXJuZWQgdmFsdWVzLCBvdGhlcndpc2UgZG9uJ3QgZW1pdCBhbnkgXCJuZXh0XCIgYW5kIGp1c3QgY29tcGxldGVcbiAgICAgICAgLy8gdGhlIGNoYWluIHdoaWNoIHdpbGwgY2FuY2VsIG5hdmlnYXRpb25cbiAgICAgICAgaWYgKGdldERhdGFLZXlzKGRhdGEpLmxlbmd0aCA9PT0ga2V5cy5sZW5ndGgpIHtcbiAgICAgICAgICByZXR1cm4gb2YoZGF0YSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIEVNUFRZO1xuICAgICAgfSksXG4gICk7XG59XG5cbmZ1bmN0aW9uIGdldERhdGFLZXlzKG9iajogT2JqZWN0KTogQXJyYXk8c3RyaW5nfHN5bWJvbD4ge1xuICByZXR1cm4gWy4uLk9iamVjdC5rZXlzKG9iaiksIC4uLk9iamVjdC5nZXRPd25Qcm9wZXJ0eVN5bWJvbHMob2JqKV07XG59XG5cbmZ1bmN0aW9uIGdldFJlc29sdmVyKFxuICAgIGluamVjdGlvblRva2VuOiBhbnksIGZ1dHVyZUFSUzogQWN0aXZhdGVkUm91dGVTbmFwc2hvdCwgZnV0dXJlUlNTOiBSb3V0ZXJTdGF0ZVNuYXBzaG90LFxuICAgIG1vZHVsZUluamVjdG9yOiBJbmplY3Rvcik6IE9ic2VydmFibGU8YW55PiB7XG4gIGNvbnN0IHJlc29sdmVyID0gZ2V0VG9rZW4oaW5qZWN0aW9uVG9rZW4sIGZ1dHVyZUFSUywgbW9kdWxlSW5qZWN0b3IpO1xuICByZXR1cm4gcmVzb2x2ZXIucmVzb2x2ZSA/IHdyYXBJbnRvT2JzZXJ2YWJsZShyZXNvbHZlci5yZXNvbHZlKGZ1dHVyZUFSUywgZnV0dXJlUlNTKSkgOlxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHdyYXBJbnRvT2JzZXJ2YWJsZShyZXNvbHZlcihmdXR1cmVBUlMsIGZ1dHVyZVJTUykpO1xufVxuIl19