/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { concat, defer, from, of } from 'rxjs';
import { concatMap, first, map, mergeMap } from 'rxjs/operators';
import { ActivationStart, ChildActivationStart } from '../events';
import { wrapIntoObservable } from '../utils/collection';
import { getCanActivateChild, getToken } from '../utils/preactivation';
import { isBoolean, isCanActivate, isCanActivateChild, isCanDeactivate, isFunction } from '../utils/type_guards';
import { prioritizedGuardValue } from './prioritized_guard_value';
export function checkGuards(moduleInjector, forwardEvent) {
    return mergeMap(t => {
        const { targetSnapshot, currentSnapshot, guards: { canActivateChecks, canDeactivateChecks } } = t;
        if (canDeactivateChecks.length === 0 && canActivateChecks.length === 0) {
            return of({ ...t, guardsResult: true });
        }
        return runCanDeactivateChecks(canDeactivateChecks, targetSnapshot, currentSnapshot, moduleInjector)
            .pipe(mergeMap(canDeactivate => {
            return canDeactivate && isBoolean(canDeactivate) ?
                runCanActivateChecks(targetSnapshot, canActivateChecks, moduleInjector, forwardEvent) :
                of(canDeactivate);
        }), map(guardsResult => ({ ...t, guardsResult })));
    });
}
function runCanDeactivateChecks(checks, futureRSS, currRSS, moduleInjector) {
    return from(checks).pipe(mergeMap(check => runCanDeactivate(check.component, check.route, currRSS, futureRSS, moduleInjector)), first(result => {
        return result !== true;
    }, true));
}
function runCanActivateChecks(futureSnapshot, checks, moduleInjector, forwardEvent) {
    return from(checks).pipe(concatMap((check) => {
        return concat(fireChildActivationStart(check.route.parent, forwardEvent), fireActivationStart(check.route, forwardEvent), runCanActivateChild(futureSnapshot, check.path, moduleInjector), runCanActivate(futureSnapshot, check.route, moduleInjector));
    }), first(result => {
        return result !== true;
    }, true));
}
/**
 * This should fire off `ActivationStart` events for each route being activated at this
 * level.
 * In other words, if you're activating `a` and `b` below, `path` will contain the
 * `ActivatedRouteSnapshot`s for both and we will fire `ActivationStart` for both. Always
 * return
 * `true` so checks continue to run.
 */
function fireActivationStart(snapshot, forwardEvent) {
    if (snapshot !== null && forwardEvent) {
        forwardEvent(new ActivationStart(snapshot));
    }
    return of(true);
}
/**
 * This should fire off `ChildActivationStart` events for each route being activated at this
 * level.
 * In other words, if you're activating `a` and `b` below, `path` will contain the
 * `ActivatedRouteSnapshot`s for both and we will fire `ChildActivationStart` for both. Always
 * return
 * `true` so checks continue to run.
 */
function fireChildActivationStart(snapshot, forwardEvent) {
    if (snapshot !== null && forwardEvent) {
        forwardEvent(new ChildActivationStart(snapshot));
    }
    return of(true);
}
function runCanActivate(futureRSS, futureARS, moduleInjector) {
    const canActivate = futureARS.routeConfig ? futureARS.routeConfig.canActivate : null;
    if (!canActivate || canActivate.length === 0)
        return of(true);
    const canActivateObservables = canActivate.map((c) => {
        return defer(() => {
            const guard = getToken(c, futureARS, moduleInjector);
            let observable;
            if (isCanActivate(guard)) {
                observable = wrapIntoObservable(guard.canActivate(futureARS, futureRSS));
            }
            else if (isFunction(guard)) {
                observable = wrapIntoObservable(guard(futureARS, futureRSS));
            }
            else {
                throw new Error('Invalid CanActivate guard');
            }
            return observable.pipe(first());
        });
    });
    return of(canActivateObservables).pipe(prioritizedGuardValue());
}
function runCanActivateChild(futureRSS, path, moduleInjector) {
    const futureARS = path[path.length - 1];
    const canActivateChildGuards = path.slice(0, path.length - 1)
        .reverse()
        .map(p => getCanActivateChild(p))
        .filter(_ => _ !== null);
    const canActivateChildGuardsMapped = canActivateChildGuards.map((d) => {
        return defer(() => {
            const guardsMapped = d.guards.map((c) => {
                const guard = getToken(c, d.node, moduleInjector);
                let observable;
                if (isCanActivateChild(guard)) {
                    observable = wrapIntoObservable(guard.canActivateChild(futureARS, futureRSS));
                }
                else if (isFunction(guard)) {
                    observable = wrapIntoObservable(guard(futureARS, futureRSS));
                }
                else {
                    throw new Error('Invalid CanActivateChild guard');
                }
                return observable.pipe(first());
            });
            return of(guardsMapped).pipe(prioritizedGuardValue());
        });
    });
    return of(canActivateChildGuardsMapped).pipe(prioritizedGuardValue());
}
function runCanDeactivate(component, currARS, currRSS, futureRSS, moduleInjector) {
    const canDeactivate = currARS && currARS.routeConfig ? currARS.routeConfig.canDeactivate : null;
    if (!canDeactivate || canDeactivate.length === 0)
        return of(true);
    const canDeactivateObservables = canDeactivate.map((c) => {
        const guard = getToken(c, currARS, moduleInjector);
        let observable;
        if (isCanDeactivate(guard)) {
            observable = wrapIntoObservable(guard.canDeactivate(component, currARS, currRSS, futureRSS));
        }
        else if (isFunction(guard)) {
            observable = wrapIntoObservable(guard(component, currARS, currRSS, futureRSS));
        }
        else {
            throw new Error('Invalid CanDeactivate guard');
        }
        return observable.pipe(first());
    });
    return of(canDeactivateObservables).pipe(prioritizedGuardValue());
}
//# sourceMappingURL=data:application/json;base64,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