/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY, EMPTY_OBJ } from '../util/empty';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { NG_COMP_DEF, NG_DIR_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
let _renderCompCount = 0;
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const type = componentDefinition.type;
        const declaredInputs = {};
        const def = {
            type: type,
            providersResolver: null,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            factory: null,
            template: componentDefinition.template || null,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            hostBindings: componentDefinition.hostBindings || null,
            hostVars: componentDefinition.hostVars || 0,
            hostAttrs: componentDefinition.hostAttrs || null,
            contentQueries: componentDefinition.contentQueries || null,
            declaredInputs: declaredInputs,
            inputs: null,
            outputs: null,
            exportAs: componentDefinition.exportAs || null,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null,
            pipeDefs: null,
            selectors: componentDefinition.selectors || EMPTY_ARRAY,
            viewQuery: componentDefinition.viewQuery || null,
            features: componentDefinition.features || null,
            data: componentDefinition.data || {},
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            id: 'c',
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            setInput: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
        };
        const directiveTypes = componentDefinition.directives;
        const feature = componentDefinition.features;
        const pipeTypes = componentDefinition.pipes;
        def.id += _renderCompCount++;
        def.inputs = invertObject(componentDefinition.inputs, declaredInputs),
            def.outputs = invertObject(componentDefinition.outputs),
            feature && feature.forEach((fn) => fn(def));
        def.directiveDefs = directiveTypes ?
            () => (typeof directiveTypes === 'function' ? directiveTypes() : directiveTypes)
                .map(extractDirectiveDef) :
            null;
        def.pipeDefs = pipeTypes ?
            () => (typeof pipeTypes === 'function' ? pipeTypes() : pipeTypes).map(extractPipeDef) :
            null;
        return def;
    });
}
/**
 * Generated next to NgModules to monkey-patch directive and pipe references onto a component's
 * definition, when generating a direct reference in the component file would otherwise create an
 * import cycle.
 *
 * See [this explanation](https://hackmd.io/Odw80D0pR6yfsOjg_7XCJg?view) for more details.
 *
 * @codeGenApi
 */
export function ɵɵsetComponentScope(type, directives, pipes) {
    const def = type.ɵcmp;
    def.directiveDefs = () => directives.map(extractDirectiveDef);
    def.pipeDefs = () => pipes.map(extractPipeDef);
}
export function extractDirectiveDef(type) {
    const def = getComponentDef(type) || getDirectiveDef(type);
    if (ngDevMode && !def) {
        throw new Error(`'${type.name}' is neither 'ComponentType' or 'DirectiveType'.`);
    }
    return def;
}
export function extractPipeDef(type) {
    const def = getPipeDef(type);
    if (ngDevMode && !def) {
        throw new Error(`'${type.name}' is not a 'PipeType'.`);
    }
    return def;
}
export const autoRegisterModuleById = {};
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    return noSideEffects(() => {
        const res = {
            type: def.type,
            bootstrap: def.bootstrap || EMPTY_ARRAY,
            declarations: def.declarations || EMPTY_ARRAY,
            imports: def.imports || EMPTY_ARRAY,
            exports: def.exports || EMPTY_ARRAY,
            transitiveCompileScopes: null,
            schemas: def.schemas || null,
            id: def.id || null,
        };
        if (def.id != null) {
            autoRegisterModuleById[def.id] = def.type;
        }
        return res;
    });
}
/**
 * Adds the module metadata that is necessary to compute the module's transitive scope to an
 * existing module definition.
 *
 * Scope metadata of modules is not used in production builds, so calls to this function can be
 * marked pure to tree-shake it from the bundle, allowing for all referenced declarations
 * to become eligible for tree-shaking as well.
 *
 * @codeGenApi
 */
export function ɵɵsetNgModuleScope(type, scope) {
    return noSideEffects(() => {
        const ngModuleDef = getNgModuleDef(type, true);
        ngModuleDef.declarations = scope.declarations || EMPTY_ARRAY;
        ngModuleDef.imports = scope.imports || EMPTY_ARRAY;
        ngModuleDef.exports = scope.exports || EMPTY_ARRAY;
    });
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 *   @Input()
 *   propName1: string;
 *
 *   @Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *

 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            let publicName = obj[minifiedKey];
            let declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export const ɵɵdefineDirective = ɵɵdefineComponent;
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    };
}
/**
 * The following getter methods retrieve the definition from the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
//# sourceMappingURL=data:application/json;base64,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