/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { isComponentHost } from '../render3/interfaces/type_checks';
import { DECLARATION_COMPONENT_VIEW } from '../render3/interfaces/view';
import { getCurrentTNode, getLView } from '../render3/state';
import { getComponentLViewByIndex } from '../render3/util/view_utils';
import { ViewRef as R3_ViewRef } from '../render3/view_ref';
/**
 * Base class that provides change detection functionality.
 * A change-detection tree collects all views that are to be checked for changes.
 * Use the methods to add and remove views from the tree, initiate change-detection,
 * and explicitly mark views as _dirty_, meaning that they have changed and need to be re-rendered.
 *
 * @see [Using change detection hooks](guide/lifecycle-hooks#using-change-detection-hooks)
 * @see [Defining custom change detection](guide/lifecycle-hooks#defining-custom-change-detection)
 *
 * @usageNotes
 *
 * The following examples demonstrate how to modify default change-detection behavior
 * to perform explicit detection when needed.
 *
 * ### Use `markForCheck()` with `CheckOnce` strategy
 *
 * The following example sets the `OnPush` change-detection strategy for a component
 * (`CheckOnce`, rather than the default `CheckAlways`), then forces a second check
 * after an interval. See [live demo](https://plnkr.co/edit/GC512b?p=preview).
 *
 * <code-example path="core/ts/change_detect/change-detection.ts"
 * region="mark-for-check"></code-example>
 *
 * ### Detach change detector to limit how often check occurs
 *
 * The following example defines a component with a large list of read-only data
 * that is expected to change constantly, many times per second.
 * To improve performance, we want to check and update the list
 * less often than the changes actually occur. To do that, we detach
 * the component's change detector and perform an explicit local check every five seconds.
 *
 * <code-example path="core/ts/change_detect/change-detection.ts" region="detach"></code-example>
 *
 *
 * ### Reattaching a detached component
 *
 * The following example creates a component displaying live data.
 * The component detaches its change detector from the main change detector tree
 * when the `live` property is set to false, and reattaches it when the property
 * becomes true.
 *
 * <code-example path="core/ts/change_detect/change-detection.ts" region="reattach"></code-example>
 *
 * @publicApi
 */
export class ChangeDetectorRef {
}
/**
 * @internal
 * @nocollapse
 */
ChangeDetectorRef.__NG_ELEMENT_ID__ = injectChangeDetectorRef;
/** Returns a ChangeDetectorRef (a.k.a. a ViewRef) */
export function injectChangeDetectorRef(flags) {
    return createViewRef(getCurrentTNode(), getLView(), (flags & 16 /* ForPipe */) === 16 /* ForPipe */);
}
/**
 * Creates a ViewRef and stores it on the injector as ChangeDetectorRef (public alias).
 *
 * @param tNode The node that is requesting a ChangeDetectorRef
 * @param lView The view to which the node belongs
 * @param isPipe Whether the view is being injected into a pipe.
 * @returns The ChangeDetectorRef to use
 */
function createViewRef(tNode, lView, isPipe) {
    if (isComponentHost(tNode) && !isPipe) {
        // The LView represents the location where the component is declared.
        // Instead we want the LView for the component View and so we need to look it up.
        const componentView = getComponentLViewByIndex(tNode.index, lView); // look down
        return new R3_ViewRef(componentView, componentView);
    }
    else if (tNode.type & (3 /* AnyRNode */ | 12 /* AnyContainer */ | 32 /* Icu */)) {
        // The LView represents the location where the injection is requested from.
        // We need to locate the containing LView (in case where the `lView` is an embedded view)
        const hostComponentView = lView[DECLARATION_COMPONENT_VIEW]; // look up
        return new R3_ViewRef(hostComponentView, lView);
    }
    return null;
}
//# sourceMappingURL=data:application/json;base64,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