export type ChangeEventPayload = {
    value: string;
};
export type ErrorEventPayload = {
    value: string;
};
export type StateEventPayload = {
    value: "pending" | "none";
};
/**
 * Options for configuring the `useShareIntent` hook.
 */
export type ShareIntentOptions = {
    /**
     * If `true`, includes additional logs for debugging.
     * @default false
     */
    debug?: boolean;
    /**
     * If `true`, resets the shared content when the
     * app goes into the background / foreground.
     * @default true
     */
    resetOnBackground?: boolean;
    /**
     * If `true`, disables shared intent.
     * @default false
     */
    disabled?: boolean;
    /**
     * Optional force application scheme to retreive ShareIntent on iOS.
     */
    scheme?: string;
    /**
     * Optional callback function that is triggered when the shared media resets.
     */
    onResetShareIntent?: () => void;
};
export type ShareIntentMeta = Record<string, string | undefined> & {
    title?: string;
};
/**
 * Base type for what shared content is common between both platforms.
 */
interface BaseShareIntent {
    meta?: ShareIntentMeta | null;
    text?: string | null;
}
/**
 * Shared intent to represent both platforms.
 */
export type ShareIntent = BaseShareIntent & {
    files: ShareIntentFile[] | null;
    type: "media" | "file" | "text" | "weburl" | null;
    webUrl: string | null;
};
/**
 * Shared intent type for Android.
 */
export interface AndroidShareIntent extends BaseShareIntent {
    files?: AndroidShareIntentFile[];
    type: "file" | "text";
}
/**
 * Shared intent type for iOS.
 */
export interface IosShareIntent extends BaseShareIntent {
    files?: IosShareIntentFile[];
    weburls?: {
        url: string;
        meta: string;
    }[];
    type: "media" | "file" | "text" | "weburl";
}
/**
 * ShareIntentFile that is common among both platforms
 */
export type ShareIntentFile = {
    fileName: string;
    mimeType: string;
    path: string;
    size: number | null;
    width: number | null;
    height: number | null;
    duration: number | null;
};
/**
 * ShareIntentFile in iOS
 */
export interface IosShareIntentFile {
    fileSize?: number;
    fileName: string;
    mimeType: string;
    path: string;
    type: "0" | "1" | "2" | "3";
    width: number | null;
    height: number | null;
    duration: number | null;
}
/**
 * ShareIntentFile in Android
 */
export interface AndroidShareIntentFile {
    contentUri: string;
    mimeType: string;
    fileName: string;
    filePath: string;
    fileSize?: string;
    width: number | null;
    height: number | null;
    duration: number | null;
}
export type NativeShareIntent = AndroidShareIntent | IosShareIntent;
export type NativeShareIntentFile = AndroidShareIntentFile | IosShareIntentFile;
export {};
//# sourceMappingURL=ExpoShareIntentModule.types.d.ts.map