/*

    Copyright 2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024  joshua.tee@gmail.com

    This file is part of wX.

    wX is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    wX is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with wX.  If not, see <http://www.gnu.org/licenses/>.

*/

package joshuatee.wx.ui

import android.content.Context
import android.view.Gravity
import android.view.ViewGroup
import android.widget.LinearLayout
import android.widget.SeekBar
import androidx.appcompat.widget.AppCompatSeekBar
import joshuatee.wx.settings.RadarPreferences
import joshuatee.wx.settings.UIPreferences
import joshuatee.wx.util.To
import joshuatee.wx.util.Utility

internal class NumberPicker(
    context: Context,
    val label: String,
    pref: String,
    strId: Int,
    private val defValue: Int,
    private val lowValue: Int,
    highValue: Int
) : Widget {

    private val card = Card(context)
    private val initValue = when (pref) {
        "RADAR_TEXT_SIZE" -> (Utility.readPrefFloat(context, pref, defValue.toFloat()) * 10).toInt()
        "UI_ANIM_ICON_FRAMES" -> To.int(
            Utility.readPref(
                context,
                pref,
                RadarPreferences.uiAnimIconFrames
            )
        )

        "CARD_CORNER_RADIUS" -> Utility.readPrefInt(context, pref, 0)
        else -> Utility.readPrefInt(context, pref, defValue)
    }
    private val text = Text(context)
    private val seekBar = AppCompatSeekBar(context)

    init {
        with(text) {
            setPadding(UIPreferences.padding)
            wrap()
            gravity = Gravity.TOP
            connect { ObjectDialogue(context, context.resources.getString(strId)) }
        }
        val vbox = VBox.centered(context)
        vbox.matchParent()
        vbox.addWidget(text)
        seekBar.max = highValue - lowValue
        seekBar.progress = convert(initValue)
        val padding = 30
        val layoutParams = LinearLayout.LayoutParams(
            ViewGroup.LayoutParams.MATCH_PARENT,
            ViewGroup.LayoutParams.WRAP_CONTENT
        )
        layoutParams.setMargins(padding, padding, padding, padding)
        seekBar.layoutParams = layoutParams
        vbox.addWidget(seekBar)
        card.addLayout(vbox)
        updateLabel()
        seekBar.setOnSeekBarChangeListener(object : SeekBar.OnSeekBarChangeListener {
            override fun onProgressChanged(seekBar: SeekBar, progress: Int, fromUser: Boolean) {
                if (pref == "TEXTVIEW_FONT_SIZE") {
                    text.setTextSize(UtilityUI.spToPx(convertForSave(seekBar.progress), context))
                }
                updateLabel()
            }

            override fun onStartTrackingTouch(seekBar: SeekBar) {
                // Write code to perform some action when touch is started.
            }

            override fun onStopTrackingTouch(seekBar: SeekBar) {
                val newVal = convertForSave(seekBar.progress)
                when (pref) {
                    "RADAR_TEXT_SIZE" -> Utility.writePrefFloat(context, pref, newVal / 10.0f)
                    "UI_ANIM_ICON_FRAMES" -> Utility.writePref(context, pref, newVal.toString())
                    else -> Utility.writePrefInt(context, pref, newVal)
                }
                Utility.writePref(context, "RESTART_NOTIF", "true")
            }
        })
    }

    private fun convert(value: Int): Int = value - lowValue

    private fun convertForSave(value: Int): Int = value + lowValue

    fun updateLabel() {
        text.text =
            label + " (default is " + defValue.toString() + "): " + convertForSave(seekBar.progress).toString()
    }

    override fun getView() = card.getView()
}
