/*

    Copyright 2013, 2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024  joshua.tee@gmail.com

    This file is part of wX.

    wX is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    wX is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with wX.  If not, see <http://www.gnu.org/licenses/>.

*/

package joshuatee.wx.radar

import joshuatee.wx.external.ExternalGeodeticCalculator
import joshuatee.wx.external.ExternalGlobalCoordinates
import joshuatee.wx.parseColumn
import joshuatee.wx.parseColumnAll
import joshuatee.wx.util.To
import java.util.regex.Pattern

internal object NexradLevel3HailIndex {

    private const val MARKER_SIZE = 0.015
    private val pattern1: Pattern = Pattern.compile("AZ/RAN(.*?)V")
    private val pattern2: Pattern = Pattern.compile("POSH/POH(.*?)V")
    private val pattern3: Pattern = Pattern.compile("MAX HAIL SIZE(.*?)V")
    private val pattern4: Pattern = Pattern.compile("[0-9]*\\.?[0-9]+")
    private val pattern5: Pattern = Pattern.compile("\\d+")

    fun decode(radarSite: String): List<Double> {
        val location = RadarSites.getLatLon(radarSite)
        val data = NexradLevel3TextProduct.download("HI", radarSite)
        val posn = data.parseColumn(pattern1)
        val posnStr = posn.joinToString("")
            .replace("/", " ")

        val hailPercent = data.parseColumn(pattern2)
        val hailPercentStr = hailPercent.joinToString("")
            .replace("/", " ")
            .replace("UNKNOWN", " 0 0 ")

        val hailSize = data.parseColumn(pattern3)
        val hailSizeStr = hailSize.joinToString("")
            .replace("/", " ")
            .replace("UNKNOWN", " 0.00 ")
            .replace("<0.50", " 0.49 ")

        val posnNumbers = posnStr.parseColumnAll(pattern5)
        val hailPercentNumbers = hailPercentStr.parseColumnAll(pattern5)
        val hailSizeNumbers = hailSizeStr.parseColumnAll(pattern4)
        val stormList = mutableListOf<Double>()
        if (posnNumbers.size == hailPercentNumbers.size && posnNumbers.size > 1 && hailSizeNumbers.isNotEmpty()) {
            var k = 0 // k is used to track hail size which is /2 of other 2 arrays
            for (s in posnNumbers.indices step 2) {
                val hailSizeDbl = To.double(hailSizeNumbers[k])
                if (hailSizeDbl > 0.49 && (To.int(hailPercentNumbers[s]) > 60 || To.double(
                        hailPercentNumbers[s + 1]
                    ) > 60.0)
                ) {
                    val degree = To.double(posnNumbers[s])
                    val nm = To.double(posnNumbers[s + 1])
                    val start = ExternalGlobalCoordinates(location)
                    val ec = ExternalGeodeticCalculator.calculateEndingGlobalCoordinates(
                        start,
                        degree,
                        nm * 1852.0
                    )
                    stormList.add(ec.latitude)
                    stormList.add(ec.longitude * -1.0)
                    if (hailSizeDbl > 0.99) {
                        stormList.add(ec.latitude + MARKER_SIZE)
                        stormList.add(ec.longitude * -1.0)
                    }
                    if (hailSizeDbl > 1.99) {
                        stormList.add(ec.latitude + MARKER_SIZE * 2.0)
                        stormList.add(ec.longitude * -1.0)
                    }
                    if (hailSizeDbl > 2.99) {
                        stormList.add(ec.latitude + MARKER_SIZE * 3.0)
                        stormList.add(ec.longitude * -1.0)
                    }
                }
                k += 1
            }
        }
        return stormList
    }
}
