/*
*   it.rgp.nyagua.pafcalc.PAFCalc - Planted Aquarium Fertilizer Calculator
*   Copyright (C) 2013 Rudi Giacomini Pilon
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
*/

/*
 * This calculator is heavily based on flores' yanc (Yet Another Nutrient Calculator)
 * at http://calc.petalphile.com/ shared as Open Source.
 * Thanks to mr. Flores for this concept piece.
 *
 */

/*
 *
 * Fertilization Solutions calculator
 * -----------------------------------
 * This form is used to do some calculations on fertilisation argument
 * Some therms and situations need to be defined to understand the code.
 *
 * The calculations may operate over commercial products or over diy (do-it-yourself) products.
 * Last ones can be powder or solutions of basic elements or compounds.
 * You can use them as pure powder or solution (as-is) or add water or mix other elements to obtain
 * your own diy solution.
 * So what is involved in operation is:
 * 1) The tank water volume
 * 2) The commercial or diy initial compound
 * 3) The target element - that is the primary element you add to the water of
 *      aquarium when you add compound (secondary elements qty are also showed)
 * 4) The target - that is the amount of element you want in aquarium
 * 5) The solution water - If you are creating a solution then the water you use is needed
 * 6) Dose volume - The dose of solution you will add to tank. It should be a confortable dose
 *  not to little that you cant measure it, not so big that you can't use
 * 7) The results depend on calculation you request.
 *
 * Calculations:
 * a) Solution to reach target
 * b) The result of a dose
 * c) Estimative Index
 * d) Estimative Index Daily
 * e) Estimative Index Weekly
 * f) Perpetual Preservation System
 * g) Poor Man Dosing Drops
 *
 *
 *
 @author Rudi Giacomini Pilon
 */

package it.rgp.nyagua.pafcalc;

import android.app.Activity;
import android.content.Intent;
import android.content.res.Resources;
import android.os.Bundle;
import android.os.CountDownTimer;
import android.support.v4.app.NavUtils;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.EditText;
import android.widget.GridView;
import android.widget.ImageView;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.Toast;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import it.rgp.nyagua.pafcalc.util.Converter;
import it.rgp.nyagua.pafcalc.util.LocUtil;
import it.rgp.nyagua.pafcalc.util.tools;

public class CalculatorActivity extends Activity {

    //elements constants
    public static final int NO3=0;
    public static final int PO4=1;
    public static final int K=2;
    public static final int Ca=3;
    public static final int Mg=4;
    public static final int Fe=5;
    public static final int Mn=6;
    public static final int B=7;
    public static final int Cu=8;
    public static final int Mo=9;
    public static final int Cl=10;
    public static final int S=11;
    public static final int Zn=12;
    public static final int gH=13;
    public static final int Solubility=14;
    public static final int Target=15;
    //Methods
    /*METHOD_TARGET=0;
      METHOD_RESULT=1;
      METHOD_EI=2;
      METHOD_EID=3;
      METHOD_EIW=4;
      METHOD_PPS=5;
      METHOD_PMDD=6;
      METHOD_WALSTAD=1;//used in graphs*/

    /*
    # method (EI = Tom Barr's Estimative Index, PPS = Edward's Perpetual Preservation System,
    # Walstad = Diana Walstad's Natural Aquariums, PMDD = Poor Man's Dupla Drops)
    #*/

    //For each method and for each element a table of values is needed
    //Constant for values column:
    public static final int PAR_METHOD=0;  //Method Target
    /*public static final int PAR_LOW=1;     //Lowest value
    public static final int PAR_HIGH=2;     //Highest value
    public static final int PAR_MARGIN=3;   //margin=High-Low*/

    /* method parameter array is a double with following dimensions:
     * [Element: NO3=0 to Mn=6]    - Elements over Mn are not in graphs by now
     *   [Method: EI=2 to  PMDD=6] - Wet and Walstad not used by now
     *      [value: Target=0 to Margin=3]
     */
    public static double [][][] methods_parameters=new double [][][] {
            {  //NO3:
                    {3,1,30,29},//Wet:
                    {0,0.443,0.553,0.11},//Walstad:
                    {7.5 , 5 , 30, 25},//EI:
                    {3.2,0,0,0},//  EI_daily:
                    {10,0,0,0},//  EI_low:
                    {1,5,10,9}, //PPS:
                    {1.4,1,5,4}//PMDD:
            },
            {//PO4:
                    {1,1,5,4},//  Wet:
                    {0,0.061,0.073,0.012},//Walstad:
                    {1.3,1,3,2},//  EI:
                    {0.6,0,0,0},//  EI_daily:
                    {1,0,0,0},// EI_low:
                    {0.1,0.1,1,0.9},//  PPS:
                    {0,0,0,0}//  PMDD:
            },
            {//K:
                    {3, 1, 20, 19},//  Wet:
                    {0, 2, 0.4, 2.4},//  Walstad:
                    {7.5, 10, 30, 20}, //EI:
                    {3.2, 0, 0,0},//  EI_daily:
                    {10, 0, 0, 0},//  EI_low:
                    {1.33, 1,20, 15},//PPS:
                    {3, 2.4, 3.6, 1.2}//  PMDD:
            },
            {//Ca:
                    {2,0,0,0},//  Wet:
                    {0, 18, 34, 6},//  Walstad:
                    {15, 15, 30, 15},//  EI:
                    {6.4, 0, 0,0},//  EI_daily:
                    {15, 0, 0, 0},//  EI_low:
                    {0.4, 20, 30, 10},//PPS:
                    {0,0,0,0}//  PMDD:
            },
            {//Mg:
                    {0.5,0,0,0},//  Wet:
                    {0, 2, 2.4, 0.4},//Walstad:
                    {5,5,10,5},//EI:
                    {2, 0, 0,0},//  EI_daily:
                    {5, 0, 0, 0},//  EI_low:
                    {0.1, 2, 5, 3},//PPS:
                    {0.2, 0.16, 0.24, 0.08}//PMDD:
            },
            {// Fe:
                    {0.2,0.5,0.1,0.4},//  Wet:
                    {0, 0.06, 0.072, 0.012},//Walstad:
                    {0.5, 0.1, 1.0, 0.9},//  EI:
                    {0.2, 0, 0,0},//  EI_daily:
                    {0.2, 0, 0, 0},//  EI_low:
                    {0.1, 0.01, 0.1, 0.09},//PPS:
                    {0.1, 0.08, 0.12, 0.04}//PMDD:
            },
            {//Mn:
                    {0.2,0.5,0.1,0.4},//  Wet:
                    {0, 0.06, 0.072, 0.012},//Walstad:
                    {0.1, 0.1, 0.5, 0.4},//  EI:
                    {0.04, 0, 0,0},//  EI_daily:
                    {0.1, 0, 0, 0},//  EI_low:
                    {0.1, 0.01, 0.1, 0.09},//PPS:
                    {0.029, 0.023, 0.035, 0.012}//PMDD:
            }
    };

    HashMap solute;
    {
        solute = new HashMap();
    }

    private int method;
    private boolean enableCalc;

    //gui elements
    private TextView aquariumWaterVolText;
    private TextView solWaterVolumeEditText;
    private TextView doseVolumeEditText;
    private TextView addingText;
    private TextView targetEditText;
    private TextView showResultTextView;
    private RadioButton powderRadioButton;
    private RadioButton solutionRadioButton;
    private RadioButton unitsGalRadioButton;
    private RadioButton doseUnitsGRadioButton;
    private Spinner solNameSpinner;
    private ImageView solubleImageView;
    private GridView resultTableContainer;


    private TextWatcher watcher = new TextWatcher(){

        @Override
        public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {
        }

        @Override
        public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
        }

        @Override
        public void afterTextChanged(Editable editable) {
            checkFields();
        }
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_calc);

        //MyCode ####################
        loadDiy();
        //loadMethods();  //deprecated - methods are load in main
        resultTableContainer=(GridView) findViewById(R.id.resultTableContainer);
        buildResultTable(resultTableContainer,null);
        enableCalc=false;   //disabling calc button

        //Interface
        addingText= (TextView) findViewById(R.id.addingText);
        addingText.addTextChangedListener(watcher);
        aquariumWaterVolText= (TextView) findViewById(R.id.aquariumWaterVolText);
        aquariumWaterVolText.addTextChangedListener(watcher);
        doseUnitsGRadioButton =(RadioButton)findViewById(R.id.un3RadioButton);
        doseVolumeEditText= (TextView) findViewById(R.id.doseVolumeEditText);
        doseVolumeEditText.addTextChangedListener(watcher);
        powderRadioButton= (RadioButton)findViewById(R.id.powderRadioButton);
        showResultTextView=(TextView) findViewById(R.id.showResultTextView);
        solubleImageView=(ImageView)findViewById(R.id.solubleImageView);
        solWaterVolumeEditText= (TextView) findViewById(R.id.solWaterVolumeEditText);
        solWaterVolumeEditText.addTextChangedListener(watcher);
        targetEditText= (TextView) findViewById(R.id.targetEditText);
        targetEditText.addTextChangedListener(watcher);
        unitsGalRadioButton=(RadioButton)findViewById(R.id.radioButton2);
        solNameSpinner= (Spinner) findViewById(R.id.solNamespinner);
        solNameSpinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parentView, View selectedItemView, int position, long id) {
                String selected= parentView.getItemAtPosition(position).toString();
                RefreshTargetLabel(selected);
                setTarget();
            }
            @Override
            public void onNothingSelected(AdapterView<?> parentView) {
                // null
            }
        });
        solutionRadioButton= (RadioButton) findViewById(R.id.solutionRadioButton);
        solutionRadioButton.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                switchSolutionFields(true);
            }
        });
        powderRadioButton= (RadioButton) findViewById(R.id.powderRadioButton);
        powderRadioButton.setOnClickListener(new View.OnClickListener() {
                public void onClick(View v) {
                    switchSolutionFields(false);
                }
        });

        //get intents and set consequent actions
        //Manage method passed from Main
        Intent intent = getIntent();
        method = intent.getIntExtra(MainActivity.EXTRA_METHOD,MainActivity.METHOD_WALSTAD);
        //methodsSpinner.setSelection(method);
        if (method== MainActivity.METHOD_RESULT){
            switchMethodFields(false);
        }else{
            switchMethodFields(true);
        }
        if (method!=MainActivity.METHOD_TARGET && method!=MainActivity.METHOD_RESULT ){
            targetEditText.setText(getTarget(method));
        }else if (method==MainActivity.METHOD_RESULT){
            targetEditText.setText("1");//NOI18N
        }
        Resources res = getResources();
        String[] methods = res.getStringArray(R.array.solutions_method);
        this.setTitle(res.getString(R.string.app_name)+ " - " + res.getString(R.string.method) +
                " " + methods[method]);
        setTarget();

    }//end onCreate

    //menu
    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.calculator, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        switch (item.getItemId()) {
            case android.R.id.home: // This ID represents the Home or Up button.
                NavUtils.navigateUpFromSameTask(this);
                return true;
            case R.id.action_clear: //React to clear button click
                clearFields();
                return true;
            case R.id.action_calc:  //React to calcButton click
                doAllCalc();
                return true;
            case R.id.action_about:
                Intent intent = new Intent(this, AboutActivity.class);

                startActivity(intent);
                return true;
        }
        return super.onOptionsItemSelected(item);
    }

    /**
     * Refresh the target label extracting the target element from the compound
     *
     * @param compound selected compound
     */
    private void RefreshTargetLabel (String compound){
        Resources res =getResources();
        String[] elements = res.getStringArray(R.array.elements_names);
        double [] values=(double[]) solute.get(compound); //default on diy
        int targetElement=(int) values[Target];
        TextView defTargetTextView = (TextView) findViewById(R.id.defTargetTextView);
        defTargetTextView.setText(elements[targetElement]);
    }


    /**
     * Creates solute HashMap and loads
     * diy products list on solName spinner
     */
    private void loadDiy(){
        solute.put("(CaSO4)2.H2O",new double [] {0,0,0,0.2761079352,0,0,0,0,0,0,0,0,0,0,2.4,Ca});//NOI18N
        solute.put("Ca(NO3)2.4(H2O)",new double [] {0.5251123221,0,0,0.1697000101,0,0,0,0,0,0,0,0,0,0,121,Ca});//NOI18N
        solute.put("CaCl2",new double [] {0,0,0,0.3611282838,0,0,0,0,0,0,0,0,0,0,750,Ca});//NOI18N
        solute.put("CaCl2.2H2O",new double [] {0,0,0,0.2725875023,0,0,0,0,0,0,0.48232,0,0,0.03808,750,Ca});//NOI18N
        solute.put("CaCl2.6H2O",new double [] {0,0,0,0.1829489513,0,0,0,0,0,0,0,0,0,0,750,Ca});//NOI18N
        solute.put("CaCO3",new double [] {0,0,0,0.400383004,0,0,0,0,0,0,0,0,0,0,0.007,Ca});//NOI18N
        solute.put("CaMg(CO3)2",new double [] {0,0,0,0.2172826362,0.1318080382,0,0,0,0,0,0,0,0,0,1000000000,Ca});//NOI18N
        solute.put("CSM+B",new double [] {0,0,0,0,0.0140060236,0.0652983532,0.0187,0.008,0.0009,0.0005,0,0,0.0037,0,1000000000,Fe});//NOI18N
        solute.put("Fleet Enema",new double [] {0,0.1313537888,0,0,0,0,0,0,0,0,0,0,0,0,1000000000,PO4});//NOI18N
        solute.put("Iron (Fe) 07% DPTA",new double [] {0,0,0,0,0,0.07,0,0,0,0,0,0,0,0,1000000000,Fe});//NOI18N
        solute.put("Iron (Fe) 10% DPTA",new double [] {0,0,0,0,0,0.1000105792,0,0,0,0,0,0,0,0,200,Fe});//NOI18N
        solute.put("Iron (Fe) 13% EDTA",new double [] {0,0,0,0,0,0.130013753,0,0,0,0,0,0,0,0,90,Fe});//NOI18N
        solute.put("Iron (Fe) Gluconate",new double [] {0,0,0,0,0,0.123404424,0,0,0,0,0,0,0,0,1000000000,Fe});//NOI18N
        solute.put("K2CO3",new double [] {0,0,0.5658054988,0,0,0,0,0,0,0,0,0,0,0,1120,K});//NOI18N
        solute.put("K2HPO4",new double [] {0,0.5452507128,0.4489498371,0,0,0,0,0,0,0,0,0,0,0,1492,PO4});//NOI18N
        solute.put("K2SO4",new double [] {0,0,0.4486848583,0,0,0,0,0,0,0,0,0.18401,0,0,120,K});//NOI18N
        solute.put("K3PO4",new double [] {0,0.447416,0.552596,0,0,0,0,0,0,0,0,0,0,0,900,PO4});//NOI18N
        solute.put("KCl",new double [] {0,0,0.5244688021,0,0,0,0,0,0,0,0.18401,0,0,0,344,K});//NOI18N
        solute.put("KH2PO4",new double [] {0,0.6979163699,0.2873127541,0,0,0,0,0,0,0,0,0,0,0,220,PO4});//NOI18N
        solute.put("KNO3",new double [] {0.6133124166,0,0.3867176674,0,0,0,0,0,0,0,0,0,0,0,360,NO3});//NOI18N
        solute.put("MgCO3",new double [] {0,0,0,0,0.288259107,0,0,0,0,0,0,0,0,0,0.4,Mg});//NOI18N
        solute.put("MgSO4.7H2O",new double [] {0,0,0,0,0.0986099769,0,0,0,0,0,0,0.1301,0,0.02255,710,Mg});//NOI18N
        solute.put("MnSO4",new double [] {0,0,0,0,0,0,0.3250597919,0,0,0,0,0.2123262543,0,0,393,Mn});//NOI18N
        solute.put("Na2HPO4",new double [] {0,0.6689958239,0,0,0,0,0,0,0,0,0,0,0,0,1000000000,PO4});//NOI18N
        solute.put("NaH2PO3",new double [] {0,0.7915674573,0,0,0,0,0,0,0,0,0,0,0,0,1000000000,PO4});//NOI18N
        //solNameComboBox.removeAllItems();
        List sortedKeys=new ArrayList(solute.keySet());
        Collections.sort(sortedKeys);
        Iterator iterator=sortedKeys.iterator();
        int counter=sortedKeys.size();
        String array_spinner[];
        array_spinner=new String[counter];
        counter=0;
        iterator=sortedKeys.iterator();
        while(iterator.hasNext()){
            Object key = iterator.next();
            array_spinner[counter]=key.toString();
            counter++;
            if (counter > sortedKeys.size()) {
                Log.e("PAFCalc", "loadDiy iterating the array -> No good to be here...");//NOI18N
                return;
            }
        }
        Spinner s = (Spinner) findViewById(R.id.solNamespinner);
        ArrayAdapter adapter = new ArrayAdapter(this,
                R.layout.spinner_item, array_spinner);
        s.setAdapter(adapter);

    }

    /**
    * Sets visible fields according to solution dosing
    *
    * @param standardFields [true for solution | false for dry]
    */
    private void switchSolutionFields (boolean standardFields){

        TextView solWaterVolTextView=(TextView)findViewById(R.id.solWaterVolTextView);
        EditText solWaterVolumeEditText = (EditText) findViewById(R.id.solWaterVolumeEditText);
        TextView doseVolumeTextView=(TextView)findViewById(R.id.doseVolumeTextView);
        EditText doseVolumeEditText = (EditText) findViewById(R.id.doseVolumeEditText);

        if (!standardFields){
            solWaterVolumeEditText.setText("1");//NOI18N
            doseVolumeEditText.setText("1");//NOI18N
        }
        solWaterVolTextView.setEnabled(standardFields);
        solWaterVolumeEditText.setEnabled(standardFields);
        doseVolumeTextView.setEnabled(standardFields);
        doseVolumeEditText.setEnabled(standardFields);
    }

    /**
     * Sets visible fields according to method
     *
     * @param standardFields [true for standard (most used) fields | false for result method]
     */
    private void switchMethodFields (boolean standardFields){
        //boolean nonStandardField=!standardFields;
        int visibility=View.VISIBLE;
        int otherVisibility=View.INVISIBLE;
        if (!standardFields){
            visibility=View.INVISIBLE;
            otherVisibility=View.VISIBLE;
        }

        TextView resultTextView = (TextView) findViewById(R.id.resultTextView);
        TextView showResultTextView = (TextView) findViewById(R.id.showResultTextView);
        TextView solubleTextView = (TextView) findViewById(R.id.solubleTextView);
        EditText targetEditText=(EditText) findViewById(R.id.targetEditText);
        ImageView solubleImageView =(ImageView) findViewById(R.id.solubleImageView);

        TextView unTextView =(TextView) findViewById(R.id.unTextView);
        TextView adding = (TextView) findViewById(R.id.adding);
        EditText addingText = (EditText) findViewById(R.id.addingText);
        RadioGroup units2BG = (RadioGroup) findViewById(R.id.units2BG);

        resultTextView.setVisibility(visibility);
        showResultTextView.setVisibility(visibility);
        solubleTextView.setVisibility(visibility);
        solubleImageView.setVisibility(visibility);
        targetEditText.setFocusableInTouchMode(standardFields);
        targetEditText.setFocusable(standardFields);

        unTextView.setVisibility(otherVisibility);
        adding.setVisibility(otherVisibility);
        addingText.setVisibility(otherVisibility);
        units2BG.setVisibility(otherVisibility);

        if (standardFields){
            unTextView.setHeight(0);
            units2BG.setMinimumHeight(0);
            RadioButton un2RB=(RadioButton)findViewById(R.id.un2RadioButton);
            RadioButton un3RB=(RadioButton)findViewById(R.id.un3RadioButton);
            un2RB.setHeight(0);
            un3RB.setHeight(0);
            adding.setHeight(0);
            addingText.setHeight(0);
        }else{
            resultTextView.setHeight(0);
            showResultTextView.setHeight(0);
            solubleTextView.setHeight(0);
            solubleImageView.setMinimumHeight(0);
            solubleImageView.setImageDrawable(null);
        }

    }

    /**
     * Do all checks for input values:
     * check for empty or zero fields,
     */
    private void checkFields(){
        //fields checked depend on method
        TextView [] jtfList;
        if (method==MainActivity.METHOD_TARGET){
            if (powderRadioButton.isChecked()){  //fields for powder
                jtfList =new TextView [] {aquariumWaterVolText,targetEditText};
            }else { //fields for solution
                jtfList =new TextView[] {aquariumWaterVolText, targetEditText,
                        solWaterVolumeEditText,doseVolumeEditText};
            }

        } else if (method==MainActivity.METHOD_RESULT){
            if (powderRadioButton.isChecked()){ //fields for powder
                jtfList =new TextView[] {aquariumWaterVolText, addingText };
            }else { //fields for solution
                jtfList =new TextView[] {aquariumWaterVolText,addingText,solWaterVolumeEditText,
                        doseVolumeEditText};
            }
        } else {
            if (powderRadioButton.isChecked()){ //fields for powder
                jtfList =new TextView [] {aquariumWaterVolText };
            }else { //fields for solution
                jtfList =new TextView []{aquariumWaterVolText,solWaterVolumeEditText, doseVolumeEditText};
            }
        }
        enableCalc=tools.CheckTextFields(jtfList);   //enabling/disabling calc button
    }


    /**
     * Get target value for the component in chosen method
     *
     * @param methodChosen    the selected method
     * @return     target component value
     */
    private String getTarget(int methodChosen){
        String target="1"; //NOI18N //default unitary target
        String compound = solNameSpinner.getSelectedItem().toString();
        double [] values=(double[]) solute.get(compound);
        //calc solute
        int targetElement=(int) values[Target];
        String target_value=LocUtil.localizeDouble(methods_parameters[targetElement][methodChosen][PAR_METHOD]);
        if (! target_value.isEmpty()){
            target=target_value;
        }
        return target;
    }

    /**
     * enable or disable target field depending on method
     */
    private void setTarget () {
        if (method < 0) {
            return;
        }
        EditText targetTextField=(EditText) findViewById(R.id.targetEditText);
        if (method== MainActivity.METHOD_TARGET || method==MainActivity.METHOD_RESULT){
            targetTextField.setText("");
            targetTextField.setEnabled(true);
        } else {
            targetTextField.setText(getTarget(method));
            targetTextField.setEnabled(false);
        }
    }

    /**
     * Check input text fields for empty or zero values
     * Check for valid number format
     * (values are surely numbers because there is an input filter on field)
     *
     * @param jtf      text field to be checked
     * @param errMsg   the error message to display
     * @return         the error message or an empty string
     */
    private String checkNumericInput(TextView jtf, String errMsg){
        String thisErrMsg="";//NOI18N
        if (jtf.getText().toString().isEmpty()){
            thisErrMsg=thisErrMsg+errMsg;
            return thisErrMsg;
        }
        try {
            if (Double.valueOf(LocUtil.delocalizeDouble(jtf.getText().toString()))==0){
                thisErrMsg=thisErrMsg+errMsg;
            }
        }catch (NumberFormatException err){
            Resources res = getResources();
            thisErrMsg = res.getString(R.string.number_format_err);
        }
        return thisErrMsg;
    }

    /**
     * Calc the amount of compound to be dissolved in water to obtain a solution
     * that will bring the target element to the desired value
     *
     * All required fields need to be non zero values checked before calling function
     *
     * /*used formula is:
     * x=  solute in mg/l = amount of powder (or liquid) to put in water to get the solution
     * target = target quantity in ppm or mg/l for the element to reach in tank
     * K=  value of target element contained in a unit of the initial compound
     * AV= Aquarium Water total volume in liters
     * SV= Solution Water volume in ml
     * DV= Dose Volume (the minimal dose used to reach target in ml
     *
     * x=target * (1 / k) * AV * (SV/DV)
     *
     * @param compound  compound to be dissolved in water
     * @param target  target quantity in ppm or mg/l for the element to reach in tank
     * @param AV    Aquarium Water total volume in liters
     * @param SV    Solution Water volume in ml
     * @param DV    Dose Volume (the minimal dose used to reach targ in ml
     *
     *
     * @return solute
     */
    private double calcSolute(String compound, double target, double AV, double SV,double DV){
        double [] values=(double[]) solute.get(compound);
        //calc solute
        int targetElement=(int) values[Target];
        double k=values[targetElement];
        double x=target * (1 / k) * AV * (SV/DV); //x=
        Resources res=getResources();

        //refresh the solubility label
        double solubility;
        solubility = values[Solubility];
        //System.out.println("g. " + (x/SV) + " - solubility:" + solubility  );
        if ((x/SV) < solubility){
            solubleImageView.setImageDrawable(res.getDrawable(R.drawable.ic_ok));
        }else{
            solubleImageView.setImageDrawable(res.getDrawable(R.drawable.ic_ko));
        }
        return x;
    }

    /**
     * If milligrams is used then the output is converted in grams wheen needed
     * If milliliters is the right one the used ml or liters when needed
     *
     * @param mg    value in milligrams
     * @return String the value+ mg or g
     */
    private String formatMilligrams (double mg){
        String fm="mg";//NOI18N
        if (mg < 1000){
            fm=LocUtil.localizeDouble(mg)+" "+fm;//NOI18N
        }else{
            fm=LocUtil.localizeDouble(mg/1000)+" g";//NOI18N
        }
        return fm;
    }

    /**
     * do calc operations
     */
    private void doAllCalc(){
        if (! enableCalc){  //show error and exit
            String msg=getString(R.string.fill_all_values);
            Toast toast = Toast.makeText(this,msg,Toast.LENGTH_LONG);
            toast.show();
            return;
        }
        if (method!=MainActivity.METHOD_TARGET && method!=MainActivity.METHOD_RESULT ){
            targetEditText.setText(getTarget(method));
        }else if (method==MainActivity.METHOD_RESULT){
            targetEditText.setText("1");//NOI18N
        }

        //do all checks for input values
        String msg="";//NOI18N
        Resources res = getResources();
        String errMsg=res.getString(R.string.zero_field_check);
        //Check for empty fields:
        //This fields need ever to be checked
        TextView labelString;
        labelString=(TextView)findViewById(R.id.targetTextView);
        msg=msg+checkNumericInput(targetEditText,labelString.getText().toString() + errMsg);
        labelString=(TextView)findViewById(R.id.aquariumWaterVol);
        msg=msg+checkNumericInput(aquariumWaterVolText,labelString.getText().toString()+errMsg);
        //these fields are needed only for Diy
        labelString=(TextView)findViewById(R.id.solWaterVolTextView);
        msg=msg+checkNumericInput(solWaterVolumeEditText,labelString.getText()+errMsg);
        labelString=(TextView)findViewById(R.id.doseVolumeTextView);
        msg=msg+checkNumericInput(doseVolumeEditText,labelString.getText()+errMsg);
        //this field is only for result method
        if (method==MainActivity.METHOD_RESULT){
            labelString=(TextView) findViewById(R.id.doseVolumeTextView);
            msg=msg+checkNumericInput(doseVolumeEditText,labelString.getText()+errMsg);
        }
        //there is an input error
        if (! msg.contentEquals("")) {//NOI18N
            Toast toast = Toast.makeText(this,msg,Toast.LENGTH_LONG);
            toast.show();
            return;
        }

        //all values are OK
        double target;
        double av;
        double sv;
        double dv;
        target = Double.valueOf(LocUtil.delocalizeDouble(targetEditText.getText().toString()));
        av = Double.valueOf(LocUtil.delocalizeDouble(aquariumWaterVolText.getText().toString()));
        sv = Double.valueOf(LocUtil.delocalizeDouble(solWaterVolumeEditText.getText().toString()));
        dv = Double.valueOf(LocUtil.delocalizeDouble(doseVolumeEditText.getText().toString()));
        //if us gal is selected--> convert to liters
        if (unitsGalRadioButton.isChecked()){
            av= Converter.gal2l(av);
        }
        //evaluate selected operation
        if (method==MainActivity.METHOD_RESULT){
            showResultTextView.setText("0");//NOI18N
            double calcResult;
            calcResult = Double.valueOf(LocUtil.delocalizeDouble(doseVolumeEditText.getText().toString()));
            //evaluate units
            if (doseUnitsGRadioButton.isChecked()){
                calcResult= Converter.g2mg(calcResult);
            }
            String solutionName=solNameSpinner.getSelectedItem().toString();
            String [] values = calcElements(solutionName,calcResult, av, sv, dv);
            buildResultTable(resultTableContainer,values);
        }else {
            double calcResult=calcSolute(solNameSpinner.getSelectedItem().toString(),target,av,sv,dv);
            showResultTextView.setText(formatMilligrams(calcResult));
            String solutionName=solNameSpinner.getSelectedItem().toString();
            String [] values = calcElements(solutionName,calcResult, av, sv, dv);
            buildResultTable(resultTableContainer,values);
        }
        //TODO Plotting result in graph
        //        DoPlot();
    }

    /**
     * Calculate the amount of each element that will be dosed at selected conditions
     *
     * @param compound the introduced compound
     * @param x    solute in mg/l = amount of powder (or liquid) to put in water to get the solution
     * @param AV   Aquarium Water total volume in liters
     * @param SV   Solution Water volume in ml
     * @param DV   Dose Volume (the minimal dose used to reach targ in ml
     * @return     An array with the amount for each element
     */
    private String [] calcElements(String compound, double x, double AV, double SV,double DV){
         /*for each element the formula is:
         * x=  solute in mg/l = amount of powder (or liquid) to put in water to get the solution
          * target = target quantity in ppm or mg/l for the element to reach in tank
          * K=  value of target element contained in a unit of the initial compound
          * AV= Aquarium Water total volume in liters
          * SV= Solution Water volume in ml
          * DV= Dose Volume (the minimal dose used to reach target in ml
          *
          * x * K * {1/ [AV * (SV/DV)]} =target
          */
        String [] elementsDosed=new String [14];
        double [] values;
        values=(double[]) solute.get(compound);
        for (int y=0; y<14;y++){
            double k=values[y];
            double tmp_value=x*k*(1 / (AV * (SV/DV)));
            elementsDosed[y]= String.valueOf(tmp_value);
            if (y==Cu){
                checkCu(elementsDosed[y]);
            }
        }
        if (compound.contains("EDDHA")){
            checkEDDHA();
        } else if (compound.contains("K3PO4")){
            checkK3PO4();
        }
        return elementsDosed;
    }

    /**
     * Build the grid to display elements values
     *
     * @param resultTableContainer  a grid view table
     * @param cellValues            values to be showed
     */
    private void buildResultTable(GridView resultTableContainer, String [] cellValues){
        ArrayList<String> listHeaders;
        ArrayList<String> listCells;
        CellsAdapter mAdapter;
        listHeaders=new ArrayList<String>();
        listCells=new ArrayList<String>();
        Resources res = getResources();
        Collections.addAll(listHeaders, res.getStringArray(R.array.elements_names));
        if  (cellValues==null) {
            cellValues=new String [14];
        }
        for (int i=0;i<14;i++){
            listCells.add(LocUtil.localizeDouble(cellValues[i]));
        }
        mAdapter = new CellsAdapter(this,listHeaders, listCells);
        resultTableContainer.setAdapter(mAdapter);
    }

    /**
     * Add a warning for toxic Cu concentration
     *
     * @param cuDose   Amount of Cu in compound
     */
    private void checkCu(String cuDose){
        double valCUDose=Double.valueOf(cuDose);
        if (valCUDose > 0.072){
            int toxic=(int) ((valCUDose/0.072)*100);
            Resources res = getResources();
            String hint=res.getString(R.string.toxicCU);
            hint=hint +" " +res.getString(R.string.Overdosed);//NOI18N
            hint=hint + " " + String.valueOf(toxic)+"x";//NOI18N
            showTimedToast(hint, 5);
        }
    }

    /**
     * Add a warning for K3PO4
     */
    private void checkK3PO4(){
        Resources res = getResources();
        String hint=res.getString(R.string.warningK3PO4);
        showTimedToast(hint, 5);
    }

    /**
     * Add a warning for EDDHA FE
     */
    private void checkEDDHA(){
        Resources res = getResources();
        String hint=res.getString(R.string.warningEDDHA);
        showTimedToast(hint, 5);
    }

    /**
     * Clean editable fields
     */
    private void clearFields(){
        TextView [] jtfList={aquariumWaterVolText,targetEditText,solWaterVolumeEditText,
                doseVolumeEditText,addingText,showResultTextView};
        tools.CleanTextFields(jtfList);
        //switchMethodFields(true); @deprecated
        switchSolutionFields(true);
        solutionRadioButton.setSelected(true);
        solNameSpinner.setSelection(0);
        buildResultTable(resultTableContainer, null);
        //TODO clean also graphic
        /*int previousCanvas=graphPanel.getComponentCount();
        while (previousCanvas > 0 ){
            graphPanel.remove(0);
            previousCanvas=graphPanel.getComponentCount();
        }*/
        solubleImageView.setImageDrawable(null);
    }

    /**
     * Show a toast for a custom number of seconds
    * 
     * @param msg   string message
     * @param seconds   int number of seconds
     */
    public void showTimedToast (String msg,int seconds){
        final Toast tag = Toast.makeText(this, msg,Toast.LENGTH_SHORT);
        tag.show();
        if (seconds<1) Log.e("PAFCalc", "error in toast use: less than 1 second");//NOI18N
        //if more than 1 sec decrease by one to allow onFinish time of 1 sec
        if (seconds>1) seconds--;
        new CountDownTimer(seconds*1000, 1000)  //goes to milliseconds
        {
            public void onTick(long millisUntilFinished) {tag.show();}
            public void onFinish() {tag.show();}
        }.start();
    }



}
